package com.biz.crm.cps.business.activity.scan.local.service.internal;

import com.alibaba.fastjson.JSONObject;
import com.biz.crm.cps.business.activity.scan.local.entity.ScanActivityParticipateScope;
import com.biz.crm.cps.business.activity.scan.local.service.ScanActivityParticipateScopeService;
import com.biz.crm.cps.business.activity.scan.sdk.dto.ScanActivityParticipateScopeDto;
import com.biz.crm.cps.business.activity.scan.sdk.enums.ScanActivityParticipateScopeTypeEnum;
import com.biz.crm.cps.business.activity.scan.sdk.enums.ScanActivityScopeEnum;
import com.biz.crm.cps.business.activity.scan.sdk.register.ScanActivityParticipateScopeRegister;
import com.biz.crm.cps.business.activity.scan.sdk.service.ScanActivityParticipateScopeVoService;
import com.biz.crm.cps.business.activity.scan.sdk.vo.ScanActivityDictVo;
import com.biz.crm.cps.business.activity.scan.sdk.vo.ScanActivityParticipateScopeVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

/**
 * 扫码活动参与范围配置业务相关接口实现
 *
 * @author sunx
 * @date 2022/2/22
 */
@Slf4j
@Service("scanActivityParticipateScopeVoService")
public class ScanActivityParticipateScopeVoServiceImpl
    implements ScanActivityParticipateScopeVoService {

  @Autowired(required = false)
  private List<ScanActivityParticipateScopeRegister> registers;

  @Autowired private ScanActivityParticipateScopeService scanActivityParticipateScopeService;

  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Override
  public List<ScanActivityDictVo> findParticipateObjSelectList() {
    if (CollectionUtils.isEmpty(registers)) {
      return Lists.newLinkedList();
    }
    List<ScanActivityDictVo> re = Lists.newArrayList();
    registers.stream()
        .sorted(Comparator.comparing(ScanActivityParticipateScopeRegister::sort))
        .forEach(
            a -> {
              ScanActivityDictVo cur = new ScanActivityDictVo();
              cur.setDictCode(a.getType());
              cur.setValue(a.getTypeDesc());
              re.add(cur);
            });
    return re;
  }

  @Override
  @Transactional
  public void createParticipateScope(
      String actCode, String participateScope, List<ScanActivityParticipateScopeDto> list) {
    Validate.notBlank(actCode, "活动编码不能为空");
    List<ScanActivityParticipateScope> entityList = Lists.newLinkedList();
    if (!ScanActivityScopeEnum.ALL.getDictCode().equals(participateScope)) {
      entityList =
          (List<ScanActivityParticipateScope>)
              this.nebulaToolkitService.copyCollectionByBlankList(
                  list,
                  ScanActivityParticipateScopeDto.class,
                  ScanActivityParticipateScope.class,
                  HashSet.class,
                  ArrayList.class);
    }
    this.scanActivityParticipateScopeService.createParticipateScope(actCode, entityList);
  }

  @Override
  public List<ScanActivityParticipateScopeVo> findByParticipateObjAndActCode(
      String participateObj, String actCode) {
    if (StringUtils.isBlank(participateObj) || StringUtils.isBlank(actCode)) {
      return Lists.newLinkedList();
    }
    List<ScanActivityParticipateScope> list =
        this.scanActivityParticipateScopeService.findByActCode(actCode);
    if (CollectionUtils.isEmpty(list)) {
      return Lists.newLinkedList();
    }
    List<ScanActivityParticipateScopeVo> re =
        (List<ScanActivityParticipateScopeVo>)
            this.nebulaToolkitService.copyCollectionByBlankList(
                list,
                ScanActivityParticipateScope.class,
                ScanActivityParticipateScopeVo.class,
                HashSet.class,
                ArrayList.class);
    // 范围类型 品牌  商品等
    String scopeType = list.get(0).getParticipateScopeType();
    if (!ScanActivityParticipateScopeTypeEnum.DEFAULT.getDictCode().equals(scopeType)) {
      return re;
    }
    // 获取业务（经销商、终端、导购）的附加信息
    Set<String> userCodeSet =
        list.stream()
            .filter(a -> StringUtils.isNotBlank(a.getRelateCode()))
            .map(ScanActivityParticipateScope::getRelateCode)
            .collect(Collectors.toSet());
    Map<String, JSONObject> map = this.getRegister(participateObj).findDataMapByCodes(userCodeSet);
    for (ScanActivityParticipateScopeVo item : re) {
      item.setData(map.get(item.getRelateCode()));
    }
    return re;
  }

  @Override
  public Set<String> findScopeCodeSet(
      String participateObj, String participateScope, List<ScanActivityParticipateScopeDto> list) {
    return this.getRegister(participateObj).findScopeCodeSet(participateScope, list);
  }

  @Override
  public Map<String, JSONObject> findDataMapByParticipateObjAndUserCodes(
      String participateObj, Set<String> userCodeSet) {
    Map<String, JSONObject> re = Maps.newHashMap();
    if (StringUtils.isBlank(participateObj) || CollectionUtils.isEmpty(userCodeSet)) {
      return re;
    }
    return this.getRegister(participateObj).findDataMapByCodes(userCodeSet);
  }

  /**
   * 根据参与类型（经销商、终端）获取具体的维度注册信息
   *
   * @param participateObj
   * @return
   */
  private ScanActivityParticipateScopeRegister getRegister(String participateObj) {
    Validate.notBlank(participateObj, "参与对象不能为空");
    Validate.isTrue(!CollectionUtils.isEmpty(registers), "无可用的参与对象类型注册信息");
    ScanActivityParticipateScopeRegister register = null;
    for (ScanActivityParticipateScopeRegister cur : registers) {
      if (cur.getType().equals(participateObj)) {
        register = cur;
        break;
      }
    }
    Validate.notNull(register, "未获取到匹配的参与类型注册信息");
    return register;
  }
}
