package com.biz.crm.cps.business.activity.scan.local.service.internal;

import com.biz.crm.cps.business.activity.scan.local.entity.ScanActivityProductScopeItem;
import com.biz.crm.cps.business.activity.scan.local.repository.ScanActivityProductScopeItemRepository;
import com.biz.crm.cps.business.activity.scan.local.service.ScanActivityProductScopeItemService;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

/**
 * 扫码活动物料范围明细(ScanActivityProductScopeItem)表服务实现类
 *
 * @author sunx
 * @date 2022-02-16 16:39:05
 */
@Slf4j
@Service("scanActivityProductScopeItemService")
public class ScanActivityProductScopeItemServiceImpl
    implements ScanActivityProductScopeItemService {

  @Autowired private ScanActivityProductScopeItemRepository scanActivityProductScopeItemRepository;

  @Override
  @Transactional
  public void createScopeItem(List<ScanActivityProductScopeItem> list, String actCode) {
    Validate.notBlank(actCode, "扫码活动编码不能为空");
    if (CollectionUtils.isEmpty(list)) {
      this.scanActivityProductScopeItemRepository.deleteByActCode(actCode);
      return;
    }
    Set<String> curProductCodeSet =
        list.stream()
            .map(ScanActivityProductScopeItem::getMaterialCode)
            .collect(Collectors.toSet());
    List<ScanActivityProductScopeItem> oldList =
        this.scanActivityProductScopeItemRepository.findByActCodes(Sets.newHashSet(actCode));
    Set<String> oldProductCodeSet = Sets.newHashSet();
    if (!CollectionUtils.isEmpty(oldList)) {
      oldProductCodeSet =
          oldList.stream()
              .map(ScanActivityProductScopeItem::getMaterialCode)
              .collect(Collectors.toSet());
    }

    Set<String> delProductCodeSet = Sets.newHashSet();
    Set<String> addProductCodeSet = Sets.newHashSet();

    if (!CollectionUtils.isEmpty(curProductCodeSet)) {
      delProductCodeSet = Sets.difference(oldProductCodeSet, curProductCodeSet);
      addProductCodeSet = Sets.difference(curProductCodeSet, oldProductCodeSet);
    }

    if (!CollectionUtils.isEmpty(delProductCodeSet)) {
      Set<String> finalDelProductCodeSet = delProductCodeSet;
      Set<String> ids =
          oldList.stream()
              .filter(a -> finalDelProductCodeSet.contains(a.getMaterialCode()))
              .map(ScanActivityProductScopeItem::getId)
              .collect(Collectors.toSet());
      if (!CollectionUtils.isEmpty(ids)) {
        this.scanActivityProductScopeItemRepository.removeByIds(ids);
      }
    }

    if (!CollectionUtils.isEmpty(addProductCodeSet)) {
      List<ScanActivityProductScopeItem> addList = Lists.newLinkedList();
      for (String item : addProductCodeSet) {
        ScanActivityProductScopeItem cur = new ScanActivityProductScopeItem();
        cur.setActCode(actCode);
        cur.setMaterialCode(item);
        addList.add(cur);
      }
      this.scanActivityProductScopeItemRepository.saveBatch(addList);
    }
  }

  @Override
  @Transactional
  public void deleteByActCode(String actCode) {
    Validate.notBlank(actCode, "扫码活动编码不能为空");
    this.scanActivityProductScopeItemRepository.deleteByActCode(actCode);
  }

  @Override
  public Set<String> findProductScopeItemActCodeSet(
      Set<String> actCodeSet, Set<String> materialCodeSet) {
    if (CollectionUtils.isEmpty(actCodeSet) || CollectionUtils.isEmpty(materialCodeSet)) {
      return Sets.newHashSet();
    }
    Set<String> set = Sets.newHashSet();
    for (List<String> list : Lists.partition(Lists.newArrayList(materialCodeSet), 500)) {
      Set<String> cur =
          this.scanActivityProductScopeItemRepository.findProductScopeItemActCodeSet(
              actCodeSet, Sets.newHashSet(list));
      if (CollectionUtils.isEmpty(cur)) {
        continue;
      }
      set.addAll(cur);
    }
    return set;
  }

  @Override
  public Set<String> findMaterialActCodeSetByMaterialCodeAndActCodeIn(
      String materialCode, Set<String> actCodeSet) {
    if (StringUtils.isBlank(materialCode) || CollectionUtils.isEmpty(actCodeSet)) {
      return Sets.newHashSet();
    }
    return this.scanActivityProductScopeItemRepository.findProductScopeItemActCodeSet(
        actCodeSet, Sets.newHashSet(materialCode));
  }
}
