package com.biz.crm.cps.business.activity.scan.local.service.internal;

import com.alibaba.fastjson.JSONObject;
import com.biz.crm.cps.business.activity.scan.local.entity.ScanActivityProductScope;
import com.biz.crm.cps.business.activity.scan.local.service.ScanActivityProductScopeService;
import com.biz.crm.cps.business.activity.scan.sdk.dto.ScanActivityProductScopeDto;
import com.biz.crm.cps.business.activity.scan.sdk.enums.ScanActivityProductScopeTypeEnum;
import com.biz.crm.cps.business.activity.scan.sdk.register.ScanActivityProductScopeRegister;
import com.biz.crm.cps.business.activity.scan.sdk.service.ScanActivityProductScopeVoService;
import com.biz.crm.cps.business.activity.scan.sdk.vo.ScanActivityDictVo;
import com.biz.crm.cps.business.activity.scan.sdk.vo.ScanActivityProductScopeVo;
import com.biz.crm.mdm.business.material.sdk.service.MaterialVoService;
import com.biz.crm.mdm.business.material.sdk.vo.MaterialVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import java.util.stream.Collectors;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

/**
 * @author sunx
 * @date 2022/2/21
 */
@Slf4j
@Service("scanActivityProductScopeVoService")
public class ScanActivityProductScopeVoServiceImpl implements ScanActivityProductScopeVoService {

  @Autowired private ScanActivityProductScopeService scanActivityProductScopeService;

  @Autowired(required = false)
  private List<ScanActivityProductScopeRegister> productScopeRegisters;

  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Autowired
  private com.biz.crm.cps.business.product.sdk.service.MaterialVoService materialVoService1;

  @Autowired private MaterialVoService materialVoService;

  @Override
  @Transactional
  public void createProductScope(List<ScanActivityProductScopeDto> list, String actCode) {
    Validate.notBlank(actCode, "活动编码不能为空");
    Validate.isTrue(!CollectionUtils.isEmpty(list), "活动物料配置信息不能为空");
    List<ScanActivityProductScope> scopes =
        (List<ScanActivityProductScope>)
            this.nebulaToolkitService.copyCollectionByBlankList(
                list,
                ScanActivityProductScopeDto.class,
                ScanActivityProductScope.class,
                HashSet.class,
                ArrayList.class);
    this.scanActivityProductScopeService.createProductScope(scopes, actCode);

    Map<String, Set<String>> map =
        list.stream()
            .filter(a -> StringUtils.isNoneBlank(a.getScopeType(), a.getRelateCode()))
            .collect(
                Collectors.groupingBy(
                    ScanActivityProductScopeDto::getScopeType,
                    Collectors.mapping(
                        ScanActivityProductScopeDto::getRelateCode, Collectors.toSet())));

    if (!map.isEmpty()) {
      for (Entry<String, Set<String>> item : map.entrySet()) {
        String dimensionFlag = item.getKey();
        CompletableFuture.runAsync(
            () -> {
              materialVoService1.sync(Lists.newArrayList(item.getValue()), dimensionFlag);
            });
      }
    }
  }

  @Override
  public List<ScanActivityProductScopeVo> findByActCode(String actCode) {
    if (StringUtils.isBlank(actCode)) {
      return Lists.newLinkedList();
    }
    List<ScanActivityProductScope> list =
        this.scanActivityProductScopeService.findByActCode(actCode);
    if (CollectionUtils.isEmpty(list)) {
      return Lists.newLinkedList();
    }
    List<ScanActivityProductScopeVo> re =
        (List<ScanActivityProductScopeVo>)
            this.nebulaToolkitService.copyCollectionByBlankList(
                list,
                ScanActivityProductScope.class,
                ScanActivityProductScopeVo.class,
                HashSet.class,
                ArrayList.class);
    // 范围类型 品牌  商品等
    String scopeType = list.get(0).getScopeType();
    if (!ScanActivityProductScopeTypeEnum.MATERIAL.getDictCode().equals(scopeType)) {
      return re;
    }
    // 获取物料（商品）的附加信息
    Set<String> materialCodeSet =
        list.stream()
            .filter(a -> StringUtils.isNotBlank(a.getRelateCode()))
            .map(ScanActivityProductScope::getRelateCode)
            .collect(Collectors.toSet());
    Map<String, MaterialVo> map = Maps.newHashMap();
    if (!CollectionUtils.isEmpty(materialCodeSet)) {
      // 获取物料明细信息
      List<MaterialVo> materialVoList =
          this.materialVoService.findDetailByMaterialCodes(materialCodeSet);
      if (!CollectionUtils.isEmpty(materialVoList)) {
        map =
            materialVoList.stream()
                .filter(a -> StringUtils.isNotBlank(a.getMaterialCode()))
                .collect(
                    Collectors.toMap(
                        MaterialVo::getMaterialCode, Function.identity(), (a, b) -> a));
      }
    }
    for (ScanActivityProductScopeVo item : re) {
      MaterialVo cur = map.get(item.getRelateCode());
      if (Objects.nonNull(cur)) {
        item.setData((JSONObject) JSONObject.toJSON(cur));
      }
    }
    return re;
  }

  @Override
  public Set<String> findMaterialCodeSet(List<ScanActivityProductScopeDto> list) {
    if (CollectionUtils.isEmpty(list)) {
      return Sets.newHashSet();
    }
    String scopeType = list.get(0).getScopeType();
    Set<String> set =
        list.stream()
            .filter(a -> StringUtils.isNotBlank(a.getRelateCode()))
            .map(ScanActivityProductScopeDto::getRelateCode)
            .collect(Collectors.toSet());
    return this.getRegister(scopeType).findByRelateCodeSet(set);
  }

  @Override
  public List<ScanActivityDictVo> findProductScopeTypeList() {
    if (CollectionUtils.isEmpty(productScopeRegisters)) {
      return Lists.newLinkedList();
    }
    List<ScanActivityDictVo> re = Lists.newArrayList();
    productScopeRegisters.stream()
        .sorted(Comparator.comparing(ScanActivityProductScopeRegister::sort))
        .forEach(
            a -> {
              ScanActivityDictVo cur = new ScanActivityDictVo();
              cur.setDictCode(a.getType());
              cur.setValue(a.getTypeDesc());
              // 适应前端排除单品选项
              if (!ScanActivityProductScopeTypeEnum.MATERIAL
                  .getDictCode()
                  .equals(cur.getDictCode())) {
                re.add(cur);
              }
            });
    return re;
  }

  /**
   * 根据物料范围类型（物料组、单品）获取具体的维度注册信息
   *
   * @param scopeType
   * @return
   */
  private ScanActivityProductScopeRegister getRegister(String scopeType) {
    Validate.notBlank(scopeType, "物料范围类型不能为空");
    Validate.isTrue(!CollectionUtils.isEmpty(productScopeRegisters), "无可用的物料范围类型注册信息");
    ScanActivityProductScopeRegister register = null;
    for (ScanActivityProductScopeRegister cur : productScopeRegisters) {
      if (cur.getType().equals(scopeType)) {
        register = cur;
        break;
      }
    }
    Validate.notNull(register, "未获取到匹配的物料范围类型注册信息");
    return register;
  }
}
