package com.biz.crm.cps.business.activity.scan.local.service.internal;

import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.cps.business.activity.scan.local.entity.ScanActivityRebate;
import com.biz.crm.cps.business.activity.scan.local.entity.ScanActivityRebateValue;
import com.biz.crm.cps.business.activity.scan.local.entity.ScanActivityReward;
import com.biz.crm.cps.business.activity.scan.local.entity.ScanActivityRewardValue;
import com.biz.crm.cps.business.activity.scan.local.repository.ScanActivityRebateRepository;
import com.biz.crm.cps.business.activity.scan.local.repository.ScanActivityRebateValueRepository;
import com.biz.crm.cps.business.activity.scan.local.repository.ScanActivityRewardRepository;
import com.biz.crm.cps.business.activity.scan.local.repository.ScanActivityRewardValueRepository;
import com.biz.crm.cps.business.activity.scan.sdk.constant.ScanActivityConstant;
import com.biz.crm.cps.business.activity.scan.sdk.dto.ScanActivityRebateDto;
import com.biz.crm.cps.business.activity.scan.sdk.dto.ScanActivityRebateValueDto;
import com.biz.crm.cps.business.activity.scan.sdk.dto.ScanActivityRewardDto;
import com.biz.crm.cps.business.activity.scan.sdk.dto.ScanActivityRewardValueDto;
import com.biz.crm.cps.business.activity.scan.sdk.enums.ScanActivityParticipateObjEnum;
import com.biz.crm.cps.business.activity.scan.sdk.enums.ScanActivityRewardScanTypeEnum;
import com.biz.crm.cps.business.activity.scan.sdk.register.ScanActivityRewardRegister;
import com.biz.crm.cps.business.activity.scan.sdk.service.ScanActivityRewardVoService;
import com.biz.crm.cps.business.activity.scan.sdk.vo.ScanActivityRebateValueVo;
import com.biz.crm.cps.business.activity.scan.sdk.vo.ScanActivityRebateVo;
import com.biz.crm.cps.business.activity.scan.sdk.vo.ScanActivityRewardObserverVo;
import com.biz.crm.cps.business.activity.scan.sdk.vo.ScanActivityRewardValueVo;
import com.biz.crm.cps.business.activity.scan.sdk.vo.ScanActivityRewardVo;
import com.biz.crm.cps.external.barcode.sdk.dto.ScanCodeRecordEventDto;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * 扫码活动规则接口实现
 *
 * @author sunx
 * @date 2022/2/21
 */
@Slf4j
@Service("scanActivityRewardVoService")
public class ScanActivityRewardVoServiceImpl implements ScanActivityRewardVoService {

  @Autowired private ScanActivityRewardRepository scanActivityRewardRepository;

  @Autowired private ScanActivityRewardValueRepository scanActivityRewardValueRepository;

  @Autowired private ScanActivityRebateRepository scanActivityRebateRepository;

  @Autowired private ScanActivityRebateValueRepository scanActivityRebateValueRepository;

  @Autowired(required = false)
  private List<ScanActivityRewardRegister> rewardRegisters;

  @Autowired(required = false)
  private GenerateCodeService generateCodeService;

  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Override
  @Transactional
  public void createReward(List<ScanActivityRewardDto> rewardList, String actCode) {
    Validate.notBlank(actCode, "扫码活动编码不能为空");
    this.validate(rewardList);
    // 删除历史数据
    this.scanActivityRewardRepository.deleteByActCode(actCode);
    this.scanActivityRewardValueRepository.deleteByActCode(actCode);
    this.scanActivityRebateRepository.deleteByActCode(actCode);
    this.scanActivityRebateValueRepository.deleteByActCode(actCode);

    List<String> rewardCodeList =
        this.generateCodeService.generateCode(
            ScanActivityConstant.SCAN_ACTIVITY_REWARD_CODE, rewardList.size());
    List<String> rebateCodeList = Lists.newLinkedList();

    Integer rebateCount = 0;
    for (ScanActivityRewardDto item : rewardList) {
      if (CollectionUtils.isNotEmpty(item.getRebateList())) {
        rebateCount += item.getRebateList().size();
      }
    }
    if (rebateCount > 0) {
      rebateCodeList =
          this.generateCodeService.generateCode(
              ScanActivityConstant.SCAN_ACTIVITY_REBATE_CODE, rebateCount);
    }

    List<ScanActivityReward> rewards = Lists.newLinkedList();
    List<ScanActivityRewardValue> rewardValues = Lists.newLinkedList();
    List<ScanActivityRebate> rebates = Lists.newLinkedList();
    List<ScanActivityRebateValue> rebateValues = Lists.newLinkedList();

    // 获取需要保存的信息
    this.initNeedSaveInfo(
        rewardList,
        actCode,
        rewards,
        rewardValues,
        rebates,
        rebateValues,
        rewardCodeList,
        rebateCodeList);

    if (CollectionUtils.isNotEmpty(rewards)) {
      this.scanActivityRewardRepository.saveBatch(rewards);
    }
    if (CollectionUtils.isNotEmpty(rewardValues)) {
      this.scanActivityRewardValueRepository.saveBatch(rewardValues);
    }
    if (CollectionUtils.isNotEmpty(rebates)) {
      this.scanActivityRebateRepository.saveBatch(rebates);
    }
    if (CollectionUtils.isNotEmpty(rebateValues)) {
      this.scanActivityRebateValueRepository.saveBatch(rebateValues);
    }
  }

  @Override
  public List<ScanActivityRewardVo> findByActCode(String actCode) {
    if (StringUtils.isBlank(actCode)) {
      return Lists.newLinkedList();
    }
    // 获取关联信息
    List<ScanActivityReward> rewards = this.scanActivityRewardRepository.findByActCode(actCode);
    if (CollectionUtils.isEmpty(rewards)) {
      return Lists.newLinkedList();
    }
    List<ScanActivityRewardValue> rewardValues =
        this.scanActivityRewardValueRepository.findByActCode(actCode);
    List<ScanActivityRebate> rebates = this.scanActivityRebateRepository.findByActCode(actCode);
    List<ScanActivityRebateValue> rebateValues =
        this.scanActivityRebateValueRepository.findByActCode(actCode);

    List<ScanActivityRewardVo> rewardVos =
        (List<ScanActivityRewardVo>)
            this.nebulaToolkitService.copyCollectionByBlankList(
                rewards,
                ScanActivityReward.class,
                ScanActivityRewardVo.class,
                HashSet.class,
                ArrayList.class);
    List<ScanActivityRewardValueVo> rewardValueVos = Lists.newLinkedList();
    List<ScanActivityRebateVo> rebateVos = Lists.newLinkedList();
    List<ScanActivityRebateValueVo> rebateValueVos = Lists.newLinkedList();

    if (CollectionUtils.isNotEmpty(rewardValues)) {
      rewardValueVos =
          (List<ScanActivityRewardValueVo>)
              this.nebulaToolkitService.copyCollectionByBlankList(
                  rewardValues,
                  ScanActivityRewardValue.class,
                  ScanActivityRewardValueVo.class,
                  HashSet.class,
                  ArrayList.class);
    }

    if (CollectionUtils.isNotEmpty(rebates)) {
      rebateVos =
          (List<ScanActivityRebateVo>)
              this.nebulaToolkitService.copyCollectionByBlankList(
                  rebates,
                  ScanActivityRebate.class,
                  ScanActivityRebateVo.class,
                  HashSet.class,
                  ArrayList.class);
    }

    if (CollectionUtils.isNotEmpty(rebateValues)) {
      rebateValueVos =
          (List<ScanActivityRebateValueVo>)
              this.nebulaToolkitService.copyCollectionByBlankList(
                  rebateValues,
                  ScanActivityRebateValue.class,
                  ScanActivityRebateValueVo.class,
                  HashSet.class,
                  ArrayList.class);
    }

    return this.findRewards(rewardVos, rewardValueVos, rebateVos, rebateValueVos);
  }

  @Override
  public List<ScanActivityRewardObserverVo> findScanActivityRewardObserverVos(
      ScanCodeRecordEventDto dto, List<ScanActivityRewardVo> rewardList) {
    ScanActivityRewardScanTypeEnum rewardScanTypeEnum =
        ScanActivityRewardScanTypeEnum.findByBarCodeType(dto.getBarCodeType());
    if (Objects.isNull(rewardScanTypeEnum)
        || StringUtils.isAnyBlank(
            dto.getBarCode(), dto.getProductCode(), dto.getScanParticipatorCode())
        || CollectionUtils.isEmpty(rewardList)) {
      return null;
    }
    ScanActivityParticipateObjEnum participateObj =
        ScanActivityParticipateObjEnum.findByTransformKey(dto.getParticipatorType());
    ScanActivityRewardRegister rewardRegister = this.findRewardRegister(participateObj);
    if (Objects.isNull(rewardRegister)) {
      return null;
    }
    return rewardRegister.findScanActivityRewardObserverVos(dto, rewardList);
  }

  /**
   * 获取需要保存的信息
   *
   * @param rewardList
   * @param actCode
   * @param rewards
   * @param rewardValues
   * @param rebates
   * @param rebateValues
   * @param rewardCodeList
   * @param rebateCodeList
   */
  private void initNeedSaveInfo(
      List<ScanActivityRewardDto> rewardList,
      String actCode,
      List<ScanActivityReward> rewards,
      List<ScanActivityRewardValue> rewardValues,
      List<ScanActivityRebate> rebates,
      List<ScanActivityRebateValue> rebateValues,
      List<String> rewardCodeList,
      List<String> rebateCodeList) {
    int c1 = 0;
    int c2 = 0;
    for (ScanActivityRewardDto item : rewardList) {
      ScanActivityReward reward =
          this.nebulaToolkitService.copyObjectByBlankList(
              item, ScanActivityReward.class, HashSet.class, ArrayList.class);
      reward.setId(StringUtils.EMPTY);
      reward.setActCode(actCode);
      reward.setRewardCode(rewardCodeList.get(c1++));
      reward.setSort(c1);
      rewards.add(reward);
      if (CollectionUtils.isNotEmpty(item.getValueList())) {
        int i = 1;
        for (ScanActivityRewardValueDto itemValue : item.getValueList()) {
          ScanActivityRewardValue value =
              this.nebulaToolkitService.copyObjectByBlankList(
                  itemValue, ScanActivityRewardValue.class, HashSet.class, ArrayList.class);
          value.setId(StringUtils.EMPTY);
          value.setActCode(actCode);
          value.setRewardCode(reward.getRewardCode());
          value.setSort(i++);
          rewardValues.add(value);
        }
      }
      if (CollectionUtils.isNotEmpty(item.getRebateList())) {
        for (ScanActivityRebateDto item1 : item.getRebateList()) {
          ScanActivityRebate rebate =
              this.nebulaToolkitService.copyObjectByBlankList(
                  item1, ScanActivityRebate.class, HashSet.class, ArrayList.class);
          rebate.setId(StringUtils.EMPTY);
          rebate.setActCode(actCode);
          rebate.setRebateCode(rebateCodeList.get(c2++));
          rebate.setRewardCode(reward.getRewardCode());
          rebate.setSort(c2);
          rebates.add(rebate);
          if (CollectionUtils.isNotEmpty(item1.getValueList())) {
            int i = 1;
            for (ScanActivityRebateValueDto item1Value : item1.getValueList()) {
              ScanActivityRebateValue value1 =
                  this.nebulaToolkitService.copyObjectByBlankList(
                      item1Value, ScanActivityRebateValue.class, HashSet.class, ArrayList.class);
              value1.setId(StringUtils.EMPTY);
              value1.setActCode(actCode);
              value1.setRebateCode(rebate.getRebateCode());
              value1.setRebateValue(item1Value.getRebateValue());
              value1.setSort(i++);
              rebateValues.add(value1);
            }
          }
        }
      }
    }
  }

  /**
   * 获取返回的规则详情
   *
   * @param rewardVos
   * @param rewardValueVos
   * @param rebateVos
   * @param rebateValueVos
   * @return
   */
  private List<ScanActivityRewardVo> findRewards(
      List<ScanActivityRewardVo> rewardVos,
      List<ScanActivityRewardValueVo> rewardValueVos,
      List<ScanActivityRebateVo> rebateVos,
      List<ScanActivityRebateValueVo> rebateValueVos) {
    if (CollectionUtils.isEmpty(rewardVos)) {
      return Lists.newLinkedList();
    }
    // k-rewardCode, v-list
    Map<String, List<ScanActivityRewardValueVo>> mapRewardValue = Maps.newHashMap();
    // k-rewardCode, v-list
    Map<String, List<ScanActivityRebateVo>> mapRebate = Maps.newHashMap();
    // k-rebateCode, v-list
    Map<String, List<ScanActivityRebateValueVo>> mapRebateValue = Maps.newHashMap();
    if (CollectionUtils.isNotEmpty(rewardValueVos)) {
      mapRewardValue =
          rewardValueVos.stream()
              .collect(Collectors.groupingBy(ScanActivityRewardValueVo::getRewardCode));
    }
    if (CollectionUtils.isNotEmpty(rebateVos)) {
      mapRebate =
          rebateVos.stream().collect(Collectors.groupingBy(ScanActivityRebateVo::getRewardCode));
    }
    if (CollectionUtils.isNotEmpty(rebateValueVos)) {
      mapRebateValue =
          rebateValueVos.stream()
              .collect(Collectors.groupingBy(ScanActivityRebateValueVo::getRebateCode));
    }

    for (ScanActivityRewardVo item : rewardVos) {
      item.setValueList(mapRewardValue.get(item.getRewardCode()));
      List<ScanActivityRebateVo> rebateVoList = mapRebate.get(item.getRewardCode());
      if (CollectionUtils.isNotEmpty(rebateVoList)) {
        for (ScanActivityRebateVo rebateVo : rebateVoList) {
          rebateVo.setValueList(mapRebateValue.get(rebateVo.getRebateCode()));
        }
        item.setRebateList(rebateVoList);
      }
    }
    return rewardVos;
  }

  /**
   * 验证数据
   *
   * @param rewardList
   */
  private void validate(List<ScanActivityRewardDto> rewardList) {
    Validate.isTrue(CollectionUtils.isNotEmpty(rewardList), "扫码活动规则配置不能为空");

    Set<String> set1 = Sets.newHashSet();
    for (ScanActivityRewardDto item : rewardList) {
      Validate.isTrue(
          set1.add(StringUtils.join(item.getType(), "-", item.getRewardType())), "正向奖励不能重复");
      Validate.isTrue(CollectionUtils.isNotEmpty(item.getValueList()), "正向奖励配置明细不能为空");
      if (Boolean.FALSE.equals(item.getRandomFlag())) {
        Validate.isTrue(1 == item.getValueList().size(), "正向奖励固定额度时，对应的明细配置错误");
      }
      if (CollectionUtils.isEmpty(item.getRebateList())) {
        continue;
      }
      Set<String> set2 = Sets.newHashSet();
      for (ScanActivityRebateDto sub : item.getRebateList()) {
        Validate.isTrue(
            set2.add(StringUtils.join(sub.getObjType(), "-", sub.getRewardType())), "反向奖励不能重复");
        Validate.isTrue(CollectionUtils.isNotEmpty(sub.getValueList()), "反向奖励配置明细不能为空");
        if (Boolean.FALSE.equals(sub.getRandomFlag())) {
          Validate.isTrue(1 == sub.getValueList().size(), "反向奖励固定额度时，对应的明细配置错误");
        }
      }
    }
  }

  /**
   * 获取奖励通知构建注册器
   *
   * @param participateObj
   * @return
   */
  private ScanActivityRewardRegister findRewardRegister(
      ScanActivityParticipateObjEnum participateObj) {
    if (CollectionUtils.isEmpty(rewardRegisters) || Objects.isNull(participateObj)) {
      return null;
    }
    Optional<ScanActivityRewardRegister> first =
        rewardRegisters.stream()
            .filter(a -> participateObj.getDictCode().equals(a.getKey()))
            .findFirst();
    return first.orElse(null);
  }
}
