package com.biz.crm.cps.business.activity.scan.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.cps.business.activity.scan.local.entity.ScanActivity;
import com.biz.crm.cps.business.activity.scan.local.entity.ScanActivityParticipateScope;
import com.biz.crm.cps.business.activity.scan.local.entity.ScanActivityProductScopeItem;
import com.biz.crm.cps.business.activity.scan.local.entity.ScanActivityUserScopeItem;
import com.biz.crm.cps.business.activity.scan.local.model.dto.ScanActivityCheckRepeatModelDto;
import com.biz.crm.cps.business.activity.scan.local.repository.ScanActivityRepository;
import com.biz.crm.cps.business.activity.scan.local.service.ScanActivityParticipateScopeService;
import com.biz.crm.cps.business.activity.scan.local.service.ScanActivityProductScopeItemService;
import com.biz.crm.cps.business.activity.scan.local.service.ScanActivityService;
import com.biz.crm.cps.business.activity.scan.local.service.ScanActivityUserScopeItemService;
import com.biz.crm.cps.business.activity.scan.sdk.dto.ScanActivityPaginationDto;
import com.biz.crm.cps.business.activity.scan.sdk.dto.ScanActivityParticipateScopeDto;
import com.biz.crm.cps.business.activity.scan.sdk.dto.ScanActivityProductScopeDto;
import com.biz.crm.cps.business.activity.scan.sdk.enums.ScanActivityParticipateTypeEnum;
import com.biz.crm.cps.business.activity.scan.sdk.enums.ScanActivityScopeEnum;
import com.biz.crm.cps.business.activity.scan.sdk.enums.ScanActivityStatusEnum;
import com.biz.crm.cps.business.activity.scan.sdk.service.ScanActivityParticipateScopeVoService;
import com.biz.crm.cps.business.activity.scan.sdk.service.ScanActivityProductScopeVoService;
import com.biz.crm.cps.business.activity.scan.sdk.vo.ScanActivityProductScopeVo;
import com.biz.crm.cps.business.common.sdk.enums.EnableStatusEnum;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.google.common.collect.Sets.SetView;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

/**
 * 扫码活动(ScanActivity)表服务实现类
 *
 * @author sunx
 * @date 2022-02-16 16:39:01
 */
@Slf4j
@Service("scanActivityService")
public class ScanActivityServiceImpl implements ScanActivityService {

  @Autowired private ScanActivityRepository scanActivityRepository;

  @Autowired private ScanActivityUserScopeItemService scanActivityUserScopeItemService;

  @Autowired private ScanActivityProductScopeItemService scanActivityProductScopeItemService;

  @Autowired private ScanActivityParticipateScopeVoService scanActivityParticipateScopeVoService;

  @Autowired private ScanActivityParticipateScopeService scanActivityParticipateScopeService;

  @Autowired private ScanActivityProductScopeVoService scanActivityProductScopeVoService;

  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Override
  public Page<ScanActivity> findByConditions(Pageable pageable, ScanActivityPaginationDto dto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    dto = Optional.ofNullable(dto).orElse(new ScanActivityPaginationDto());
    Page<ScanActivity> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    Page<ScanActivity> pageResult = this.scanActivityRepository.findByConditions(page, dto);
    if (Objects.nonNull(pageResult) && !CollectionUtils.isEmpty(pageResult.getRecords())) {
      for (ScanActivity item : pageResult.getRecords()) {
        item.setActStatus(this.findActStatus(item));
      }
    }
    return pageResult;
  }

  /**
   * 获取活动状态
   *
   * @param item
   * @return
   */
  private String findActStatus(ScanActivity item) {
    if (Objects.isNull(item)
        || Objects.isNull(item.getStartTime())
        || Objects.isNull(item.getEndTime())) {
      return StringUtils.EMPTY;
    }
    Date now = new Date();
    if (now.compareTo(item.getStartTime()) < 0) {
      return ScanActivityStatusEnum.DEFAULT.getValue();
    }
    if (now.compareTo(item.getEndTime()) > 0) {
      return ScanActivityStatusEnum.FINISH.getValue();
    }
    return ScanActivityStatusEnum.RUNNING.getValue();
  }

  @Override
  public ScanActivity findDetailById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.scanActivityRepository.getById(id);
  }

  @Override
  public List<ScanActivity> findDetailByCodes(Set<String> codeSet) {
    if (CollectionUtils.isEmpty(codeSet)) {
      return Lists.newLinkedList();
    }
    return this.scanActivityRepository.findDetailsByCodes(codeSet);
  }

  @Override
  @Transactional
  public ScanActivity create(ScanActivity scanActivity) {
    Validate.notNull(scanActivity, "扫码活动信息不能为空");
    this.scanActivityRepository.saveOrUpdate(scanActivity);
    return scanActivity;
  }

  @Override
  @Transactional
  public ScanActivity update(ScanActivity scanActivity) {
    Validate.notNull(scanActivity, "扫码活动信息不能为空");
    Validate.isTrue(StringUtils.isNotBlank(scanActivity.getId()), "扫码活动id不能为空");
    String currentId = scanActivity.getId();
    ScanActivity current = this.scanActivityRepository.getById(currentId);
    Validate.notNull(current, "修改信息不存在");
    Validate.isTrue(scanActivity.getCode().equals(current.getCode()), "扫码活动编码不能修改");
    this.scanActivityRepository.saveOrUpdate(scanActivity);
    return scanActivity;
  }

  @Override
  @Transactional
  public void enable(String id) {
    Validate.notBlank(id, "id不能为空");
    ScanActivity activity = this.scanActivityRepository.getById(id);
    Validate.notNull(activity, "扫码活动不存在");
    Validate.isTrue(
        !EnableStatusEnum.ENABLE.getCode().equals(activity.getEnableStatus()), "扫码活动已启用，无需执行该操作!");
    // 启用扫码活动详细逻辑
    activity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    this.handleEnableActivity(activity);
    this.scanActivityRepository.updateEnableStatusByIds(
        Lists.newArrayList(id), EnableStatusEnum.ENABLE);
  }

  @Override
  @Transactional
  public void disable(String id) {
    Validate.notBlank(id, "id不能为空");
    ScanActivity activity = this.scanActivityRepository.getById(id);
    Validate.notNull(activity, "扫码活动不存在");
    Validate.isTrue(
        !EnableStatusEnum.DISABLE.getCode().equals(activity.getEnableStatus()), "扫码活动已禁用，无需执行该操作!");
    this.scanActivityRepository.updateEnableStatusByIds(
        Lists.newArrayList(id), EnableStatusEnum.DISABLE);
  }

  /**
   * 启用扫码活动详细逻辑
   *
   * @param activity
   */
  @Transactional
  public void handleEnableActivity(ScanActivity activity) {
    Validate.isTrue(
        EnableStatusEnum.ENABLE.getCode().equals(activity.getEnableStatus()), "扫码活动已启用，无需执行该操作!");
    // 是自动参与
    Boolean flag =
        ScanActivityParticipateTypeEnum.AUTOMATIC
            .getDictCode()
            .equals(activity.getParticipateType());
    String participateScope = activity.getParticipateScope();
    Set<String> userCodeSet = this.findParticipateScopeCodeSet(activity, participateScope);
    Set<String> materialCodeSet = this.findMaterialCodeSet(activity);

    // 重要的验证逻辑
    ScanActivityCheckRepeatModelDto repeatModelDto = new ScanActivityCheckRepeatModelDto();
    repeatModelDto.setActCode(activity.getCode());
    repeatModelDto.setActStartTime(activity.getStartTime());
    repeatModelDto.setActEndTime(activity.getEndTime());
    repeatModelDto.setParticipateObj(activity.getParticipateObj());
    repeatModelDto.setParticipateScope(activity.getParticipateScope());
    repeatModelDto.setUserCodeSet(userCodeSet);
    repeatModelDto.setMaterialCodeSet(materialCodeSet);
    this.validateRepeat(repeatModelDto);

    // 保存物料关联活动信息
    List<ScanActivityProductScopeItem> productScopeItemList = Lists.newLinkedList();
    if (!CollectionUtils.isEmpty(materialCodeSet)) {
      for (String item : materialCodeSet) {
        ScanActivityProductScopeItem cur = new ScanActivityProductScopeItem();
        cur.setActCode(activity.getCode());
        cur.setMaterialCode(item);
        productScopeItemList.add(cur);
      }
    }
    this.scanActivityProductScopeItemService.createScopeItem(
        productScopeItemList, activity.getCode());

    // 保存参与人员关联活动明细
    List<ScanActivityUserScopeItem> userScopeItemList = Lists.newLinkedList();
    if (!CollectionUtils.isEmpty(userCodeSet)) {
      Date now = new Date();
      for (String item : userCodeSet) {
        ScanActivityUserScopeItem cur = new ScanActivityUserScopeItem();
        cur.setActCode(activity.getCode());
        cur.setUserCode(item);
        cur.setParticipateObj(activity.getParticipateObj());
        cur.setJoinFlag(flag);
        if (Boolean.TRUE.equals(flag)) {
          cur.setCreateTime(now);
        }
        userScopeItemList.add(cur);
      }
    }
    this.scanActivityUserScopeItemService.createScopeItem(userScopeItemList, activity.getCode());
  }

  /**
   * 验证重复-工具方法
   *
   * @param repeatModelDto
   */
  @Override
  public void validateRepeat(ScanActivityCheckRepeatModelDto repeatModelDto) {
    if (Objects.isNull(repeatModelDto)) {
      Validate.isTrue(false, "活动配置不能为空");
    }
    if (CollectionUtils.isEmpty(repeatModelDto.getMaterialCodeSet())
        || (!ScanActivityScopeEnum.ALL.getDictCode().equals(repeatModelDto.getParticipateScope())
            && CollectionUtils.isEmpty(repeatModelDto.getUserCodeSet()))) {
      return;
    }
    List<ScanActivity> list =
        this.scanActivityRepository.findEnableRepeatByTimeBetweenAndParticipateObj(
            repeatModelDto.getActStartTime(),
            repeatModelDto.getActEndTime(),
            repeatModelDto.getParticipateObj());
    if (CollectionUtils.isEmpty(list)) {
      return;
    }

    // 所有的活动编码
    Set<String> allActCodeSet = Sets.newHashSet();
    // 参与范围类型为全部的活动编码
    Set<String> actCodeSet1 = Sets.newHashSet();
    // 参与范围类型不为全部的活动编码
    Set<String> actCodeSet2 = Sets.newHashSet();
    for (ScanActivity item : list) {
      if (repeatModelDto.getActCode().equals(item.getCode())) {
        continue;
      }
      allActCodeSet.add(item.getCode());
      if (ScanActivityScopeEnum.ALL.getDictCode().equals(item.getParticipateScope())) {
        actCodeSet1.add(item.getCode());
      } else {
        actCodeSet2.add(item.getCode());
      }
    }
    if (CollectionUtils.isEmpty(allActCodeSet)) {
      return;
    }

    // 验证商品是否有重叠的活动
    Set<String> productScopeItemActCodeSet =
        this.scanActivityProductScopeItemService.findProductScopeItemActCodeSet(
            allActCodeSet, repeatModelDto.getMaterialCodeSet());
    productScopeItemActCodeSet =
        Optional.ofNullable(productScopeItemActCodeSet).orElse(Sets.newHashSet());

    if (ScanActivityScopeEnum.ALL.getDictCode().equals(repeatModelDto.getParticipateScope())) {
      if (!CollectionUtils.isEmpty(productScopeItemActCodeSet)) {
        Validate.isTrue(
            false, "当前活动与" + StringUtils.join(productScopeItemActCodeSet, ",") + "活动编码对应的参与信息存在重复");
      } else {
        return;
      }
    }

    // 验证参与是否有重复
    SetView<String> intersection1 = Sets.intersection(actCodeSet1, productScopeItemActCodeSet);
    if (!intersection1.isEmpty()) {
      Validate.isTrue(false, "当前活动与" + StringUtils.join(intersection1, ",") + "活动编码对应的参与信息存在重复");
    }
    if (CollectionUtils.isEmpty(actCodeSet2)) {
      return;
    }
    Set<String> userScopeItemActCodeSet =
        this.scanActivityUserScopeItemService.findUserScopeItemActCodeSet(
            repeatModelDto.getParticipateObj(), actCodeSet2, repeatModelDto.getUserCodeSet());
    SetView<String> intersection2 =
        Sets.intersection(userScopeItemActCodeSet, productScopeItemActCodeSet);
    if (!intersection2.isEmpty()) {
      Validate.isTrue(false, "当前活动与" + StringUtils.join(intersection2, ",") + "活动编码对应的参与信息存在重复");
    }
  }

  /**
   * 获取参与范围对应的业务编码集合-经销商编码、终端编码、导购编码
   *
   * @param activity
   * @param participateScope
   * @return
   */
  private Set<String> findParticipateScopeCodeSet(ScanActivity activity, String participateScope) {
    Set<String> codeSet = Sets.newHashSet();
    if (ScanActivityScopeEnum.ALL.getDictCode().equals(participateScope)) {
      return codeSet;
    }
    List<ScanActivityParticipateScope> participateScopeList =
        this.scanActivityParticipateScopeService.findByActCode(activity.getCode());
    List<ScanActivityParticipateScopeDto> dtoList = Lists.newLinkedList();
    if (!CollectionUtils.isEmpty(participateScopeList)) {
      dtoList =
          (List<ScanActivityParticipateScopeDto>)
              this.nebulaToolkitService.copyCollectionByBlankList(
                  participateScopeList,
                  ScanActivityParticipateScope.class,
                  ScanActivityParticipateScopeDto.class,
                  HashSet.class,
                  ArrayList.class);
    }
    return this.scanActivityParticipateScopeVoService.findScopeCodeSet(
        activity.getParticipateObj(), activity.getParticipateScope(), dtoList);
  }

  /**
   * 根据活动配置获取物料范围对应的物料编码集合
   *
   * @param activity
   * @return
   */
  private Set<String> findMaterialCodeSet(ScanActivity activity) {
    List<ScanActivityProductScopeVo> productScopeVoList =
        this.scanActivityProductScopeVoService.findByActCode(activity.getCode());
    List<ScanActivityProductScopeDto> scopeList = Lists.newLinkedList();
    if (!CollectionUtils.isEmpty(productScopeVoList)) {
      scopeList =
          (List<ScanActivityProductScopeDto>)
              this.nebulaToolkitService.copyCollectionByBlankList(
                  productScopeVoList,
                  ScanActivityProductScopeVo.class,
                  ScanActivityProductScopeDto.class,
                  HashSet.class,
                  ArrayList.class);
    }
    return this.scanActivityProductScopeVoService.findMaterialCodeSet(scopeList);
  }
}
