package com.biz.crm.cps.business.activity.scan.local.service.internal;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.date.DateUtil;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.cps.business.activity.scan.local.entity.ScanActivity;
import com.biz.crm.cps.business.activity.scan.local.entity.ScanActivityProductScopeItem;
import com.biz.crm.cps.business.activity.scan.local.entity.ScanActivityUserArea;
import com.biz.crm.cps.business.activity.scan.local.entity.ScanActivityUserScopeItem;
import com.biz.crm.cps.business.activity.scan.local.model.dto.ScanActivityCheckRepeatModelDto;
import com.biz.crm.cps.business.activity.scan.local.service.ScanActivityProductScopeItemService;
import com.biz.crm.cps.business.activity.scan.local.service.ScanActivityService;
import com.biz.crm.cps.business.activity.scan.local.service.ScanActivityUserAreaService;
import com.biz.crm.cps.business.activity.scan.local.service.ScanActivityUserScopeItemService;
import com.biz.crm.cps.business.activity.scan.sdk.constant.ScanActivityConstant;
import com.biz.crm.cps.business.activity.scan.sdk.dto.ScanActivityDto;
import com.biz.crm.cps.business.activity.scan.sdk.dto.ScanActivityQueryDto;
import com.biz.crm.cps.business.activity.scan.sdk.dto.ScanActivityRebateDto;
import com.biz.crm.cps.business.activity.scan.sdk.dto.ScanActivityRewardDto;
import com.biz.crm.cps.business.activity.scan.sdk.dto.ScanActivityShareProfitDto;
import com.biz.crm.cps.business.activity.scan.sdk.dto.ScanActivityUserAreaDto;
import com.biz.crm.cps.business.activity.scan.sdk.enums.ScanActivityParticipateObjEnum;
import com.biz.crm.cps.business.activity.scan.sdk.enums.ScanActivityParticipateTypeEnum;
import com.biz.crm.cps.business.activity.scan.sdk.enums.ScanActivityRewardTypeEnum;
import com.biz.crm.cps.business.activity.scan.sdk.enums.ScanActivityScopeEnum;
import com.biz.crm.cps.business.activity.scan.sdk.service.ScanActivityParticipateScopeVoService;
import com.biz.crm.cps.business.activity.scan.sdk.service.ScanActivityProductScopeVoService;
import com.biz.crm.cps.business.activity.scan.sdk.service.ScanActivityRewardVoService;
import com.biz.crm.cps.business.activity.scan.sdk.service.ScanActivityVoService;
import com.biz.crm.cps.business.activity.scan.sdk.vo.ScanActivityParticipateScopeVo;
import com.biz.crm.cps.business.activity.scan.sdk.vo.ScanActivityProductScopeVo;
import com.biz.crm.cps.business.activity.scan.sdk.vo.ScanActivityRewardAccountVo;
import com.biz.crm.cps.business.activity.scan.sdk.vo.ScanActivityRewardVo;
import com.biz.crm.cps.business.activity.scan.sdk.vo.ScanActivityUserAreaVo;
import com.biz.crm.cps.business.activity.scan.sdk.vo.ScanActivityVo;
import com.biz.crm.cps.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.cps.external.tax.raise.sdk.service.capital.TaxRaiseCapitalAccountVoService;
import com.biz.crm.cps.external.tax.raise.sdk.vo.capital.TaxRaiseCapitalAccountVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * @author sunx
 * @date 2022/2/21
 */
@Slf4j
@Service("scanActivityVoService")
public class ScanActivityVoServiceImpl implements ScanActivityVoService {

  @Autowired private ScanActivityService scanActivityService;

  @Autowired private ScanActivityRewardVoService scanActivityRewardVoService;

  @Autowired private ScanActivityProductScopeVoService scanActivityProductScopeVoService;

  @Autowired private ScanActivityParticipateScopeVoService scanActivityParticipateScopeVoService;

  @Autowired private ScanActivityUserScopeItemService scanActivityUserScopeItemService;

  @Autowired private ScanActivityProductScopeItemService scanActivityProductScopeItemService;

  @Autowired private TaxRaiseCapitalAccountVoService taxRaiseCapitalAccountVoService;

  @Autowired private ScanActivityUserAreaService scanActivityUserAreaService;

  @Autowired(required = false)
  private GenerateCodeService generateCodeService;

  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Override
  public ScanActivityVo findDetailById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    // 活动基本信息
    ScanActivityVo vo = this.findById(id);
    // 参与范围
    List<ScanActivityParticipateScopeVo> participateScopeVoList =
        this.scanActivityParticipateScopeVoService.findByParticipateObjAndActCode(
            vo.getParticipateObj(), vo.getCode());
    // 物料范围
    List<ScanActivityProductScopeVo> productScopeVoList =
        this.scanActivityProductScopeVoService.findByActCode(vo.getCode());
    // 活动规则
    List<ScanActivityRewardVo> rewardVoList =
        this.scanActivityRewardVoService.findByActCode(vo.getCode());
    // 消费者扫码活动区域配置
    List<ScanActivityUserAreaVo> areaVoList = Lists.newLinkedList();
    List<ScanActivityUserArea> areaList =
        this.scanActivityUserAreaService.findByActCode(vo.getCode());
    if (CollectionUtils.isNotEmpty(areaList)) {
      areaVoList =
          (List<ScanActivityUserAreaVo>)
              this.nebulaToolkitService.copyCollectionByBlankList(
                  areaList,
                  ScanActivityUserArea.class,
                  ScanActivityUserAreaVo.class,
                  HashSet.class,
                  ArrayList.class);
    }
    if (StringUtils.isNotBlank(vo.getAccountCode())) {
      // 红包放发账户明细
      TaxRaiseCapitalAccountVo taxRaiseCapitalAccountVo =
          this.taxRaiseCapitalAccountVoService.findByMerchantUserAccount(vo.getAccountCode());
      if (Objects.nonNull(taxRaiseCapitalAccountVo)) {
        ScanActivityRewardAccountVo accountVo = new ScanActivityRewardAccountVo();
        accountVo.setAccountCode(taxRaiseCapitalAccountVo.getAccountCode());
        accountVo.setAccountName(taxRaiseCapitalAccountVo.getAccountName());
        accountVo.setAmount(taxRaiseCapitalAccountVo.getAmount());
        vo.setAccount(accountVo);
      }
    }
    vo.setParticipateScopeList(participateScopeVoList);
    vo.setProductList(productScopeVoList);
    vo.setRewardList(rewardVoList);
    vo.setAreaList(areaVoList);
    return vo;
  }

  @Override
  public ScanActivityVo findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    // 活动基本信息
    ScanActivity scanActivity = this.scanActivityService.findDetailById(id);
    if (Objects.isNull(scanActivity)) {
      return null;
    }
    return this.nebulaToolkitService.copyObjectByBlankList(
        scanActivity, ScanActivityVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  @Transactional
  public ScanActivityVo create(ScanActivityDto dto) {
    Validate.notNull(dto, "扫码活动配置信息不能为空");
    dto.setCode(
        this.generateCodeService.generateCode(ScanActivityConstant.SCAN_ACTIVITY_CODE, 1).get(0));
    this.validate(dto);

    // 保存活动基本信息
    this.scanActivityService.create(
        this.nebulaToolkitService.copyObjectByWhiteList(
            dto, ScanActivity.class, HashSet.class, ArrayList.class));

    // 保存活动附加信息
    this.saveScanActivityExtInfo(dto);
    return null;
  }

  @Override
  @Transactional
  public ScanActivityVo update(ScanActivityDto dto) {
    Validate.notNull(dto, "扫码活动配置信息不能为空");
    this.validate(dto);

    // 保存活动基本信息
    this.scanActivityService.update(
        this.nebulaToolkitService.copyObjectByWhiteList(
            dto, ScanActivity.class, HashSet.class, ArrayList.class));

    // 保存活动附加信息
    this.saveScanActivityExtInfo(dto);
    return null;
  }

  @Override
  public String findActCodeByScanActivityQueryDto(ScanActivityQueryDto dto) {
    if (Objects.isNull(dto)
        || StringUtils.isAnyBlank(
            dto.getParticipateObj(), dto.getUserCode(), dto.getMaterialCode())) {
      return StringUtils.EMPTY;
    }
    // 获取当前人已参与正在执行的扫码活动编码集合
    Set<String> actCodeSet =
        this.scanActivityUserScopeItemService.findCurExecActCodeSetByParticipateObjAndUserCode(
            dto.getParticipateObj(), dto.getUserCode());
    if (CollectionUtils.isEmpty(actCodeSet)) {
      return StringUtils.EMPTY;
    }
    // 验证当前传入的物料编码是否包含在对应活动集合内
    Set<String> diffActCodeSet =
        this.scanActivityProductScopeItemService.findMaterialActCodeSetByMaterialCodeAndActCodeIn(
            dto.getMaterialCode(), actCodeSet);
    if (CollectionUtils.isEmpty(diffActCodeSet)) {
      return StringUtils.EMPTY;
    }
    if (!dto.getParticipateObj().equals(ScanActivityParticipateObjEnum.USER.getDictCode())) {
      Optional<String> first = diffActCodeSet.stream().findFirst();
      return first.orElse(StringUtils.EMPTY);
    }
    // 验证当前传入的省市配置是否有复核的活动
    Set<String> diffActCodeSet1 =
        this.scanActivityUserAreaService.findAreaActCodeSetByAreaAndActCodeIn(
            dto.getProvinceName(), dto.getCityName(), diffActCodeSet);
    if (CollectionUtils.isEmpty(diffActCodeSet1)) {
      return StringUtils.EMPTY;
    }
    Optional<String> first = diffActCodeSet1.stream().findFirst();
    return first.orElse(StringUtils.EMPTY);
  }

  @Override
  public List<ScanActivityVo> findMainByActCodes(Set<String> actCodeSet) {
    if (CollectionUtils.isEmpty(actCodeSet)) {
      return Lists.newLinkedList();
    }
    List<ScanActivity> list = this.scanActivityService.findDetailByCodes(actCodeSet);
    if (CollectionUtils.isEmpty(list)) {
      return Lists.newLinkedList();
    }
    return (List<ScanActivityVo>)
        this.nebulaToolkitService.copyCollectionByBlankList(
            list, ScanActivity.class, ScanActivityVo.class, HashSet.class, ArrayList.class);
  }

  /**
   * 保存活动附加信息（范围-物料范围-规则信息）
   *
   * @param dto
   */
  @Transactional
  public void saveScanActivityExtInfo(ScanActivityDto dto) {
    // 保存参与范围信息
    this.scanActivityParticipateScopeVoService.createParticipateScope(
        dto.getCode(), dto.getParticipateScope(), dto.getParticipateScopeList());
    // 保存物料规则信息
    this.scanActivityProductScopeVoService.createProductScope(dto.getProductList(), dto.getCode());
    // 保存活动规则信息
    this.scanActivityRewardVoService.createReward(dto.getRewardList(), dto.getCode());
    // 保存消费者扫码区域配置
    List<ScanActivityUserArea> areaList = Lists.newLinkedList();
    if (CollectionUtils.isNotEmpty(dto.getAreaList())) {
      areaList =
          (List<ScanActivityUserArea>)
              this.nebulaToolkitService.copyCollectionByBlankList(
                  dto.getAreaList(),
                  ScanActivityUserAreaDto.class,
                  ScanActivityUserArea.class,
                  HashSet.class,
                  ArrayList.class);
    }
    this.scanActivityUserAreaService.createArea(areaList, dto.getCode());

    if (!EnableStatusEnum.ENABLE.getCode().equals(dto.getEnableStatus())) {
      return;
    }
    // 如果扫码活动状态是启用的需要额外执行以下逻辑
    this.handleEnableActivity(dto);
  }

  /**
   * 活动启用需要执行对应的启用逻辑
   * 与com.biz.crm.cps.business.activity.scan.local.service.internal.ScanActivityServiceImpl#handleEnableActivity逻辑类似
   * 参数来源不同，逻辑类似
   *
   * @param dto
   */
  @Transactional
  public void handleEnableActivity(ScanActivityDto dto) {
    Validate.isTrue(
        EnableStatusEnum.ENABLE.getCode().equals(dto.getEnableStatus()), "扫码活动已启用，无需执行该操作!");
    // 是自动参与
    Boolean flag =
        ScanActivityParticipateTypeEnum.AUTOMATIC.getDictCode().equals(dto.getParticipateType());
    String participateScope = dto.getParticipateScope();
    Set<String> userCodeSet = Sets.newHashSet();
    if (!ScanActivityScopeEnum.ALL.getDictCode().equals(participateScope)) {
      userCodeSet =
          this.scanActivityParticipateScopeVoService.findScopeCodeSet(
              dto.getParticipateObj(), dto.getParticipateScope(), dto.getParticipateScopeList());
    }
    Set<String> materialCodeSet =
        this.scanActivityProductScopeVoService.findMaterialCodeSet(dto.getProductList());

    // 重要的验证逻辑
    ScanActivityCheckRepeatModelDto repeatModelDto = new ScanActivityCheckRepeatModelDto();
    repeatModelDto.setActCode(dto.getCode());
    repeatModelDto.setActStartTime(dto.getStartTime());
    repeatModelDto.setActEndTime(dto.getEndTime());
    repeatModelDto.setParticipateObj(dto.getParticipateObj());
    repeatModelDto.setParticipateScope(dto.getParticipateScope());
    repeatModelDto.setUserCodeSet(userCodeSet);
    repeatModelDto.setMaterialCodeSet(materialCodeSet);
    this.scanActivityService.validateRepeat(repeatModelDto);

    // 保存物料关联活动信息
    List<ScanActivityProductScopeItem> productScopeItemList = Lists.newLinkedList();
    if (!org.springframework.util.CollectionUtils.isEmpty(materialCodeSet)) {
      for (String item : materialCodeSet) {
        ScanActivityProductScopeItem cur = new ScanActivityProductScopeItem();
        cur.setActCode(dto.getCode());
        cur.setMaterialCode(item);
        productScopeItemList.add(cur);
      }
    }
    this.scanActivityProductScopeItemService.createScopeItem(productScopeItemList, dto.getCode());

    // 保存参与人员关联活动明细
    List<ScanActivityUserScopeItem> userScopeItemList = Lists.newLinkedList();
    if (!org.springframework.util.CollectionUtils.isEmpty(userCodeSet)) {
      Date now = new Date();
      for (String item : userCodeSet) {
        ScanActivityUserScopeItem cur = new ScanActivityUserScopeItem();
        cur.setActCode(dto.getCode());
        cur.setUserCode(item);
        cur.setParticipateObj(dto.getParticipateObj());
        cur.setJoinFlag(flag);
        if (Boolean.TRUE.equals(flag)) {
          cur.setCreateTime(now);
        }
        userScopeItemList.add(cur);
      }
    }
    this.scanActivityUserScopeItemService.createScopeItem(userScopeItemList, dto.getCode());
  }

  /**
   * 参数验证
   *
   * @param dto
   */
  private void validate(ScanActivityDto dto) {
    Validate.notBlank(dto.getParticipateObj(), "参与对象不能为空");
    Validate.notBlank(dto.getCode(), "扫码活动编码不能为空");
    Validate.isTrue(
        Objects.nonNull(dto.getStartTime())
            && Objects.nonNull(dto.getEndTime())
            && dto.getEndTime().compareTo(dto.getStartTime()) > 0,
        "活动有效期不能为空且结束时间大于开始时间");
    Validate.notBlank(dto.getParticipateScope(), "参与范围不能为空");
    Validate.notBlank(dto.getProductScope(), "物料范围不能为空");
    if (!dto.getParticipateScope().equals(ScanActivityScopeEnum.ALL.getDictCode())) {
      Validate.isTrue(CollectionUtils.isNotEmpty(dto.getProductList()), "物料范围配置信息不能为空");
    }
    Validate.isTrue(CollectionUtils.isNotEmpty(dto.getRewardList()), "活动规则信息不能为空");
    Validate.notNull(dto.getParticipateType(), "参与类型不能为空");
    if (ScanActivityParticipateTypeEnum.MANUAL.getDictCode().equals(dto.getParticipateType())) {
      Validate.isTrue(
          Objects.nonNull(dto.getParticipateStartTime())
              && Objects.nonNull(dto.getParticipateEndTime())
              && dto.getParticipateEndTime().compareTo(dto.getParticipateStartTime()) > 0,
          "扫码活动参与有效期不能为空且结束时间大于开始时间");
    }

    // 是否包含红包规则
    Boolean f1 = false;

    for (ScanActivityRewardDto item : dto.getRewardList()) {
      if (ScanActivityRewardTypeEnum.RED_PACKET.getDictCode().equals(item.getRewardType())) {
        f1 = true;
        break;
      }
      if (CollectionUtil.isNotEmpty(item.getRebateList())) {
        for (ScanActivityRebateDto sub : item.getRebateList()) {
          if (ScanActivityRewardTypeEnum.RED_PACKET.getDictCode().equals(sub.getRewardType())) {
            f1 = true;
            break;
          }
        }
      }
    }
    if (Boolean.TRUE.equals(f1)) {
      Validate.isTrue(
          StringUtils.isNoneBlank(dto.getAccountCode()) && Objects.nonNull(dto.getAccount()),
          "请选择发放红包的充值账户");
    } else {
      Validate.isTrue(StringUtils.isBlank(dto.getAccountCode()), "无需添加充值账户");
    }

    Validate.notBlank(dto.getEnableStatus(), "启用/禁用状态不能为空");

    if (EnableStatusEnum.ENABLE.getCode().equals(dto.getEnableStatus()) && Boolean.TRUE.equals(f1)) {
      Validate.isTrue(
          Optional.ofNullable(dto.getAccount().getAmount())
                  .orElse(BigDecimal.ZERO)
                  .compareTo(BigDecimal.ZERO)
              > 0,
          "所选红包放发账户可用余额需大于0");
    }

    if (ScanActivityParticipateObjEnum.USER.getDictCode().equals(dto.getParticipateObj())
        && Boolean.TRUE.equals(dto.getAreaFlag())) {
      Validate.isTrue(CollectionUtils.isNotEmpty(dto.getAreaList()), "请配置扫码区域限制信息");
      Set<String> set = Sets.newHashSet();
      for (ScanActivityUserAreaDto item : dto.getAreaList()) {
        Validate.notBlank(item.getProvinceCode(), "请选择省份信息");
        boolean f =
            set.add(
                StringUtils.join(
                    item.getProvinceCode(),
                    "-",
                    Optional.ofNullable(item.getActCode()).orElse("")));
        Validate.isTrue(f, "存在重复的区域配置信息");
      }
    }
  }

  /**
   * 分利参数验证
   *
   * @param dto
   */
  private void shareProfitValidate(ScanActivityShareProfitDto dto) {
    Validate.notNull(dto, "分利参数为空!");
    Validate.isTrue(
        StringUtils.isNoneBlank(
            dto.getBarCode(),
            dto.getBarCodeType(),
            dto.getActCode(),
            dto.getParticipateObj(),
            dto.getProductCode(),
            dto.getUserCode()),
        "分利参数异常");
  }
}
