package com.biz.crm.cps.business.activity.scan.local.service.register.reward;

import com.biz.crm.business.common.sdk.model.LoginUserDetailsForCPS;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.cps.business.activity.scan.sdk.enums.ScanActivityParticipateObjEnum;
import com.biz.crm.cps.business.activity.scan.sdk.enums.ScanActivityRebateObjTypeEnum;
import com.biz.crm.cps.business.activity.scan.sdk.enums.ScanActivityRewardScanTypeEnum;
import com.biz.crm.cps.business.activity.scan.sdk.enums.ScanActivityRewardTypeEnum;
import com.biz.crm.cps.business.activity.scan.sdk.register.ScanActivityRewardRegister;
import com.biz.crm.cps.business.activity.scan.sdk.vo.RewardValueVo;
import com.biz.crm.cps.business.activity.scan.sdk.vo.ScanActivityRebateValueVo;
import com.biz.crm.cps.business.activity.scan.sdk.vo.ScanActivityRebateVo;
import com.biz.crm.cps.business.activity.scan.sdk.vo.ScanActivityRewardObserverVo;
import com.biz.crm.cps.business.activity.scan.sdk.vo.ScanActivityRewardValueVo;
import com.biz.crm.cps.business.activity.scan.sdk.vo.ScanActivityRewardVo;
import com.biz.crm.cps.business.participator.sdk.service.DealerVoService;
import com.biz.crm.cps.business.participator.sdk.service.TerminalVoService;
import com.biz.crm.cps.business.participator.sdk.vo.DealerVo;
import com.biz.crm.cps.business.participator.sdk.vo.TerminalVo;
import com.biz.crm.cps.external.barcode.sdk.dto.ScanBarCodeRecordFindDto;
import com.biz.crm.cps.external.barcode.sdk.dto.ScanCodeRecordEventDto;
import com.biz.crm.cps.external.barcode.sdk.service.ScanCodeRecordVoService;
import com.biz.crm.mdm.business.terminal.sdk.service.TerminalSupplyVoService;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import java.util.stream.Collectors;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 终端奖励通知信息构建实现
 *
 * @author sunx
 * @date 2022/3/11
 */
@Component
public class TerminalScanActivityRewardRegisterImpl implements ScanActivityRewardRegister {

  @Autowired private TerminalSupplyVoService terminalSupplyVoService;

  @Autowired private ScanCodeRecordVoService scanCodeRecordVoService;

  @Autowired(required = false)
  private LoginUserService loginUserService;

  @Autowired private DealerVoService dealerVoService;

  @Autowired private TerminalVoService terminalVoService;

  @Override
  public String getKey() {
    return ScanActivityParticipateObjEnum.TERMINAL.getDictCode();
  }

  @Override
  public List<ScanActivityRewardObserverVo> findScanActivityRewardObserverVos(
      ScanCodeRecordEventDto dto, List<ScanActivityRewardVo> rewardList) {
    List<ScanActivityRewardObserverVo> list = Lists.newLinkedList();
    ScanActivityRewardScanTypeEnum rewardScanTypeEnum =
        ScanActivityRewardScanTypeEnum.findByBarCodeType(dto.getBarCodeType());
    List<ScanActivityRewardVo> curRewardList =
        rewardList.stream()
            .filter(a -> rewardScanTypeEnum.getDictCode().equals(a.getType()))
            .collect(Collectors.toList());
    if (CollectionUtils.isEmpty(curRewardList)) {
      return list;
    }
    // 是否有经销商返利规则配置
    Boolean flag = Boolean.FALSE;
    for (ScanActivityRewardVo item : curRewardList) {
      List<ScanActivityRebateVo> rebateList = item.getRebateList();
      if (!CollectionUtils.isEmpty(rebateList)) {
        for (ScanActivityRebateVo sub : rebateList) {
          if (ScanActivityRebateObjTypeEnum.CUSTOMER.getDictCode().equals(sub.getObjType())) {
            flag = true;
            break;
          }
        }
      }
    }
    String customerCode =
        this.findRebateCustomerCode(
            flag,
            dto.getScanParticipatorCode(),
            dto.getBarCode(),
            rewardScanTypeEnum.getBarCodeType(),
            dto.getProductCode());

    LoginUserDetailsForCPS loginUser = loginUserService.getLoginDetails(LoginUserDetailsForCPS.class);
    Validate.notNull(loginUser, "未获取到操作人信息");
    String consumerName = loginUser.getConsumerName();
    // 终端编码
    String terminalCode = dto.getScanParticipatorCode();
    List<TerminalVo> terminalVoList =
        this.terminalVoService.findByTerminalCodes(Lists.newArrayList(terminalCode));
    Validate.isTrue(!CollectionUtils.isEmpty(terminalVoList), "终端不存在");
    TerminalVo terminalVo = terminalVoList.stream().findFirst().get();
    for (ScanActivityRewardVo item : curRewardList) {
      ScanActivityRewardObserverVo cur = new ScanActivityRewardObserverVo();
      cur.setScanActCode(dto.getScanActCode());
      cur.setAccountCode(dto.getAccountCode());
      cur.setMaterialCode(dto.getProductCode());
      cur.setTriggerAction(StringUtils.join("扫", rewardScanTypeEnum.getValue()));
      cur.setRecordCode(dto.getRecordCode());
      ScanActivityRewardTypeEnum rewardTypeEnum =
          ScanActivityRewardTypeEnum.findByDictCode(item.getRewardType());
      if (Objects.isNull(rewardTypeEnum)) {
        continue;
      }
      cur.setRewardType(rewardTypeEnum.getTransformKey());
      cur.setParticipateObj(ScanActivityParticipateObjEnum.TERMINAL.getTransformKey());
      cur.setUserCode(dto.getScanParticipatorCode());
      cur.setName(consumerName);
      cur.setOrgCode(terminalVo.getOrganizationCode());
      cur.setOrgName(terminalVo.getOrganization());
      cur.setChannel(terminalVo.getChannel());
      cur.setRandomFlag(item.getRandomFlag());
      cur.setValueList(this.findRewardValueList(item.getValueList()));
      cur.setItemList(this.findRebateItem(cur, customerCode, item.getRebateList()));
      list.add(cur);
    }
    return list;
  }

  /**
   * 获取反向分利的经销商编码
   *
   * <p>1、当供货关系不存商品时，终端根据产品的经销商扫码记录来进行返利，没有查询到经销商扫码记录就不返
   *
   * <p>2、当供货关系存在商品时，终端根据商品所对应的经销商进行返利
   *
   * <p>3、当供货关系存在商品且多个经销商的供货商品重复时，根据产品的经销商扫码记录返利，没有查询到经销商扫码记录就不返
   *
   * @param flag 是否存在经销商反向分利奖励规则 true 存在 false 不存在
   * @param userCode
   * @param barCode
   * @param barCodeType
   * @param materialCode
   * @return
   */
  private String findRebateCustomerCode(
      boolean flag, String userCode, String barCode, String barCodeType, String materialCode) {
    if (!flag) {
      return StringUtils.EMPTY;
    }
    // 获取存在供货关系的经销商编码集合
    Set<String> customerCodeSet =
        this.terminalSupplyVoService.findTerminalSupplyCustomerCodeSet(userCode, materialCode);
    if (CollectionUtils.isEmpty(customerCodeSet)) {
      return StringUtils.EMPTY;
    }
    if (customerCodeSet.size() > 1) {
      // 获取当前条码最近扫码的经销商编码
      ScanBarCodeRecordFindDto scanBarCodeRecordFindDto = new ScanBarCodeRecordFindDto();
      scanBarCodeRecordFindDto.setBarCodeSet(Sets.newHashSet(barCode));
      scanBarCodeRecordFindDto.setBarCodeType(barCodeType);
      scanBarCodeRecordFindDto.setParticipatorType(
          ScanActivityParticipateObjEnum.CUSTOMER.getTransformKey());
      scanBarCodeRecordFindDto.setUserCodeSet(customerCodeSet);
      return this.scanCodeRecordVoService.findLastScanUserCodeByScanBarCodeRecordFindDto(
          scanBarCodeRecordFindDto);
    } else {
      return customerCodeSet.stream().findFirst().orElse(StringUtils.EMPTY);
    }
  }

  /**
   * 获取终端反向分利item数据
   *
   * @param reward
   * @param customerCode
   * @param rebateList
   * @return
   */
  private List<ScanActivityRewardObserverVo> findRebateItem(
      ScanActivityRewardObserverVo reward,
      String customerCode,
      List<ScanActivityRebateVo> rebateList) {
    if (StringUtils.isBlank(customerCode) || CollectionUtils.isEmpty(rebateList)) {
      return Lists.newLinkedList();
    }

    // 获取经销商名称
    DealerVo dealerVo = this.dealerVoService.findByCustomerCode(customerCode);
    Validate.notNull(dealerVo, "反向分利的经销商信息不存在");
    String customerName = dealerVo.getCustomerName();

    List<ScanActivityRewardObserverVo> re = Lists.newArrayList();
    for (ScanActivityRebateVo item : rebateList) {
      if (!ScanActivityRebateObjTypeEnum.CUSTOMER.getDictCode().equals(item.getObjType())) {
        continue;
      }
      ScanActivityRewardObserverVo cur = new ScanActivityRewardObserverVo();
      cur.setUserCode(customerCode);
      cur.setName(customerName);
      cur.setScanActCode(reward.getScanActCode());
      cur.setAccountCode(reward.getAccountCode());
      cur.setChannel(dealerVo.getChannelCode());
      cur.setOrgCode(dealerVo.getOrgCode());
      cur.setOrgName(dealerVo.getOrgName());
      cur.setMaterialCode(reward.getMaterialCode());
      cur.setTriggerAction(reward.getTriggerAction());
      cur.setRecordCode(reward.getRecordCode());
      ScanActivityRewardTypeEnum rewardTypeEnum =
          ScanActivityRewardTypeEnum.findByDictCode(item.getRewardType());
      if (Objects.isNull(rewardTypeEnum)) {
        continue;
      }
      cur.setRewardType(rewardTypeEnum.getTransformKey());
      cur.setRandomFlag(item.getRandomFlag());
      cur.setParticipateObj(ScanActivityParticipateObjEnum.CUSTOMER.getTransformKey());
      cur.setValueList(this.findRebateValueList(item.getValueList()));
      re.add(cur);
    }
    return re;
  }

  /**
   * 获取返利配置信息
   *
   * @param list
   * @return
   */
  private List<RewardValueVo> findRebateValueList(List<ScanActivityRebateValueVo> list) {
    if (CollectionUtils.isEmpty(list)) {
      return Lists.newLinkedList();
    }
    List<RewardValueVo> re = Lists.newArrayList();
    for (ScanActivityRebateValueVo item : list) {
      RewardValueVo cur = new RewardValueVo();
      cur.setPercent(item.getRate());
      cur.setCash(item.getRebateValue());
      re.add(cur);
    }
    return re;
  }

  /**
   * 获取奖励配置信息
   *
   * @param list
   * @return
   */
  private List<RewardValueVo> findRewardValueList(List<ScanActivityRewardValueVo> list) {
    if (CollectionUtils.isEmpty(list)) {
      return Lists.newLinkedList();
    }
    List<RewardValueVo> re = Lists.newArrayList();
    for (ScanActivityRewardValueVo item : list) {
      RewardValueVo cur = new RewardValueVo();
      cur.setPercent(item.getRate());
      cur.setCash(item.getRewardValue());
      re.add(cur);
    }
    return re;
  }
}
