package com.biz.crm.cps.business.agreement.local.repository;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.core.toolkit.StringUtils;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.cps.business.agreement.local.entity.ProfitAgreementTemplate;
import com.biz.crm.cps.business.agreement.local.mapper.ProfitAgreementTemplateMapper;
import com.biz.crm.cps.business.agreement.sdk.dto.ProfitAgreementTemplatePaginationDto;
import com.biz.crm.cps.business.common.sdk.enums.EnableStatusEnum;

import java.util.Set;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.util.List;

/**
 * 分利协议模板数据处理
 *
 * @author jerry7
 * @date 2021-07-23
 */
@Component
public class ProfitAgreementTemplateRepository extends ServiceImpl<ProfitAgreementTemplateMapper, ProfitAgreementTemplate> {

  @Resource
  private ProfitAgreementTemplateMapper profitAgreementTemplateMapper;

  /**
   * 根据主键查询分利模板详细信息（包括关联表）
   *
   * @param id 主键id
   * @return 分利模板实体信息
   */
  public ProfitAgreementTemplate findDetailsById(String id) {
    return this.profitAgreementTemplateMapper.findDetailsById(id);
  }

  /**
   * 分页查询
   *
   * queryFlag=H5时接口说明，此方法的查询条件：组织编码、渠道、标签特殊条件处理如下
   *    *       and
   *    *       case when org.org_code is not null
   *    *       then org.org_code = #{dto.orgCode}
   *    *       else 1=1 end
   *    *       and
   *    *       case when channel.channel_code is not null
   *    *       then channel.channel_code = #{dto.channelCode}
   *    *       else 1=1 end
   * @param pageable
   * @param dto
   * @return
   */
  public Page<ProfitAgreementTemplate> findByConditions(Pageable pageable, ProfitAgreementTemplatePaginationDto dto) {
    Page<ProfitAgreementTemplate> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    return profitAgreementTemplateMapper.findByConditions(page, dto);
  }

  /**
   * 条件查询
   * @param dto
   * @return
   */
  public List<ProfitAgreementTemplate> findByConditions(ProfitAgreementTemplatePaginationDto dto) {
    return profitAgreementTemplateMapper.findByConditions(dto);
  }

  /**
   * 批量启用-禁用
   *
   * @param enable 启用或禁用
   * @param ids    操作主键集合
   */
  public void updateEnableStatusByIds(EnableStatusEnum enable, List<String> ids) {
    if (CollectionUtils.isEmpty(ids)) {
      return;
    }
    UpdateWrapper<ProfitAgreementTemplate> updateWrapper = new UpdateWrapper<>();
    updateWrapper.set("enable_status", enable.getCode());
    updateWrapper.in("id", ids);
    this.update(updateWrapper);
  }

  /**
   * 根据模板编码查询对象
   *
   * @param templateCode
   * @return
   */
  public ProfitAgreementTemplate findByTemplateCode(String templateCode) {
    QueryWrapper<ProfitAgreementTemplate> queryWrapper = new QueryWrapper<>();
    queryWrapper.eq("template_code", templateCode);
    return this.getOne(queryWrapper);
  }

  /**
   * 根据id集合查询数据集合
   * @param ids
   * @return
   */
  public List<ProfitAgreementTemplate> findByIds(List<String> ids) {
    QueryWrapper<ProfitAgreementTemplate> queryWrapper = new QueryWrapper<>();
    queryWrapper.in("id", ids);
    return this.list(queryWrapper);
  }

  /**
   * 编辑（更改）协议模板
   * @param profitAgreementTemplate 协议模板实体
   */
  public void update(ProfitAgreementTemplate profitAgreementTemplate){
    this.profitAgreementTemplateMapper.updateById(profitAgreementTemplate);
  }

  /**
   * 根据协议编码获取分利协议信息
   *
   * @param templateCodeSet
   * @return
   */
  public List<ProfitAgreementTemplate> findByTemplateCodes(Set<String> templateCodeSet) {
    return this.lambdaQuery().in(ProfitAgreementTemplate::getTemplateCode,templateCodeSet).list();
  }
}
