package com.biz.crm.cps.business.agreement.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.model.LoginUserDetailsForCPS;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.cps.business.agreement.local.entity.Agreement;
import com.biz.crm.cps.business.agreement.local.entity.AgreementSign;
import com.biz.crm.cps.business.agreement.local.entity.ProfitAgreementTemplate;
import com.biz.crm.cps.business.agreement.local.repository.AgreementRepository;
import com.biz.crm.cps.business.agreement.local.repository.AgreementSignRepository;
import com.biz.crm.cps.business.agreement.local.repository.ProfitAgreementTemplateRepository;
import com.biz.crm.cps.business.agreement.local.service.AgreementService;
import com.biz.crm.cps.business.agreement.sdk.common.constant.AgreementCodeConstant;
import com.biz.crm.cps.business.agreement.sdk.common.enums.AgreementStatusEnum;
import com.biz.crm.cps.business.agreement.sdk.common.enums.AutoSignEnum;
import com.biz.crm.cps.business.agreement.sdk.common.enums.SignStatusEnum;
import com.biz.crm.cps.business.agreement.sdk.common.enums.SignatoryEnum;
import com.biz.crm.cps.business.agreement.sdk.dto.AgreementDto;
import com.biz.crm.cps.business.agreement.sdk.dto.AgreementSignDto;
import com.biz.crm.cps.business.agreement.sdk.dto.AgreementSignEventDto;
import com.biz.crm.cps.business.agreement.sdk.dto.LoginUserAgreementDto;
import com.biz.crm.cps.business.agreement.sdk.service.observer.AgreementSignObserver;
import com.biz.crm.cps.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.cps.business.participator.sdk.service.DealerVoService;
import com.biz.crm.cps.business.participator.sdk.service.TerminalVoService;
import com.biz.crm.cps.business.participator.sdk.vo.DealerVo;
import com.biz.crm.cps.business.participator.sdk.vo.TerminalSupplyVo;
import com.biz.crm.cps.business.participator.sdk.vo.TerminalVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.TreeSet;
import java.util.stream.Collectors;

/**
 * 分利协议相关业务处理实现
 *
 * @author songjingen
 */
@Service
public class AgreementServiceImpl implements AgreementService {

  @Autowired
  private AgreementRepository agreementRepository;
  @Autowired
  private AgreementSignRepository agreementSignRepository;
  @Autowired(required = false)
  private LoginUserService loginUserService;
  @Autowired(required = false)
  private GenerateCodeService generateCodeService;
  @Autowired
  private TerminalVoService terminalVoService;
  @Autowired
  private DealerVoService dealerVoService;
  @Autowired
  private ProfitAgreementTemplateRepository profitAgreementTemplateRepository;
  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Autowired(required = false)
  private List<AgreementSignObserver> agreementSignObservers;

  /**
   * 根据id签署（单个或者批量） 品牌商
   *
   * @param ids
   */
  @Transactional
  @Override
  public void enable(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "请传入要操作的数据");
    //查询要签署的协议信息
    List<Agreement> agreements = this.agreementRepository.findByIds(ids);
    Validate.isTrue(!CollectionUtils.isEmpty(agreements), "未查询到要签署的协议");
    List<Agreement> collect = agreements.stream().filter(agreement -> SignStatusEnum.SIGNING.getCode().equals(agreement.getSignStatus())).collect(Collectors.toList());
    Validate.isTrue(!CollectionUtils.isEmpty(collect), "未查询到要签署的协议");
    //保存签署记录
    ArrayList<AgreementSign> agreementSigns = new ArrayList<>();
    Date date = new Date();
    collect.forEach(agreement -> {
      agreement.setSignStatus(SignStatusEnum.SIGNED.getCode());
      AgreementSign agreementSign = new AgreementSign();
      agreementSign.setId(null);
      agreementSign.setSignatory(SignatoryEnum.BRAND.getDictCode());
      agreementSign.setSignatoryCode(TenantUtils.getTenantCode());
      agreementSign.setSignatoryName(TenantUtils.getTenantCode());
      agreementSign.setSignDate(date);
      agreementSign.setSignAccount(TenantUtils.getTenantCode());
      agreementSign.setTenantCode(TenantUtils.getTenantCode());
      agreementSign.setAgreementCode(agreement.getAgreementCode());
      agreementSigns.add(agreementSign);
    });
    //更改签署状态
    this.agreementRepository.saveOrUpdateBatch(collect);
    this.agreementSignRepository.saveOrUpdateBatch(agreementSigns);
    //通知上层模块
    ArrayList<Agreement> newAgreement = collect.stream().collect(Collectors.collectingAndThen(Collectors.toCollection(() -> new TreeSet<>(
        Comparator.comparing(p -> p.getTemplateCode() + ";" + p.getTerminalCode()))), ArrayList::new));
    newAgreement.stream().forEach(agreement -> {
      this.notifyEvent(agreement);
    });
  }

  /**
   * 根据id终止（单个或者批量）
   *
   * @param ids
   */
  @Transactional
  @Override
  public void disable(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "请传入要操作的数据");
    //查询要终止的协议信息
    List<Agreement> agreements = this.agreementRepository.findByIds(ids);
    Validate.isTrue(!CollectionUtils.isEmpty(agreements), "未查询到要终止的协议");
    List<String> newIds = agreements.stream().filter(agreement -> SignStatusEnum.SIGNED.getCode().equals(agreement.getSignStatus())).map(Agreement::getId).collect(Collectors.toList());
    Validate.isTrue(!CollectionUtils.isEmpty(newIds), "未查询到要终止的协议");
    agreementRepository.updateAgreementStatusByIds(AgreementStatusEnum.TERMINATED, newIds);
  }

  /**
   * 根据templateCode终止（单个或者批量）
   *
   * @param templateCodes
   */
  @Transactional
  @Override
  public void disableByTemplateCodes(List<String> templateCodes) {
    Validate.isTrue(!CollectionUtils.isEmpty(templateCodes), "请传入要操作的数据");
    //查询要终止的协议信息
    List<Agreement> agreementList = this.findByTemplateCodes(templateCodes);
    Validate.isTrue(!CollectionUtils.isEmpty(agreementList), "未查询到要终止的协议");
    agreementList.stream().forEach(agreement -> agreement.setAgreementStatus(AgreementStatusEnum.TERMINATED.getCode()));
    this.agreementRepository.saveOrUpdateBatch(agreementList);
  }

  /**
   * 协议新增（终端协议签署）
   *
   * @param dto
   * @return
   */
  @Transactional
  @Override
  public void create(AgreementSignDto dto) {
    //查询当前登录对象
    LoginUserDetailsForCPS loginUser = loginUserService.getLoginDetails(LoginUserDetailsForCPS.class);
    Validate.notNull(loginUser, "未查询到当前登录信息");
    this.createValidation(dto);
    List<Agreement> agreements = this.createForm(dto, loginUser);
    agreementRepository.saveOrUpdateBatch(agreements);
    //保存终端签署记录
    List<AgreementSign> agreementSigns = this.createAgreementSignForm(dto, agreements);
    agreementSignRepository.saveBatch(agreementSigns);
    //通知上层模块
    ArrayList<Agreement> newAgreement = agreements.stream().collect(Collectors.collectingAndThen(Collectors.toCollection(() -> new TreeSet<>(
        Comparator.comparing(p -> p.getTemplateCode() + ";" + p.getTerminalCode()))), ArrayList::new));
    for (Agreement agreement : newAgreement) {
      if (SignStatusEnum.SIGNED.getCode().equals(agreement.getSignStatus())) {
        this.notifyEvent(agreement);
      }
    }
    //更新模板的签署数量
    ProfitAgreementTemplate profitAgreementTemplate = this.profitAgreementTemplateRepository.findByTemplateCode(dto.getTemplateCode());
    if (ObjectUtils.allNotNull(profitAgreementTemplate)) {
      Integer generatedAgreementNumber = profitAgreementTemplate.getGeneratedAgreementNumber();
      generatedAgreementNumber = Objects.isNull(generatedAgreementNumber) ? 0:generatedAgreementNumber;
      profitAgreementTemplate.setGeneratedAgreementNumber(generatedAgreementNumber + agreements.size());
      this.profitAgreementTemplateRepository.updateById(profitAgreementTemplate);
    }
  }

  /**
   * 签署成功后，通知上层模块
   */
  private void notifyEvent(Agreement agreement) {
    if (CollectionUtils.isEmpty(agreementSignObservers)) {
      return;
    }
    if (Objects.isNull(agreement)) {
      return;
    }
    AgreementSignEventDto agreementSignEventDto = this.nebulaToolkitService.copyObjectByWhiteList(agreement, AgreementSignEventDto.class, null, null);
    for (AgreementSignObserver agreementSignObserver : agreementSignObservers) {
      agreementSignObserver.onSuccess(agreementSignEventDto);
    }
  }

  /**
   * 新增时配置协议参数信息
   *
   * @param dto
   * @param loginUser
   * @return
   */
  private List<Agreement> createForm(AgreementSignDto dto, LoginUserDetailsForCPS loginUser) {
    /**
     * 0、校验是否已经签署协议
     * 1、验证模板信息
     * 2、查询终端信息
     * 3、查询当前终端关联的供货关系集合
     * 4、遍历经销商信息集合，组装协议信息
     */
    // 0、======
    this.findByTemplateCodeAndTerminalCode(dto.getTemplateCode(), loginUser.getConsumerCode());
    //1、======
    ProfitAgreementTemplate template = profitAgreementTemplateRepository.findByTemplateCode(dto.getTemplateCode());
    Validate.notNull(template, "未查询到模板信息");
    Validate.isTrue(EnableStatusEnum.ENABLE.getCode().equals(template.getEnableStatus()), "模板已禁用，无法签署！");
    //1.1、验证是否在签署时间范围
    Date date = new Date();
    Validate.isTrue(date.after(template.getSignStartTime()) && date.before(template.getSignEndTime()), "不在规定的签署时间范围内！");
    //2、======
    TerminalVo terminalVo = this.findTerminalByTerminalCode(loginUser.getConsumerCode());
    //3、======
    List<TerminalSupplyVo> terminalSupplies = terminalVo.getTerminalSupplies();
    Validate.isTrue(!CollectionUtils.isEmpty(terminalSupplies), "未查询到当前用户的供货关系！");
    List<String> customerCodes = terminalSupplies.stream().map(TerminalSupplyVo::getCustomerCode).collect(Collectors.toList());
    List<DealerVo> dealerVos = this.dealerVoService.findByCustomerCodes(customerCodes);
    Validate.isTrue(!CollectionUtils.isEmpty(dealerVos), "未查询到当前用户的供货关系对应的经销商信息！");
    //4、======
    ArrayList<Agreement> agreements = new ArrayList<>();
    for (DealerVo dealerVo : dealerVos) {
      Agreement agreement = this.buildAgreement(template, terminalVo, dealerVo, date);
      agreements.add(agreement);
    }
    return agreements;
  }

  /**
   * 组装协议信息
   *
   * @param template   模板
   * @param terminalVo 终端信息
   * @param dealerVo   经销商信息
   * @param date       时间
   * @return 协议信息
   */
  private Agreement buildAgreement(ProfitAgreementTemplate template, TerminalVo terminalVo, DealerVo dealerVo, Date date) {
    Agreement agreement = new Agreement();
    agreement.setAgreementCode(generateCodeService.generateCode(AgreementCodeConstant.AGREEMENT_CODE, 1).get(0));
    agreement.setAgreementName(StringUtils.joinWith("-", template.getTemplateName(), terminalVo.getTerminalName()));
    agreement.setTemplateCode(template.getTemplateCode());
    agreement.setBelongTemplate(template.getTemplateName());
    agreement.setTerminalCode(terminalVo.getTerminalCode());
    agreement.setRelationTerminal(terminalVo.getTerminalName());
    agreement.setCustomerCode(dealerVo.getCustomerCode());
    agreement.setRelationDealer(dealerVo.getCustomerName());
    agreement.setTerminalSignStatus(SignStatusEnum.SIGNED.getCode());
    agreement.setDealerSignStatus(SignStatusEnum.SIGNED.getCode());
    agreement.setTenantCode(TenantUtils.getTenantCode());
    agreement.setCreateTime(date);
    agreement.setCreateAccount(loginUserService.getLoginAccountName());
    agreement.setModifyTime(date);
    agreement.setModifyAccount(loginUserService.getLoginAccountName());
    //根据模板的自动签署状态设置品牌商的签署状态
    if (AutoSignEnum.YES.getCode().equals(template.getAutoSign())) {
      agreement.setSignStatus(SignStatusEnum.SIGNED.getCode());
    } else {
      agreement.setSignStatus(SignStatusEnum.SIGNING.getCode());
    }
    //根据模板的协议时间设置协议状态
    if (date.before(template.getEffectiveStartTime())) {
      agreement.setAgreementStatus(AgreementStatusEnum.NOT_STARTED.getCode());
    } else if (date.after(template.getEffectiveStartTime()) && date.before(template.getEffectiveEndTime())) {
      agreement.setAgreementStatus(AgreementStatusEnum.EXECUTING.getCode());
    } else {
      agreement.setAgreementStatus(AgreementStatusEnum.COMPLETED.getCode());
    }
    return agreement;
  }

  /**
   * 查询协议信息的详情
   *
   * @param id
   * @return
   */
  @Override
  public Agreement findById(String id) {
    Validate.notBlank(id, "进行此操作时，入参不能为空！");
    return agreementRepository.getById(id);
  }

  /**
   * 查询分利协议基本信息详情包含签署情况
   *
   * @param id
   * @return
   */
  @Override
  public Agreement findDetailsById(String id) {
    Validate.notBlank(id, "进行此操作时，入参不能为空！");
    Agreement agreement = agreementRepository.findDetailsById(id);
    return agreement;
  }

  /**
   * 分页查询数据
   *
   * @param pageable
   * @param dto
   * @return
   */
  @Override
  public Page<Agreement> findByConditions(Pageable pageable, AgreementDto dto) {
    ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    if (Objects.isNull(dto)) {
      dto = new AgreementDto();
    }
    return this.agreementRepository.findByConditions(pageable, dto);
  }

  /**
   * 查询协议协议生效开始时间小于等于当前时间，且协议状态未开始；
   * 查询协议协议生效结束时间小于当前时间，且协议状态进行中
   *
   * @param date
   * @return
   */
  @Override
  public List<Agreement> findByCurrentTime(Date date) {
    if (Objects.isNull(date)) {
      date = new Date();
    }
    return this.agreementRepository.findByCurrentTime(date);
  }

  /**
   * 批量新增
   *
   * @param list
   */
  @Transactional
  @Override
  public void createOrUpdateBatch(List<Agreement> list) {
    Validate.isTrue(!CollectionUtils.isEmpty(list), "批量操作时，集合不能为空！");
    this.agreementRepository.saveOrUpdateBatch(list);
  }

  /**
   * 根据templateCodes集合查询数据集合
   * @param templateCodes
   * @return
   */
  @Override
  public List<Agreement> findByTemplateCodes(List<String> templateCodes) {
    if (CollectionUtils.isEmpty(templateCodes)) {
      return null;
    }
    return this.agreementRepository.findByTemplateCodes(templateCodes);
  }

  /**
   * 根据id撤销终止（单个或者批量）
   * @param ids
   */
  @Override
  @Transactional
  public void revoke(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "进行此操作时，入参不能为空");
    //根据ids查询需要撤销终止的数据
    AgreementDto agreementDto = new AgreementDto();
    agreementDto.setIds(ids);
    List<Agreement> agreements = this.agreementRepository.findByConditions(agreementDto);
    Validate.isTrue(!CollectionUtils.isEmpty(agreements), "未查询到需要撤销终止的协议！");
    List<Agreement> newAgreements = agreements.stream().filter(agreement -> AgreementStatusEnum.TERMINATED.getCode().equals(agreement.getAgreementStatus()) && !Objects.isNull(agreement.getProfitAgreementTemplate()) && EnableStatusEnum.ENABLE.getCode().equals(agreement.getProfitAgreementTemplate().getEnableStatus())).collect(Collectors.toList());
    Validate.isTrue(!CollectionUtils.isEmpty(newAgreements), "未查询到需要撤销终止的协议或者模板已禁用！");
    Date date = new Date();
    newAgreements.stream().forEach(agreement -> {
      ProfitAgreementTemplate template = agreement.getProfitAgreementTemplate();
      Date startTime = template.getEffectiveStartTime();
      Date endTime = template.getEffectiveEndTime();
      if (date.before(startTime)) {
        agreement.setAgreementStatus(AgreementStatusEnum.NOT_STARTED.getCode());
      } else if (date.after(startTime) && date.before(endTime)) {
        agreement.setAgreementStatus(AgreementStatusEnum.EXECUTING.getCode());
      } else {
        agreement.setAgreementStatus(AgreementStatusEnum.COMPLETED.getCode());
      }
    });
    this.agreementRepository.saveOrUpdateBatch(newAgreements);
  }

  /**
   * 新增操作时，先进行参数校验
   *
   * @param dto
   */
  private void createValidation(AgreementSignDto dto) {
    //验证必填项，以及字段长度
    Validate.notNull(dto, "签署操作时，签署对象不能为空！");
    Validate.notBlank(dto.getTemplateCode(), "签署操作时，模板编码不能为空！");
    Validate.notBlank(dto.getSignaturePath(), "签署操作时，签名照片文件夹不能为空！");
    Validate.notBlank(dto.getSignatureFilename(), "签署操作时，签名照片文件名不能为空！");
  }

  /**
   * 组装签署记录数据
   *
   * @param dto
   * @param agreements
   * @return
   */
  private List<AgreementSign> createAgreementSignForm(AgreementSignDto dto, List<Agreement> agreements) {
    List<AgreementSign> signList = new ArrayList<>();
    for (Agreement agreement : agreements) {
      //经销商签署
      AgreementSign dealerAgreementSign = new AgreementSign();
      dealerAgreementSign.setAgreementCode(agreement.getAgreementCode());
      dealerAgreementSign.setSignatory(SignatoryEnum.DEALER.getDictCode());
      dealerAgreementSign.setSignatoryCode(agreement.getCustomerCode());
      dealerAgreementSign.setSignatoryName(agreement.getRelationDealer());
      dealerAgreementSign.setSignDate(new Date());
      dealerAgreementSign.setSignAccount(agreement.getRelationDealer());
      dealerAgreementSign.setTenantCode(TenantUtils.getTenantCode());
      //终端签署
      AgreementSign terminalAgreementSign = this.nebulaToolkitService.copyObjectByWhiteList(dealerAgreementSign, AgreementSign.class, null, null);
      terminalAgreementSign.setSignatureFilename(dto.getSignatureFilename());
      terminalAgreementSign.setSignaturePath(dto.getSignaturePath());
      terminalAgreementSign.setSignatory(SignatoryEnum.TERMINAL.getDictCode());
      terminalAgreementSign.setSignatoryCode(agreement.getTerminalCode());
      terminalAgreementSign.setSignatoryName(agreement.getRelationTerminal());
      terminalAgreementSign.setSignAccount(agreement.getRelationTerminal());
      //品牌商
      AgreementSign agreementSign = this.nebulaToolkitService.copyObjectByWhiteList(dealerAgreementSign, AgreementSign.class, null, null);
      agreementSign.setSignatory(SignatoryEnum.BRAND.getDictCode());
      //todo 暂时没有通过租户获取品牌商名称的接口
      agreementSign.setSignatoryCode(TenantUtils.getTenantCode());
      agreementSign.setSignatoryName(TenantUtils.getTenantCode());
      agreementSign.setSignAccount(TenantUtils.getTenantCode());
      signList.add(dealerAgreementSign);
      signList.add(terminalAgreementSign);
      if (SignStatusEnum.SIGNED.getCode().equals(agreement.getSignStatus())) {
        signList.add(agreementSign);
      }
    }
    return signList;
  }

  /**
   * 验证是否已经签署模板生成协议实例
   * @param templateCode
   * @param terminalCode
   */
  private void findByTemplateCodeAndTerminalCode(String templateCode, String terminalCode){
    LoginUserAgreementDto loginUserAgreementDto = new LoginUserAgreementDto();
    ArrayList<String> templateCodes = new ArrayList<>();
    templateCodes.add(templateCode);
    loginUserAgreementDto.setTemplateCodes(templateCodes);
    loginUserAgreementDto.setTerminalCode(terminalCode);
    List<Agreement> byConditions = this.agreementRepository.findByConditions(loginUserAgreementDto);
    Validate.isTrue(CollectionUtils.isEmpty(byConditions), "已签署当前协议！");
  }

  /**
   * 查询终端信息
   * @param terminalCode
   * @return
   */
  private TerminalVo findTerminalByTerminalCode(String terminalCode) {
    ArrayList<String> list = new ArrayList<>();
    list.add(terminalCode);
    List<TerminalVo> terminalVoList = terminalVoService.findByTerminalCodes(list);
    Validate.notNull(terminalVoList, "当前终端信息不存在，请检查！");
    TerminalVo terminalVo = terminalVoList.get(0);
    return terminalVo;
  }
}
