package com.biz.crm.cps.business.agreement.local.service.internal;

import com.alibaba.fastjson.JSONObject;
import com.biz.crm.business.common.sdk.model.LoginUserDetailsForCPS;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.cps.business.agreement.local.entity.ProfitAgreementTemplate;
import com.biz.crm.cps.business.agreement.local.entity.TemplateChannelRelationship;
import com.biz.crm.cps.business.agreement.local.entity.TemplateOrgRelationship;
import com.biz.crm.cps.business.agreement.local.entity.TemplatePolicyRelationship;
import com.biz.crm.cps.business.agreement.local.entity.TemplateTag;
import com.biz.crm.cps.business.agreement.local.entity.TemplateTerminalRelationship;
import com.biz.crm.cps.business.agreement.local.repository.ProfitAgreementTemplateRepository;
import com.biz.crm.cps.business.agreement.local.repository.TemplateChannelRelationshipRepository;
import com.biz.crm.cps.business.agreement.local.repository.TemplateOrgRelationshipRepository;
import com.biz.crm.cps.business.agreement.local.repository.TemplatePolicyRelationshipRepository;
import com.biz.crm.cps.business.agreement.local.repository.TemplateTagRepository;
import com.biz.crm.cps.business.agreement.local.repository.TemplateTerminalRelationshipRepository;
import com.biz.crm.cps.business.agreement.local.service.AgreementService;
import com.biz.crm.cps.business.agreement.local.service.ProfitAgreementTemplateService;
import com.biz.crm.cps.business.agreement.sdk.common.constant.TemplateCodeConstant;
import com.biz.crm.cps.business.agreement.sdk.service.observer.AgreementPolicyMountRegister;
import com.biz.crm.cps.business.agreement.sdk.service.observer.AgreementPolicyServiceObserver;
import com.biz.crm.cps.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.cps.business.common.sdk.enums.EnableStatusEnum;
import com.bizunited.nebula.common.util.JsonUtils;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * @author jerry7
 * @date 2021-07-22
 * 分离协议模板实现类
 */
@Service
@ConditionalOnMissingBean(name = "ProfitAgreementTemplateServiceExpandImpl")
public class AgreementTemplateServiceImpl implements ProfitAgreementTemplateService {
  @Autowired
  private ProfitAgreementTemplateRepository profitAgreementTemplateRepository;
  @Autowired
  private TemplateOrgRelationshipRepository templateOrgRelationshipRepository;
  @Autowired
  private TemplatePolicyRelationshipRepository templatePolicyRelationshipRepository;
  @Autowired
  private TemplateChannelRelationshipRepository templateChannelRelationshipRepository;
  @Autowired
  private TemplateTagRepository templateTagRepository;
  @Autowired(required = false)
  private GenerateCodeService generateCodeService;
  @Autowired(required = false)
  private LoginUserService loginUserService;

  @Autowired(required = false)
  private List<AgreementPolicyServiceObserver> agreementPolicyServiceObservers;

  @Autowired(required = false)
  private List<AgreementPolicyMountRegister> agreementPolicyMountRegisters;

  @Autowired
  private AgreementService agreementService;
  @Autowired
  private TemplateTerminalRelationshipRepository templateTerminalRelationshipRepository;

  @Transactional
  @Override
  public ProfitAgreementTemplate create(ProfitAgreementTemplate profitAgreementTemplate) {
    ProfitAgreementTemplate templateCurrent = this.createForm(profitAgreementTemplate);
    this.profitAgreementTemplateRepository.save(templateCurrent);
    //保存关联组织
    if (!CollectionUtils.isEmpty(templateCurrent.getTemplateOrgRelationships())) {
      templateCurrent.getTemplateOrgRelationships().forEach(templateOrgRelationship -> templateOrgRelationship.setTemplateCode(templateCurrent.getTemplateCode()));
      this.templateOrgRelationshipRepository.saveBatch(templateCurrent.getTemplateOrgRelationships());
    }
    //保存关联政策
    if (!CollectionUtils.isEmpty(templateCurrent.getTemplatePolicyRelationships())) {
      templateCurrent.getTemplatePolicyRelationships().forEach(templatePolicyRelationship -> templatePolicyRelationship.setTemplateCode(templateCurrent.getTemplateCode()));
      this.templatePolicyRelationshipRepository.saveBatch(templateCurrent.getTemplatePolicyRelationships());
    }
    //保存关联标签
    if (!CollectionUtils.isEmpty(templateCurrent.getTemplateTags())) {
      templateCurrent.getTemplateTags().forEach(templateTag -> templateTag.setTemplateCode(templateCurrent.getTemplateCode()));
      this.templateTagRepository.saveBatch(templateCurrent.getTemplateTags());
    }
    //保存关联渠道
    if (!CollectionUtils.isEmpty(templateCurrent.getTemplateChannelRelationships())) {
      templateCurrent.getTemplateChannelRelationships().forEach(channelRelationship -> channelRelationship.setTemplateCode(templateCurrent.getTemplateCode()));
      this.templateChannelRelationshipRepository.saveBatch(templateCurrent.getTemplateChannelRelationships());
    }
    //保存指定终端关联
    if(!CollectionUtils.isEmpty(templateCurrent.getTemplateTerminalRelationships())){
      templateCurrent.getTemplateTerminalRelationships().forEach(terminalRelationship -> terminalRelationship.setTemplateCode(templateCurrent.getTemplateCode()));
      templateTerminalRelationshipRepository.saveBatch(templateCurrent.getTemplateTerminalRelationships());
    }
    return profitAgreementTemplate;
  }

  @Transactional
  @Override
  public ProfitAgreementTemplate createForm(ProfitAgreementTemplate profitAgreementTemplate) {
    Validate.notNull(profitAgreementTemplate, "进行当前操作时，信息对象必须传入!!");
    this.createValidation(profitAgreementTemplate);
    Date now = new Date();
    profitAgreementTemplate.setCreateTime(now);
    profitAgreementTemplate.setModifyTime(now);
    if (Objects.nonNull(generateCodeService)) {
      profitAgreementTemplate.setTemplateCode(generateCodeService.generateCode(TemplateCodeConstant.AGREEMENT_TEMPLATE_CODE, 1).get(0));
    }
    profitAgreementTemplate.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    profitAgreementTemplate.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    profitAgreementTemplate.setTenantCode(TenantUtils.getTenantCode());
    profitAgreementTemplate.setCreateAccount(loginUserService.getLoginAccountName());
    profitAgreementTemplate.setModifyAccount(loginUserService.getLoginAccountName());
    return profitAgreementTemplate;
  }

  @Override
  public ProfitAgreementTemplate findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.profitAgreementTemplateRepository.getById(id);
  }

  @Override
  public ProfitAgreementTemplate findDetailsById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.profitAgreementTemplateRepository.findDetailsById(id);
  }

  @Transactional
  @Override
  public void enableBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "请选中要操作的数据");
    this.profitAgreementTemplateRepository.updateEnableStatusByIds(EnableStatusEnum.ENABLE, ids);
  }

  @Transactional
  @Override
  public void disableBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "请选中要操作的数据");
    this.profitAgreementTemplateRepository.updateEnableStatusByIds(EnableStatusEnum.DISABLE, ids);
  }

  /**
   * 创建模板信息，以及其子表，并通知到政策观察者
   * @param jsonObject
   * @return
   */
  @Override
  @Transactional
  public ProfitAgreementTemplate create(JSONObject jsonObject) {
    //解析模板信息
    ProfitAgreementTemplate profitAgreementTemplate = this.getProfitAgreementTemplate(jsonObject);
    //创建模板信息
    this.create(profitAgreementTemplate);
    jsonObject.put("templateCode", profitAgreementTemplate.getTemplateCode());
    // 保存其他政策的信息
    for (AgreementPolicyServiceObserver agreementPolicyServiceObserver : agreementPolicyServiceObservers) {
      agreementPolicyServiceObserver.onCreate(jsonObject);
    }
    return profitAgreementTemplate;
  }

  /**
   * 解析出模板信息
   * @param jsonObject
   * @return
   */
  private ProfitAgreementTemplate getProfitAgreementTemplate(JSONObject jsonObject) {
    Validate.notNull(jsonObject, "未获取到请求信息");
    // 获取协议模板的配置  TODO 字符串不能直接写这里
    JSONObject templateJson = jsonObject.getJSONObject("profitAgreementTemplate");
    Validate.notNull(jsonObject, "未获取到协议模板信息");
    //校验模板信息是否携带具体政策数据
    boolean hasPolicyData = false;
    for (AgreementPolicyMountRegister mountRegister : agreementPolicyMountRegisters) {
      //循环政策绑定服务，获取对应政策的数据，获取到任一数据则将携带政策数据置为true，并退出循环
      Object policyData = jsonObject.get(mountRegister.getKey());
      if (Objects.nonNull(policyData)) {
        hasPolicyData = true;
        break;
      }
    }
    Validate.isTrue(hasPolicyData, "创建模板时需要指定政策内容！");
    ProfitAgreementTemplate profitAgreementTemplate = JsonUtils.json2Obj(templateJson.toJSONString(), ProfitAgreementTemplate.class);
    return profitAgreementTemplate;
  }

  /**
   * 修改模板信息，以及其关联信息
   * @param jsonObject 协议模板实体类
   */
  @Transactional
  @Override
  public ProfitAgreementTemplate update(JSONObject jsonObject) {
    //解析模板信息
    ProfitAgreementTemplate profitAgreementTemplate = this.getProfitAgreementTemplate(jsonObject);
    //修改模板信息
    ProfitAgreementTemplate template = this.findById(profitAgreementTemplate.getId());
    Validate.notNull(template, "修改操作时，未查询到模板信息！");
    Date date = new Date();
    boolean before = date.before(template.getSignStartTime());
    profitAgreementTemplate = this.updateDetails(profitAgreementTemplate, template, before);
    jsonObject.put("templateCode", profitAgreementTemplate.getTemplateCode());
    jsonObject.put("before", before);
    // 保存其他政策的信息
    for (AgreementPolicyServiceObserver agreementPolicyServiceObserver : agreementPolicyServiceObservers) {
      agreementPolicyServiceObserver.onChange(jsonObject);
    }
    return profitAgreementTemplate;
  }

  @Override
  public List<ProfitAgreementTemplate> findByTemplateCodes(Set<String> templateCodeSet) {
    if(CollectionUtils.isEmpty(templateCodeSet)) {
      return Lists.newLinkedList();
    }
    return this.profitAgreementTemplateRepository.findByTemplateCodes(templateCodeSet);
  }

  /**
   * 修改模板详情，以及修改其子表信息
   * @param profitAgreementTemplate
   * @return
   */
  private ProfitAgreementTemplate updateDetails(ProfitAgreementTemplate profitAgreementTemplate, ProfitAgreementTemplate template, Boolean before) {
    //修改模板信息
    if (before) {
      BeanUtils.copyProperties(profitAgreementTemplate, template, "id", "templateCode", "tenantCode", "delFlag", "enableStatus", "createAccount", "createTime");
    } else {
      template.setPermissionType(profitAgreementTemplate.getPermissionType());
      template.setTemplateName(profitAgreementTemplate.getTemplateName());
      template.setAgreementText(profitAgreementTemplate.getAgreementText());
    }
    template.setModifyAccount(loginUserService.getLoginDetails(LoginUserDetailsForCPS.class).getConsumerName());
    template.setModifyTime(new Date());
    template.setTemplatePolicyRelationships(profitAgreementTemplate.getTemplatePolicyRelationships());
    this.updateValidation(template);
    this.profitAgreementTemplateRepository.updateById(template);

    String templateCode = template.getTemplateCode();
    //修改关联组织
    this.templateOrgRelationshipRepository.deleteByTemplateCode(templateCode);
    Set<TemplateOrgRelationship> templateOrgRelationships = profitAgreementTemplate.getTemplateOrgRelationships();
    if (!CollectionUtils.isEmpty(templateOrgRelationships)) {
      templateOrgRelationships.forEach(templateOrgRelationship -> templateOrgRelationship.setTemplateCode(templateCode));
      this.templateOrgRelationshipRepository.saveBatch(templateOrgRelationships);
      template.setTemplateOrgRelationships(templateOrgRelationships);
    }
    //修改关联政策，当前时间在签署开始时间之前可进行修改关联表
    Set<TemplatePolicyRelationship> templatePolicyRelationships = profitAgreementTemplate.getTemplatePolicyRelationships();
    if (before) {
      this.templatePolicyRelationshipRepository.deleteByTemplateCode(templateCode);
      if (!CollectionUtils.isEmpty(templatePolicyRelationships)) {
        templatePolicyRelationships.forEach(templatePolicyRelationship -> {
          templatePolicyRelationship.setTemplateCode(templateCode);
        });
        this.templatePolicyRelationshipRepository.saveBatch(templatePolicyRelationships);
        template.setTemplatePolicyRelationships(templatePolicyRelationships);
      }
    } else {
      List<TemplatePolicyRelationship> oldList =  this.templatePolicyRelationshipRepository.findByTemplateCode(templateCode);
      Set<String> oldSet = oldList.stream().map(TemplatePolicyRelationship::getPolicyCode).collect(Collectors.toSet());
      Set<String> newSet = templatePolicyRelationships.stream().map(TemplatePolicyRelationship::getPolicyCode).collect(Collectors.toSet());
      newSet.retainAll(oldSet);
      Validate.isTrue(newSet.size() == oldSet.size(), "修改模板时，政策不能改变！");
    }
    //修改关联标签
    this.templateTagRepository.deleteByTemplateCode(templateCode);
    Set<TemplateTag> templateTags = profitAgreementTemplate.getTemplateTags();
    if (!CollectionUtils.isEmpty(templateTags)) {
      templateTags.forEach(templateTag -> templateTag.setTemplateCode(templateCode));
      this.templateTagRepository.saveBatch(templateTags);
      template.setTemplateTags(templateTags);
    }
    //修改关联渠道
    this.templateChannelRelationshipRepository.deleteByTemplateCode(templateCode);
    Set<TemplateChannelRelationship> templateChannelRelationships = profitAgreementTemplate.getTemplateChannelRelationships();
    if (!CollectionUtils.isEmpty(templateChannelRelationships)) {
      templateChannelRelationships.forEach(channelRelationship -> channelRelationship.setTemplateCode(templateCode));
      this.templateChannelRelationshipRepository.saveBatch(templateChannelRelationships);
      template.setTemplateChannelRelationships(templateChannelRelationships);
    }
    //保存指定终端关联
    this.templateTerminalRelationshipRepository.deleteByTemplateCode(templateCode);
    Set<TemplateTerminalRelationship> templateTerminalRelationships = profitAgreementTemplate.getTemplateTerminalRelationships();
    if(!CollectionUtils.isEmpty(templateTerminalRelationships)){
      templateTerminalRelationships.forEach(terminalRelationship -> terminalRelationship.setTemplateCode(templateCode));
      templateTerminalRelationshipRepository.saveBatch(templateTerminalRelationships);
      template.setTemplateTerminalRelationships(templateTerminalRelationships);
    }
    return template;
  }

  /**
   * 在创建一个新的ProfitAgreementTemplate模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(ProfitAgreementTemplate profitAgreementTemplate) {
    Validate.isTrue(StringUtils.isBlank(profitAgreementTemplate.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    profitAgreementTemplate.setId(null);
    Validate.notBlank(profitAgreementTemplate.getTemplateName(), "添加信息时，分利模板名称不能为空！");
    Validate.notBlank(profitAgreementTemplate.getAgreementText(), "添加信息时，协议文本不能为空！");
    Validate.notNull(profitAgreementTemplate.getEffectiveStartTime(), "添加信息时，分利模板生效开始时间不能为空！");
    Validate.notNull(profitAgreementTemplate.getEffectiveEndTime(), "添加信息时，分利模板生效结束时间不能为空！");
    Validate.notNull(profitAgreementTemplate.getSignStartTime(), "添加信息时，分利模板签署开始时间不能为空！");
    Validate.notNull(profitAgreementTemplate.getSignEndTime(), "添加信息时，分利模板签署结束时间不能为空！");
    Validate.notNull(profitAgreementTemplate.getAutoSign(), "添加信息时，分利模板是否自动签署不能为空！");
    Validate.notEmpty(profitAgreementTemplate.getTemplatePolicyRelationships(), "添加信息时，分利模板包含政策不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况）
    Validate.isTrue(profitAgreementTemplate.getTemplateName() == null || profitAgreementTemplate.getTemplateName().length() < 64, "分利终端编码,在进行添加时填入值超过了限定长度(64)，请检查!");
    //验证时间，模板生效和签署时间的结束时间需大于开始时间，生效结束时间需大于签署结束时间
    Validate.isTrue(profitAgreementTemplate.getEffectiveEndTime().after(profitAgreementTemplate.getEffectiveStartTime()), "分利模板生效结束时间必须大于生效开始时间！");
    Validate.isTrue(profitAgreementTemplate.getSignEndTime().after(profitAgreementTemplate.getSignStartTime()), "分利模板签署结束时间必须大于签署开始时间！");
    Validate.isTrue(profitAgreementTemplate.getEffectiveEndTime().getTime()>=profitAgreementTemplate.getSignEndTime().getTime(), "分利模板生效结束时间必须大于签署结束时间！");
    Validate.isTrue(!CollectionUtils.isEmpty(profitAgreementTemplate.getTemplatePolicyRelationships()), "模板必须指定至少一条政策！");
  }

  /**
   * 在做修改操作之前，必须要保证该profitAgreementTemplate对象内的各项属性的正确性，确保其主键真实存在
   * @param profitAgreementTemplate
   */
  private void updateValidation(ProfitAgreementTemplate profitAgreementTemplate){
    Validate.notNull(profitAgreementTemplate.getId(),"做数据操作时，信息的数据编号（主键）不能为空!");
    Validate.notBlank(profitAgreementTemplate.getTemplateName(), "做数据操作时，分利模板名称不能为空！");
    Validate.notBlank(profitAgreementTemplate.getAgreementText(), "做数据操作时，协议文本不能为空！");
    Validate.notNull(profitAgreementTemplate.getEffectiveStartTime(), "做数据操作时，分利模板生效开始时间不能为空！");
    Validate.notNull(profitAgreementTemplate.getEffectiveEndTime(), "做数据操作时，分利模板生效结束时间不能为空！");
    Validate.notNull(profitAgreementTemplate.getSignStartTime(), "做数据操作时，分利模板签署开始时间不能为空！");
    Validate.notNull(profitAgreementTemplate.getSignEndTime(), "做数据操作时，分利模板签署结束时间不能为空！");
    Validate.notNull(profitAgreementTemplate.getAutoSign(), "做数据操作时，分利模板是否自动签署不能为空！");
    Validate.notEmpty(profitAgreementTemplate.getTemplatePolicyRelationships(), "做数据操作时，分利模板包含政策不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况）
    Validate.isTrue(profitAgreementTemplate.getTemplateName() == null || profitAgreementTemplate.getTemplateName().length() < 64, "分利终端编码,在进行添加时填入值超过了限定长度(64)，请检查!");
    //验证时间，模板生效和签署时间的结束时间需大于开始时间，生效结束时间需大于签署结束时间
    Validate.isTrue(profitAgreementTemplate.getEffectiveEndTime().after(profitAgreementTemplate.getEffectiveStartTime()), "分利模板生效结束时间必须大于生效开始时间！");
    Validate.isTrue(profitAgreementTemplate.getSignEndTime().after(profitAgreementTemplate.getSignStartTime()), "分利模板签署结束时间必须大于签署开始时间！");
    Validate.isTrue(profitAgreementTemplate.getEffectiveEndTime().getTime()>=profitAgreementTemplate.getSignEndTime().getTime(), "分利模板生效结束时间必须大于签署结束时间！");
    Validate.isTrue(!CollectionUtils.isEmpty(profitAgreementTemplate.getTemplatePolicyRelationships()), "模板必须指定至少一条政策！");
  }

}
