package com.biz.crm.cps.business.agreement.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.model.LoginUserDetailsForCPS;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.cps.business.agreement.local.entity.Agreement;
import com.biz.crm.cps.business.agreement.local.entity.AgreementSign;
import com.biz.crm.cps.business.agreement.local.entity.ProfitAgreementTemplate;
import com.biz.crm.cps.business.agreement.local.repository.AgreementRepository;
import com.biz.crm.cps.business.agreement.sdk.common.enums.AgreementStatusEnum;
import com.biz.crm.cps.business.agreement.sdk.common.enums.SignStatusEnum;
import com.biz.crm.cps.business.agreement.sdk.common.enums.SignatoryEnum;
import com.biz.crm.cps.business.agreement.sdk.dto.AgreementDto;
import com.biz.crm.cps.business.agreement.sdk.dto.LoginUserAgreementDto;
import com.biz.crm.cps.business.agreement.sdk.service.AgreementVoService;
import com.biz.crm.cps.business.agreement.sdk.vo.AgreementVo;
import com.biz.crm.cps.business.participator.sdk.common.enums.ParticipatorTypeEnum;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 协议实例voService实现类
 * @author songjingen
 */
@Service
@Slf4j
public class AgreementVoServiceImpl implements AgreementVoService {

  @Resource
  private AgreementRepository agreementRepository;
  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Autowired(required = false)
  private LoginUserService loginUserService;

  /**
   * 根据登录信息查询出协议实例集合
   * @param dto
   * @return
   */
  @Override
  public List<AgreementVo> findByConditions(LoginUserAgreementDto dto) {
    if (dto == null){
      dto = new LoginUserAgreementDto();
    }
    List<Agreement> agreements = agreementRepository.findByConditions(dto);
    if (CollectionUtils.isEmpty(agreements)) {
      return null;
    }
    List<AgreementVo> agreementVos = new ArrayList<>();
    agreements.stream().forEach(agreement -> {
      AgreementVo agreementVo = this.buildAgreementVo(agreement);
      agreementVos.add(agreementVo);
    });
    return agreementVos;
  }

  /**
   * 根据条件分页查询分利协议信息
   *
   * @param pageable
   * @param dto
   * @return
   */
  @Override
  public Page<AgreementVo> findByConditions(Pageable pageable, AgreementDto dto) {
    /**
     * 1、查询协议信息
     * 2、取协议信息中的模板信息，签署时间和协议时间赋值到协议vo中
     */
    ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    if(dto == null) {
      dto = new AgreementDto();
    }
    Page<Agreement> byConditions = agreementRepository.findByConditions(pageable, dto);
    ArrayList<AgreementVo> agreementVos = new ArrayList<>();
    if (!CollectionUtils.isEmpty(byConditions.getRecords())) {
      byConditions.getRecords().stream().forEach(agreement -> {
        AgreementVo agreementVo = this.buildAgreementVo(agreement);
        agreementVos.add(agreementVo);
      });
    }
    Page<AgreementVo> agreementVoPage = new Page<AgreementVo>();
    agreementVoPage.setCurrent(byConditions.getCurrent());
    agreementVoPage.setSize(byConditions.getSize());
    agreementVoPage.setTotal(byConditions.getTotal());
    agreementVoPage.setRecords(agreementVos);
    return agreementVoPage;
  }

  /**
   * 根据模板编码查询协议文本
   * @param id
   * @return
   */
  @Override
  public AgreementVo findById(String id) {
    /**
     * 1、查询当前协议信息
     * 2、把签署记录的签名名称、签名路径赋值到协议vo中
     * 3、把协议文本赋值到协议vo中
     */
    if (StringUtils.isEmpty(id)) {
      return null;
    }
    Agreement agreement =  this.agreementRepository.findDetailsById(id);
    if (Objects.isNull(agreement)) {
      return null;
    }
    AgreementVo agreementVo = this.nebulaToolkitService.copyObjectByWhiteList(agreement, AgreementVo.class, null, null);
    if (Objects.isNull(agreement.getProfitAgreementTemplate())) {
      return null;
    }
    Set<AgreementSign> agreementSigns = agreement.getAgreementSigns();
    if (!CollectionUtils.isEmpty(agreementSigns)) {
      List<AgreementSign> sign = agreementSigns.stream().filter(agreementSign -> SignatoryEnum.TERMINAL.getDictCode().equals(agreementSign.getSignatory())).collect(Collectors.toList());
      if (!CollectionUtils.isEmpty(sign)) {
        AgreementSign agreementSign = sign.get(0);
        agreementVo.setSignaturePath(agreementSign.getSignaturePath());
        agreementVo.setSignatureFilename(agreementSign.getSignatureFilename());
        agreementVo.setSignTime(agreementSign.getSignDate());
      }
    }
    agreementVo.setAgreementText(agreement.getProfitAgreementTemplate().getAgreementText());
    return agreementVo;
  }

  /**
   * 根据协议编码查询协议基本信息
   * @param agreementCode
   * @return
   */
  @Override
  public AgreementVo findByAgreementCode(String agreementCode) {
    if (StringUtils.isEmpty(agreementCode)) {
      return null;
    }
    Agreement agreement = this.agreementRepository.findByAgreementCode(agreementCode);
    if (Objects.isNull(agreement)) {
      return null;
    }
    return this.buildAgreementVo(agreement);
  }

  /**
   * 查询当前用户的协议信息是否存在
   * @return
   */
  @Override
  public Boolean findExistByCurrentUser() {
    LoginUserDetailsForCPS loginUser = this.loginUserService.getLoginDetails(LoginUserDetailsForCPS.class);
    log.info("当前登录用户信息：" + loginUser);
    if (Objects.isNull(loginUser)) {
      return false;
    }
    String usertype = loginUser.getUsertype();
    String consumerCode = loginUser.getConsumerCode();

    LoginUserAgreementDto dto = new LoginUserAgreementDto();
    if (ParticipatorTypeEnum.DEALER.getKey().equals(usertype)) {
      dto.setCustomerCode(consumerCode);
    } else if (ParticipatorTypeEnum.TERMINAL.getKey().equals(usertype)) {
      dto.setTerminalCode(consumerCode);
    } else {
      return false;
    }
    dto.setAgreementStatus(AgreementStatusEnum.EXECUTING.getCode());
    dto.setSignStatus(SignStatusEnum.SIGNED.getCode());
    List<Agreement> agreements = this.agreementRepository.findByConditions(dto);
    if (CollectionUtils.isEmpty(agreements)) {
      return false;
    }
    return true;
  }

  /**
   * 组装协议vo
   * @param agreement
   * @return
   */
  private AgreementVo buildAgreementVo(Agreement agreement) {
    if (Objects.isNull(agreement)) {
      return null;
    }
    AgreementVo agreementVo = nebulaToolkitService.copyObjectByWhiteList(agreement, AgreementVo.class, null, null);
    ProfitAgreementTemplate profitAgreementTemplate = agreement.getProfitAgreementTemplate();
    if (profitAgreementTemplate != null) {
      agreementVo.setEffectiveStartTime(profitAgreementTemplate.getEffectiveStartTime());
      agreementVo.setEffectiveEndTime(profitAgreementTemplate.getEffectiveEndTime());
      agreementVo.setSignStartTime(profitAgreementTemplate.getSignStartTime());
      agreementVo.setSignEndTime(profitAgreementTemplate.getSignEndTime());
    }
    return agreementVo;
  }
}
