package com.biz.crm.cps.business.agreement.local.service.notifier;

import com.biz.crm.cps.business.agreement.sdk.common.enums.AgreementStatusEnum;
import com.biz.crm.cps.business.agreement.sdk.common.enums.SignStatusEnum;
import com.biz.crm.cps.business.agreement.sdk.dto.LoginUserAgreementDto;
import com.biz.crm.cps.business.agreement.sdk.dto.PolicyObserverDto;
import com.biz.crm.cps.business.agreement.sdk.service.AgreementVoService;
import com.biz.crm.cps.business.agreement.sdk.service.observer.AgreementPolicyServiceObserver;
import com.biz.crm.cps.business.agreement.sdk.vo.AgreementVo;
import com.biz.crm.cps.business.participator.sdk.common.enums.ParticipatorTypeEnum;
import com.biz.crm.cps.business.product.sdk.dto.MaterialDimensionDto;
import com.biz.crm.cps.business.product.sdk.dto.ProductDimensionDto;
import com.biz.crm.cps.business.product.sdk.service.MaterialVoService;
import com.biz.crm.cps.business.product.sdk.vo.MaterialDimensionVo;
import com.biz.crm.cps.external.barcode.sdk.common.enums.BarCodeTypeEnum;
import com.biz.crm.cps.external.barcode.sdk.dto.ConsumerScanCodeDto;
import com.biz.crm.cps.external.barcode.sdk.dto.ScanCodeExceptionStrategyDto;
import com.biz.crm.cps.external.barcode.sdk.dto.ScanCodeRecordEventDto;
import com.biz.crm.cps.external.barcode.sdk.event.ScanCodeRecordEventListener;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * TODO
 * 实现模块【外部扫码】的事件接口
 * @author hefan
 */
@Component
public class ScanCodeRecordEventListenerImpl implements ScanCodeRecordEventListener {

  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Autowired
  private AgreementVoService agreementVoService;

  @Autowired(required = false)
  private List<AgreementPolicyServiceObserver> agreementPolicyServiceObserver;

  @Autowired(required = false)
  private MaterialVoService materialVoService;

  @Override
  public void onSuccessScan(ScanCodeRecordEventDto scanCodeRecordEventDto) {
    /**
     * 查询该扫码者可以使用的协议模板code集合
     * 根据入参信息查询【该事件主体已有的生效协议】
     * 没有生效中的协议，正常阻断，结束后续代码
     * 保存条码跟协议们的关系
     * 根据商品泛化编码查询商品在各个维度上的编码 -- 入参
     * 有生效的协议。告知观察者们，叫他们起来干活
     *
     */
    //根据入参信息查询【该事件主体已有的生效协议】
    List<AgreementVo> agreementVos = this.findAgreementVos(scanCodeRecordEventDto);
    if (CollectionUtils.isEmpty(agreementVos)) {
      return;
    }
    //查询该扫码者可以使用的协议模板code集合
    Set<String> templateCodes = agreementVos.stream().map(AgreementVo::getTemplateCode).collect(Collectors.toSet());
    //没有生效中的协议，正常阻断，结束后续代码
    if (CollectionUtils.isEmpty(templateCodes)) {
      return;
    }
    // 根据物料泛化编码查询维度
    MaterialDimensionDto materialDimensionDto = this.findMaterialDimension(scanCodeRecordEventDto.getProductCode());
    // 组装协议到政策观察者的dto
    PolicyObserverDto policyObserverDto = this.buildPolicyObserverDto(scanCodeRecordEventDto, templateCodes,null, materialDimensionDto);
    //有生效的协议。告知观察者们，叫他们起来干活。
    this.callPolicyObservers(policyObserverDto);
  }

  /**
   * 组装协议到政策观察者的dto
   *
   * @param scanCodeRecordEventDto
   * @param templateCodes
   * @param productDimensionDto
   * @param materialDimensionDto
   * @return
   */
  private PolicyObserverDto buildPolicyObserverDto(ScanCodeRecordEventDto scanCodeRecordEventDto, Set<String> templateCodes, ProductDimensionDto productDimensionDto,MaterialDimensionDto materialDimensionDto) {
    PolicyObserverDto policyObserverDto = new PolicyObserverDto();
    policyObserverDto.setTemplateCodes(templateCodes);
    policyObserverDto.setBarCodeType(scanCodeRecordEventDto.getBarCodeType());
    policyObserverDto.setParticipatorCode(scanCodeRecordEventDto.getScanParticipatorCode());
    String participatorType = scanCodeRecordEventDto.getParticipatorType();
    ParticipatorTypeEnum typeEnum = ParticipatorTypeEnum.getByKey(participatorType);
    policyObserverDto.setParticipatorType(typeEnum.getDictCode());
    policyObserverDto.setProductCode(scanCodeRecordEventDto.getProductCode());
    policyObserverDto.setRecordCode(scanCodeRecordEventDto.getRecordCode());
    policyObserverDto.setProductDimensionDto(productDimensionDto);
    policyObserverDto.setMaterialDimensionDto(materialDimensionDto);
    return policyObserverDto;
  }

  /**
   * 根据物料泛化编码查询维度
   * @param materialCode
   * @return
   */
  private MaterialDimensionDto findMaterialDimension(String materialCode) {
    Validate.notNull(materialVoService, "系统类缺少MaterialVoService实现");
    MaterialDimensionVo materialDimensionVo = materialVoService.findByMaterialCode(materialCode);
    Validate.notNull(materialDimensionVo, "物料/商品信息不存在");
    return this.nebulaToolkitService.copyObjectByBlankList(materialDimensionVo, MaterialDimensionDto.class, HashSet.class, LinkedList.class);
  }

  /**
   * 获取当前登录人
   * 查询当前参与者的已生效的协议信息
   * @param scanCodeRecordEventDto
   * @return
   */
  private List<AgreementVo> findAgreementVos(ScanCodeRecordEventDto scanCodeRecordEventDto) {
    Validate.notNull(scanCodeRecordEventDto, "扫码记录信息不能为空！");
    //查询当前参与者的已生效的协议信息
    LoginUserAgreementDto loginUserAgreementDto = new LoginUserAgreementDto();
    if (BarCodeTypeEnum.COVER_CODE.getFlag().equals(scanCodeRecordEventDto.getBarCodeType())) {
      ConsumerScanCodeDto consumerScanCodeDto = scanCodeRecordEventDto.getConsumerScanCodeDto();
      if (StringUtils.isBlank(consumerScanCodeDto.getBarCodeParticipatorCode())) {
        return null;
      }
      this.buildLoginUserAgreementDto(consumerScanCodeDto.getBarCodeParticipatorCode(),consumerScanCodeDto.getParticipatorType(),loginUserAgreementDto);
    } else {
      this.buildLoginUserAgreementDto(scanCodeRecordEventDto.getBarCodeParticipatorCode(),scanCodeRecordEventDto.getParticipatorType(),loginUserAgreementDto);
    }
    List<AgreementVo> agreementVos = agreementVoService.findByConditions(loginUserAgreementDto);
    if(CollectionUtils.isEmpty(agreementVos)) {
      return Lists.newLinkedList();
    }
    return agreementVos;
  }

  /**
   * 组装已生效协议查询条件
   * @param barCodeParticipatorCode
   * @param participatorType
   */
  private void buildLoginUserAgreementDto(String barCodeParticipatorCode, String participatorType, LoginUserAgreementDto loginUserAgreementDto) {
    if (ParticipatorTypeEnum.TERMINAL.getKey().equals(participatorType)) {
      loginUserAgreementDto.setTerminalCode(barCodeParticipatorCode);
    } else if (ParticipatorTypeEnum.DEALER.getKey().equals(participatorType)){
      loginUserAgreementDto.setCustomerCode(barCodeParticipatorCode);
    }
    loginUserAgreementDto.setAgreementStatus(AgreementStatusEnum.EXECUTING.getCode());
    loginUserAgreementDto.setSignStatus(SignStatusEnum.SIGNED.getCode());
  }


  /**
   * 调用政策模块组内AgreementPolicyServiceObserver的实现类
   * 完成扫码的后续处理
   * @param policyObserverDto
   */
  private void callPolicyObservers(PolicyObserverDto policyObserverDto) {
    if (CollectionUtils.isEmpty(agreementPolicyServiceObserver)) {
      return;
    }
    for (AgreementPolicyServiceObserver policyServiceObserver : agreementPolicyServiceObserver) {
      policyServiceObserver.onProcessScanCode(policyObserverDto);
    }
  }

  /**
   * 接收到下层扫码失败事件后进行逻辑处理
   * @param dto
   */
  @Override
  public void onFailScan(ScanCodeRecordEventDto dto) {
    //当前模块暂时不需要处理此事件
  }
}
