package com.biz.crm.cps.business.attendance.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.model.LoginUserDetailsForCPS;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.cps.business.attendance.local.entity.AttendanceClock;
import com.biz.crm.cps.business.attendance.local.entity.AttendanceShiftApplication;
import com.biz.crm.cps.business.attendance.local.entity.AttendanceShiftPlan;
import com.biz.crm.cps.business.attendance.local.repository.AttendanceClockRepository;
import com.biz.crm.cps.business.attendance.local.service.AttendanceClockPictureRelationshipService;
import com.biz.crm.cps.business.attendance.local.service.AttendanceClockService;
import com.biz.crm.cps.business.attendance.local.service.AttendanceShiftApplicationService;
import com.biz.crm.cps.business.attendance.local.service.AttendanceShiftPlanService;
import com.biz.crm.cps.business.attendance.sdk.common.enums.ClockStatusEnum;
import com.biz.crm.cps.business.attendance.sdk.common.enums.ClockTypeEnum;
import com.biz.crm.cps.business.attendance.sdk.common.enums.ShiftApplicationAuditStatusEnum;
import com.biz.crm.cps.business.attendance.sdk.common.enums.ShiftPlanTypeEnum;
import com.biz.crm.cps.business.attendance.sdk.dto.AttendanceClockConditionDto;
import com.biz.crm.cps.business.attendance.sdk.dto.AttendanceClockDto;
import com.biz.crm.cps.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.cps.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.mdm.business.terminal.sdk.service.TerminalVoService;
import com.biz.crm.mdm.business.terminal.sdk.vo.TerminalVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.time.DateUtils;
import org.gavaghan.geodesy.Ellipsoid;
import org.gavaghan.geodesy.GeodeticCalculator;
import org.gavaghan.geodesy.GeodeticCurve;
import org.gavaghan.geodesy.GlobalCoordinates;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Objects;

/**
 * 考勤打卡明细(AttendanceClock)表服务实现类
 *
 * @author dy
 * @since 2022-03-07 10:29:41
 */
@Service("attendanceClockService")
public class AttendanceClockServiceImpl implements AttendanceClockService {

  @Autowired
  private AttendanceClockRepository attendanceClockRepository;
  @Autowired
  private AttendanceShiftPlanService attendanceShiftPlanService;
  @Autowired
  private AttendanceShiftApplicationService attendanceShiftApplicationService;
  @Autowired
  private LoginUserService loginUserService;
  @Autowired
  private AttendanceClockPictureRelationshipService attendanceClockPictureRelationshipService;
  @Autowired
  private TerminalVoService terminalVoService;

  /**
   * 最远打卡距离 单位(m)
   */
  private static final double MAXIMUM_CLOCK_DISTANCE = 1000;

  /**
   * 分页查询数据
   *
   * @param pageable        分页对象
   * @param dto 实体对象
   * @return
   */
  @Override
  public Page<AttendanceClock> findByConditions(Pageable pageable, AttendanceClockConditionDto dto) {
    ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    if (Objects.isNull(dto)) {
      dto = new AttendanceClockConditionDto();
    }
    return this.attendanceClockRepository.findByConditions(pageable, dto);
  }

  /**
   * 通过主键查询单条数据
   *
   * @param id 主键
   * @return 单条数据
   */
  @Override
  public AttendanceClock findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.attendanceClockRepository.getById(id);
  }

  /**
   * 删除数据
   *
   * @param idList 主键结合
   * @return 删除结果
   */
  @Transactional
  @Override
  public void delete(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "删除数据时，主键集合不能为空！");
    this.attendanceClockRepository.removeByIds(idList);
  }

  @Override
  @Transactional(rollbackFor = Exception.class)
  public AttendanceClock create(AttendanceClockDto attendanceClockDto) {

    this.clockValidate(attendanceClockDto);

    Date now = new Date();
    String clockType = attendanceClockDto.getClockType();
    AttendanceClock attendanceClock = new AttendanceClock();
    attendanceClock.setLatitude(attendanceClockDto.getLatitude());
    attendanceClock.setLongitude(attendanceClockDto.getLongitude());

    if(ClockTypeEnum.CLOCK_IN.getDictCode().equals(clockType)){
      this.handleClockIn(attendanceClock);
    }else if(ClockTypeEnum.CLOCK_OUT.getDictCode().equals(clockType)){
      this.handleClockOut(attendanceClock);
    }else {
      throw new IllegalArgumentException("打卡类型参数错误");
    }

    attendanceClock.setClockDate(now);
    // 打卡参数拷贝
    BeanUtils.copyProperties(attendanceClockDto,attendanceClock);
    attendanceClock.setTenantCode(TenantUtils.getTenantCode());
    attendanceClock.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    attendanceClock.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    attendanceClock.setCreateAccount(loginUserService.getLoginAccountName());
    attendanceClock.setCreateTime(now);
    attendanceClock.setModifyAccount(loginUserService.getLoginAccountName());
    attendanceClock.setModifyTime(now);

    this.attendanceClockRepository.saveOrUpdate(attendanceClock);
    attendanceClockPictureRelationshipService.createBatch(attendanceClock.getId(),attendanceClockDto.getPictures());
    return attendanceClock;
  }

  @Transactional(rollbackFor = Exception.class)
  @Override
  public void handleClockIn(AttendanceClock attendanceClock) {
    /*
      1. 获取此时的排班计划详情
        1.1 先判断前一天是否有排班计划，如果有计划，是跨天班次并且此时时间还在前一天的工作时间内，再判断是否打过上班卡,如果没打过上班卡，排班计划为前一天的计划
        1.2 如果前一天没有计划，判断当天是否有排班计划，如果有计划，排班计划为当天的计划
      2. 判断计划是否可用，包括审核状态，排班类型,是否在签到范围内
      3. 计算打卡结果并保存，如果打卡时间在上班时间或者最迟打卡时间之前,打卡状态为正常，否者为迟到
     */

    Date now = new Date();
    Date previousDays = DateUtils.addDays(now, -1);
    LoginUserDetailsForCPS loginUser = loginUserService.getLoginDetails(LoginUserDetailsForCPS.class);
    Validate.notNull(loginUser,"获取当前登录用户信息失败");
    String account = loginUser.getConsumerCode();

    // 1
    AttendanceShiftPlan plan = null;
    AttendanceShiftPlan previousDayPlan = attendanceShiftPlanService.findByUserAccountAndSchedulingDate(account, previousDays);
    AttendanceShiftPlan todayPlan = attendanceShiftPlanService.findByUserAccountAndSchedulingDate(account, now);
    // 1.1
    if(Objects.nonNull(previousDayPlan) && Objects.nonNull(previousDayPlan.getWorkEndDate()) && now.before(previousDayPlan.getWorkEndDate())){
      AttendanceShiftApplication application = attendanceShiftApplicationService.findByApplyCode(plan.getApplyCode());
      if(Objects.nonNull(application) && ShiftApplicationAuditStatusEnum.PASS.getDictCode().equals(application.getAuditStatus())){
        AttendanceClock previousDayPlanClock = this.attendanceClockRepository.findByUserAccountAndAttendanceDateAndClockType(account, previousDayPlan.getSchedulingDate(), ClockTypeEnum.CLOCK_IN.getDictCode());
        plan = previousDayPlanClock == null ? previousDayPlan : null;
      }
    }
    // 1.2
    if(plan == null && Objects.nonNull(todayPlan)){
      AttendanceClock todayPlanClock = this.attendanceClockRepository.findByUserAccountAndAttendanceDateAndClockType(account, todayPlan.getSchedulingDate(), ClockTypeEnum.CLOCK_IN.getDictCode());
      Validate.isTrue(todayPlanClock == null,String.format("您在%tF 的班次 %s已经打过上班卡",todayPlan.getSchedulingDate(),todayPlan.getShiftName()));
      plan = todayPlan;
    }
    // 2
    Validate.notNull(plan,"当前暂无排班计划");
    AttendanceShiftApplication application = attendanceShiftApplicationService.findByApplyCode(plan.getApplyCode());
    Validate.isTrue(application != null,"当前暂未排班计划，请先创建");
    Validate.isTrue(ShiftApplicationAuditStatusEnum.PASS.getDictCode().equals(application.getAuditStatus()),"排班计划申请正在审核中或被拒绝，请联系相关业务员处理");
    // 只有早班，中班，晚班，三种类型需要打卡，全天调休的班次不需要打卡
    Validate.isTrue(StringUtils.equalsAny(plan.getShiftPlanType(), ShiftPlanTypeEnum.MORNING_SHIFT.getDictCode(),ShiftPlanTypeEnum.MIDDLE_SHIFT.getDictCode(),ShiftPlanTypeEnum.NIGHT_SHIFT.getDictCode()),"当前班次类型不需要打卡");

    this.clockAddressValid(plan.getTerminalCode(),attendanceClock);

    // 业务数据
    attendanceClock.setShiftPlanId(plan.getId());
    attendanceClock.setAttendanceDate(plan.getSchedulingDate());
    attendanceClock.setUserAccount(loginUser.getConsumerCode());
    attendanceClock.setUserCode(loginUser.getAccount());
    attendanceClock.setUserName(loginUser.getConsumerName());
    // 3
    Date workStartDate = plan.getWorkStartDate();
    Date latestSignInDate = plan.getLatestSignInDate();
    // 如果打卡时间在上班时间或者最迟打卡时间之前,打卡状态为正常，上班打卡时间在下班时间
    if(now.before(workStartDate) || (latestSignInDate != null) && now.before(latestSignInDate)){
      attendanceClock.setClockStatus(ClockStatusEnum.NORMAL.getDictCode());
    }else {
      // 否则算迟到
      attendanceClock.setClockStatus(ClockStatusEnum.LATE.getDictCode());
    }
  }


  @Transactional(rollbackFor = Exception.class)
  @Override
  public void handleClockOut(AttendanceClock attendanceClock) {

    /*
      1. 获取此时的排班计划详情
        1.1 先判断前一天是否有排班计划,如果有计划，是跨天班次并且此时时间还在前一天的工作时间内，排班计划为前一天的计划
        1.2 如果前一天没有排班计划，判断当天是否有排班计划，如果有计划，排班计划为当天的计划
      2. 判断计划是否可用，包括审核状态，排班类型，是否在签到范围内
      3. 判断是否已经打过上班卡，没有打上班卡，不能打下班卡
      4. 判断是否已经打过卡，下班卡可以重复打，以最后一次为准
      5. 计算打卡结果，如果打卡时间在下班时间或者最早下班时间之后,打卡状态为正常，否者为早退
     */

    Date now = new Date();
    Date previousDays = DateUtils.addDays(now, -1);
    LoginUserDetailsForCPS loginUser = loginUserService.getLoginDetails(LoginUserDetailsForCPS.class);
    Validate.notNull(loginUser,"获取当前登录用户信息失败");
    String account = loginUser.getConsumerCode();

    // 1
    AttendanceShiftPlan plan = null;
    AttendanceClock existClock = null;
    AttendanceShiftPlan previousDayPlan = attendanceShiftPlanService.findByUserAccountAndSchedulingDate(account, previousDays);
    AttendanceShiftPlan todayPlan = attendanceShiftPlanService.findByUserAccountAndSchedulingDate(account, now);

    // 1.1
    if(Objects.nonNull(previousDayPlan) && Objects.nonNull(previousDayPlan.getWorkEndDate()) && now.before(previousDayPlan.getWorkEndDate())){
      existClock = this.attendanceClockRepository.findByUserAccountAndAttendanceDateAndClockType(account, previousDayPlan.getSchedulingDate(), ClockTypeEnum.CLOCK_OUT.getDictCode());
      plan = previousDayPlan;
    }
    // 1.2
    if(plan == null && Objects.nonNull(todayPlan)){
      existClock = this.attendanceClockRepository.findByUserAccountAndAttendanceDateAndClockType(account, todayPlan.getSchedulingDate(), ClockTypeEnum.CLOCK_OUT.getDictCode());
      plan = todayPlan;
    }

    // 2
    Validate.notNull(plan,"当前暂无排班计划");
    AttendanceShiftApplication application = attendanceShiftApplicationService.findByApplyCode(plan.getApplyCode());
    Validate.isTrue(application != null,"当前暂未排班计划，请先创建");
    Validate.isTrue(ShiftApplicationAuditStatusEnum.PASS.getDictCode().equals(application.getAuditStatus()),"排班计划申请正在审核中或被拒绝，请联系相关业务员处理");
    // 只有早班，中班，晚班，三种类型需要打卡，全天调休的班次不需要打卡
    Validate.isTrue(StringUtils.equalsAny(plan.getShiftPlanType(), ShiftPlanTypeEnum.MORNING_SHIFT.getDictCode(),ShiftPlanTypeEnum.MIDDLE_SHIFT.getDictCode(),ShiftPlanTypeEnum.NIGHT_SHIFT.getDictCode()),"当前班次类型不需要打卡");

    this.clockAddressValid(plan.getTerminalCode(),attendanceClock);

    // 3
    AttendanceClock clockIn = attendanceClockRepository.findByUserAccountAndAttendanceDateAndClockType(account, plan.getSchedulingDate(), ClockTypeEnum.CLOCK_IN.getDictCode());
    Validate.notNull(clockIn,"当前未打上班卡，请先完成上班卡后重试");
    // 业务数据
    attendanceClock.setShiftPlanId(plan.getId());
    attendanceClock.setAttendanceDate(plan.getSchedulingDate());
    attendanceClock.setUserAccount(loginUser.getConsumerCode());
    attendanceClock.setUserCode(loginUser.getAccount());
    attendanceClock.setUserName(loginUser.getConsumerName());

    // 4
    if(existClock != null){
      attendanceClock.setId(existClock.getId());
    }

    // 5
    Date workEndDate = plan.getWorkEndDate();
    Date earliestSignBackDate = plan.getEarliestSignBackDate();
    // 如果打卡时间在上班时间或者最迟打卡时间之前,打卡状态为正常，上班打卡时间在下班时间
    if((earliestSignBackDate != null) && now.after(earliestSignBackDate) || now.after(workEndDate)){
      attendanceClock.setClockStatus(ClockStatusEnum.NORMAL.getDictCode());
    }else {
      attendanceClock.setClockStatus(ClockStatusEnum.LEAVE_EARLY.getDictCode());
    }
  }

  @Override
  public List<AttendanceClock> findByShiftPlanId(String shiftPlanId) {
    if(StringUtils.isBlank(shiftPlanId)){
      return Lists.newArrayList();
    }
    return this.attendanceClockRepository.findByShiftPlanId(shiftPlanId);
  }


  /**
   * 打卡记录创建前的参数校验
   * @param attendanceClockDto
   */
  private void clockValidate(AttendanceClockDto attendanceClockDto){
    Validate.notBlank(attendanceClockDto.getClockType(),"打卡示，打卡类型参数不能为空");
    Validate.notBlank(attendanceClockDto.getClockAddress(), "打卡时， 打卡详细地址地址 不能为空！");
    Validate.notBlank(attendanceClockDto.getClockAddressCode(),"打卡时，打卡详细地址编码不能为空");
    Validate.notNull(attendanceClockDto.getLatitude(), "打卡时， 打卡纬度 不能为空！");
    Validate.notNull(attendanceClockDto.getLongitude(), "打卡时， 打卡经度 不能为空！");
  }

  /**
   * 打卡地址校验,校验打卡经纬度和终端经纬度之间的距离是否小于最大打卡距离
   * @param attendanceClock
   * @param terminalCode
   */
  private void clockAddressValid(String terminalCode,AttendanceClock attendanceClock){

    List<TerminalVo> terminalVos = terminalVoService.findMainDetailsByTerminalCodes(Arrays.asList(terminalCode));
    Validate.isTrue(!CollectionUtils.isEmpty(terminalVos),"未能找到对应的终端信息");
    TerminalVo terminalVo = terminalVos.get(0);
    Validate.notNull(terminalVo,"未能找到对应的终端信息");

    BigDecimal terminalLongitude = terminalVo.getLongitude();
    BigDecimal terminalLatitude = terminalVo.getLatitude();
    Validate.isTrue(terminalLatitude != null && terminalLongitude != null,"终端的经度，纬度不能为空");

    //计算坐标距离
    GlobalCoordinates source = new GlobalCoordinates(terminalLatitude.doubleValue(), terminalLongitude.doubleValue());
    GlobalCoordinates target = new GlobalCoordinates(attendanceClock.getLatitude().doubleValue(),attendanceClock.getLongitude().doubleValue());
    GeodeticCurve geoCurve = new GeodeticCalculator().calculateGeodeticCurve(Ellipsoid.Sphere, source, target);
    double distance = Math.abs(geoCurve.getEllipsoidalDistance());
    Validate.isTrue(distance <= MAXIMUM_CLOCK_DISTANCE,"当前不在签到范围内");
  }
}

