package com.biz.crm.cps.business.attendance.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.cps.business.attendance.local.entity.AttendanceClock;
import com.biz.crm.cps.business.attendance.local.entity.AttendanceDetailSettlement;
import com.biz.crm.cps.business.attendance.local.entity.AttendanceShiftApplication;
import com.biz.crm.cps.business.attendance.local.entity.AttendanceShiftPlan;
import com.biz.crm.cps.business.attendance.local.repository.AttendanceClockRepository;
import com.biz.crm.cps.business.attendance.local.repository.AttendanceDetailSettlementRepository;
import com.biz.crm.cps.business.attendance.local.service.AttendanceDetailSettlementService;
import com.biz.crm.cps.business.attendance.local.service.AttendanceShiftApplicationService;
import com.biz.crm.cps.business.attendance.local.service.AttendanceShiftPlanService;
import com.biz.crm.cps.business.attendance.sdk.common.enums.AttendanceStatusEnum;
import com.biz.crm.cps.business.attendance.sdk.common.enums.ClockStatusEnum;
import com.biz.crm.cps.business.attendance.sdk.common.enums.ClockTypeEnum;
import com.biz.crm.cps.business.attendance.sdk.common.enums.ShiftApplicationAuditStatusEnum;
import com.biz.crm.cps.business.attendance.sdk.common.enums.ShiftPlanTypeEnum;
import com.biz.crm.cps.business.attendance.sdk.dto.AttendanceDetailConditionDto;
import com.biz.crm.cps.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.cps.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.mdm.business.user.sdk.service.UserFeignVoService;
import com.biz.crm.mdm.business.user.sdk.vo.UserVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.task.annotations.DynamicTaskService;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.time.DateUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 考勤详情结算(AttendanceDetailSettlement)表服务实现类
 *
 * @author dy
 * @since 2022-03-03 17:36:14
 */
@Service("attendanceDetailSettlementService")
public class AttendanceDetailSettlementServiceImpl implements AttendanceDetailSettlementService {

  @Autowired
  private AttendanceDetailSettlementRepository attendanceDetailSettlementRepository;
  @Autowired
  private AttendanceShiftPlanService attendanceShiftPlanService;
  @Autowired
  private AttendanceShiftApplicationService attendanceShiftApplicationService;
  @Autowired
  private AttendanceClockRepository attendanceClockRepository;
  @Autowired
  private UserFeignVoService userFeignVoService;

  /**
   * 分页查询数据
   *
   * @param pageable                   分页对象
   * @param dto 查询条件实体对象
   * @return
   */
  @Override
  public Page<AttendanceDetailSettlement> findByConditions(Pageable pageable, AttendanceDetailConditionDto dto) {
    ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    if (Objects.isNull(dto)) {
      dto = new AttendanceDetailConditionDto();
    }
    return this.attendanceDetailSettlementRepository.findByConditions(pageable, dto);
  }

  /**
   * 通过主键查询单条数据
   *
   * @param id 主键
   * @return 单条数据
   */
  @Override
  public AttendanceDetailSettlement findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.attendanceDetailSettlementRepository.getById(id);
  }

  /**
   * 删除数据
   *
   * @param idList 主键结合
   * @return 删除结果
   */
  @Transactional
  @Override
  public void delete(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "删除数据时，主键集合不能为空！");
    this.attendanceDetailSettlementRepository.removeByIds(idList);
  }

  @Override
  @Transactional(rollbackFor = Exception.class)
  public void hanlderAttendanceReportByDate(Date date) {
     /*
      1. 获取考勤日期为前天的且已经审核通过的排班申请详情
      2. 遍历这部分排班详情
        2.1 如果是休息日，不做处理
        2.2 如果是早班，中班，晚班，拷贝打卡数据到结算信息内，如果没有打卡信息，同步在打卡明细表内生成未打卡数据
     */
    Validate.notNull(date,"指定日期不能为空");
    Date now = new Date();
    attendanceDetailSettlementRepository.deleteByAttendanceDate(DateUtils.truncate(date, Calendar.DATE));
    List<AttendanceShiftPlan> plans = attendanceShiftPlanService.findBySchedulingDateAndAuditStatus(date, ShiftApplicationAuditStatusEnum.PASS.getDictCode());
    if(CollectionUtils.isEmpty(plans)){
      return;
    }
    List<AttendanceDetailSettlement> attendanceDetailSettlements = Lists.newArrayList();
    List<AttendanceClock> notAttendanceClocks = Lists.newArrayList();

    for (AttendanceShiftPlan plan : plans) {
      if(ShiftPlanTypeEnum.REST_DAY.getDictCode().equals(plan.getShiftPlanType())){
        continue;
      }
      AttendanceDetailSettlement attendanceDetailSettlement = new AttendanceDetailSettlement();
      attendanceDetailSettlement.setCreateAccount("admin");
      attendanceDetailSettlement.setModifyAccount("admin");
      attendanceDetailSettlement.setCreateTime(now);
      attendanceDetailSettlement.setModifyTime(now);
      attendanceDetailSettlement.setTenantCode(TenantUtils.getTenantCode());
      attendanceDetailSettlement.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
      attendanceDetailSettlement.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());

      AttendanceShiftApplication application = attendanceShiftApplicationService.findByApplyCode(plan.getApplyCode());
      attendanceDetailSettlement.setAttendanceDate(plan.getSchedulingDate());
      attendanceDetailSettlement.setUserAccount(application.getUserAccount());
      attendanceDetailSettlement.setUserCode(application.getUserCode());
      attendanceDetailSettlement.setUserName(application.getUserName());
      attendanceDetailSettlement.setContactPhone(application.getContactPhone());
      attendanceDetailSettlement.setShiftPlanType(plan.getShiftPlanType());
      attendanceDetailSettlement.setShiftPlanName(plan.getShiftName());
      attendanceDetailSettlement.setGoalWorkHours(BigDecimal.ZERO);
      attendanceDetailSettlement.setRealWorkHours(BigDecimal.ZERO);

      List<UserVo> userVos = userFeignVoService.findByUserNames(Arrays.asList(application.getUserAccount()));
      if(!CollectionUtils.isEmpty(userVos) && userVos.get(0) != null){
        UserVo userVo = userVos.get(0);
        attendanceDetailSettlement.setPositionCode(userVo.getPositionLevelCode());
        attendanceDetailSettlement.setPositionName(userVo.getPositionLevelName());
        attendanceDetailSettlement.setOrgCode(userVo.getOrgCode());
        attendanceDetailSettlement.setOrgName(userVo.getOrgName());
      }

      // 如果是全天调休
      if(ShiftPlanTypeEnum.DAY_COMPENSATORY_LEAVE.getDictCode().equals(plan.getShiftPlanType())){
        attendanceDetailSettlement.setCompensatoryLeaveLength(this.calculateWorkHours(plan.getWorkStartDate(),plan.getWorkEndDate()));
        attendanceDetailSettlement.setAttendanceStatus(AttendanceStatusEnum.DONT_NEED_CLOCK.getDictCode());
        attendanceDetailSettlements.add(attendanceDetailSettlement);
        continue;
      }
      // 说明是需要打卡的早班，中班，晚班
      attendanceDetailSettlement.setGoalWorkHours(plan.getNormalWorkHours());
      attendanceDetailSettlement.setCompensatoryLeaveLength(plan.getCompensatoryLeaveLength());
      // 打卡记录
      List<AttendanceClock> attendanceClocks = attendanceClockRepository.findByShiftPlanId(plan.getId());
      attendanceClocks = attendanceClocks == null ? Lists.newArrayList() : attendanceClocks;
      // 在打卡明细表中生成未打卡记录
      Set<String> clockTypeSet = attendanceClocks.stream().map(AttendanceClock::getClockType).collect(Collectors.toSet());
      for (ClockTypeEnum value : ClockTypeEnum.values()) {
        if(!clockTypeSet.contains(value.getDictCode())){
          AttendanceClock attendanceClock = new AttendanceClock();
          attendanceClock.setClockType(value.getDictCode());
          attendanceClock.setClockStatus(ClockStatusEnum.NOT_CLOCK.getDictCode());
          attendanceClock.setAttendanceDate(plan.getSchedulingDate());
          attendanceClock.setUserAccount(application.getUserAccount());
          attendanceClock.setTenantCode(application.getTenantCode());
          attendanceClock.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
          attendanceClock.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
          attendanceClock.setCreateTime(now);
          attendanceClock.setModifyTime(now);
          attendanceClock.setCreateAccount("admin");
          attendanceClock.setModifyAccount("admin");
          notAttendanceClocks.add(attendanceClock);
        }
      }
      if(CollectionUtils.isEmpty(attendanceClocks)){
        attendanceDetailSettlement.setAttendanceStatus(AttendanceStatusEnum.NOT_CLOCK.getDictCode());
        attendanceDetailSettlements.add(attendanceDetailSettlement);
        continue;
      }
      String clockInStatus = null;
      String clockOutStatus = null;
      Date clockInDate = null;
      Date clockOutDate = null;

      for (AttendanceClock attendanceClock : attendanceClocks) {
        if(ClockTypeEnum.CLOCK_IN.getDictCode().equals(attendanceClock.getClockType())){
          attendanceDetailSettlement.setSignInDate(attendanceClock.getClockDate());
          clockInDate = attendanceClock.getClockDate();
          clockInStatus = attendanceClock.getClockStatus();
        }
        if(ClockTypeEnum.CLOCK_OUT.getDictCode().equals(attendanceClock.getClockType())){
          attendanceDetailSettlement.setSignBackDate(attendanceClock.getClockDate());
          clockOutDate = attendanceClock.getClockDate();
          clockOutStatus = attendanceClock.getClockStatus();
        }

        // 拷贝打卡地址信息
        attendanceDetailSettlement.setAddressCode(attendanceClock.getClockAddressCode());
        attendanceDetailSettlement.setAddress(attendanceClock.getClockAddress());
        attendanceDetailSettlement.setLatitude(attendanceClock.getLatitude());
        attendanceDetailSettlement.setLongitude(attendanceClock.getLongitude());
      }
      // 计算最终的考勤状态
      attendanceDetailSettlement.setAttendanceStatus(this.getAttendanceStatus(clockInStatus,clockOutStatus).getDictCode());
      // 计算实际工时,如果没打下班卡，工时计算为0
      if(clockInDate != null && clockOutDate != null){
        attendanceDetailSettlement.setRealWorkHours(this.calculateWorkHours(clockInDate, clockOutDate));
      }
      attendanceDetailSettlements.add(attendanceDetailSettlement);
    }
    this.attendanceClockRepository.saveBatch(notAttendanceClocks);
    this.attendanceDetailSettlementRepository.saveBatch(attendanceDetailSettlements);
  }

  /**
   * 考勤数据结算定时任务
   */
  @Transactional(rollbackFor = Exception.class)
  @DynamicTaskService(cornExpression = "0 0 0 * * ?", taskDesc = "每日生成前天的考勤数据")
  public void attendanceSettleTask() {
    Date now = new Date();
    Date theDayBefore = DateUtils.addDays(now, -2);
    this.hanlderAttendanceReportByDate(theDayBefore);
  }

  /**
   * 根据上班打卡状态和下班打卡状态确定最终考勤结果
   * @param clockInStatus
   * @param clockOutStatus
   * @return
   */
  private AttendanceStatusEnum getAttendanceStatus(String clockInStatus,String clockOutStatus){
    /*
      1、正常考勤：当天的上下班打卡均无异常
      2、异常考勤：当天的上下班打卡均有异常
      3、迟到：存在上班迟到打卡记录
      4、早退：存在上班早退打卡记录
      5、上班未打卡：没有上班打卡记录
      6、下班未打卡：没有下班打卡记录
      7、未打卡：上下班均无打卡记录
     */
    AttendanceStatusEnum attendanceStatusEnum = null;
    attendanceStatusEnum = attendanceStatusEnum == null && StringUtils.isAllBlank(clockInStatus,clockOutStatus) ? AttendanceStatusEnum.NOT_CLOCK : attendanceStatusEnum;
    attendanceStatusEnum = attendanceStatusEnum == null && StringUtils.isBlank(clockInStatus) ? AttendanceStatusEnum.NOT_CLOCK_IN : attendanceStatusEnum;
    attendanceStatusEnum = attendanceStatusEnum == null && StringUtils.isBlank(clockOutStatus) ? AttendanceStatusEnum.NOT_CLOCK_OUT : attendanceStatusEnum;
    attendanceStatusEnum = attendanceStatusEnum == null && StringUtils.equals(clockInStatus, ClockStatusEnum.LATE.getDictCode()) && StringUtils.equals(clockOutStatus,ClockStatusEnum.LEAVE_EARLY.getDictCode()) ? AttendanceStatusEnum.LATE_AND_LEAVE_EARLY : attendanceStatusEnum;
    attendanceStatusEnum = attendanceStatusEnum == null && StringUtils.equals(clockInStatus,ClockStatusEnum.LATE.getDictCode()) ? AttendanceStatusEnum.LATE : attendanceStatusEnum;
    attendanceStatusEnum = attendanceStatusEnum == null && StringUtils.equals(clockOutStatus,ClockStatusEnum.LEAVE_EARLY.getDictCode()) ? AttendanceStatusEnum.LEAVE_EARLY :attendanceStatusEnum;
    attendanceStatusEnum = attendanceStatusEnum == null && StringUtils.equals(clockInStatus,ClockStatusEnum.NORMAL.getDictCode()) && StringUtils.equals(clockOutStatus,ClockStatusEnum.NORMAL.getDictCode()) ? AttendanceStatusEnum.NORMAL : attendanceStatusEnum;
    if (attendanceStatusEnum == null){
      attendanceStatusEnum = AttendanceStatusEnum.NOT_CLOCK;
    }
    return attendanceStatusEnum;
  }

  /**
   * 计算工时
   * @param start
   * @param end
   * @return
   */
  private BigDecimal calculateWorkHours(Date start,Date end){
    /*
    工时计算方式：
      上班签到时间为9:35= 9+35/60=9+0.58(四舍五入)=9.58h
      下班签退时间为18:30=18+30/60=18+0.5=18.50h
      工时时长为：18.50h-9.58h=8.92h
     */

    BigDecimal s = BigDecimal.valueOf(start.getTime());
    BigDecimal e = BigDecimal.valueOf(end.getTime());
    BigDecimal hourMinutes = BigDecimal.valueOf(60 * 60 * 1000);
    BigDecimal totalMinutes = e.subtract(s);
    return totalMinutes.divide(hourMinutes,2,BigDecimal.ROUND_HALF_UP);
  }
}

