package com.biz.crm.cps.business.attendance.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.model.LoginUserDetailsForCPS;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.cps.business.attendance.local.entity.AttendanceShiftApplication;
import com.biz.crm.cps.business.attendance.local.repository.AttendanceShiftApplicationRepository;
import com.biz.crm.cps.business.attendance.local.service.AttendanceShiftApplicationService;
import com.biz.crm.cps.business.attendance.local.service.AttendanceShiftPlanService;
import com.biz.crm.cps.business.attendance.sdk.common.enums.ShiftApplicationAuditStatusEnum;
import com.biz.crm.cps.business.attendance.sdk.dto.AttendanceShiftApplicationApproveDto;
import com.biz.crm.cps.business.attendance.sdk.dto.AttendanceShiftApplicationDto;
import com.biz.crm.cps.business.attendance.sdk.dto.AttendanceShiftPlanDto;
import com.biz.crm.cps.business.attendance.sdk.dto.ShiftApplicationConditionDto;
import com.biz.crm.cps.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.cps.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.mdm.business.user.sdk.service.UserFeignVoService;
import com.biz.crm.mdm.business.user.sdk.vo.UserVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.time.DateUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import static com.biz.crm.cps.business.attendance.sdk.common.constant.AttendanceCodeConstant.SHIFT_APPLICATION_CODE_RULE;

/**
 * 排班申请(ShiftApplication)表服务实现类
 *
 * @author dy
 * @since 2022-02-16 14:55:15
 */
@Service("AttendanceShiftApplicationService")
public class AttendanceShiftApplicationServiceImpl implements AttendanceShiftApplicationService {

  @Autowired
  private AttendanceShiftApplicationRepository attendanceShiftApplicationRepository;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired(required = false)
  private GenerateCodeService generateCodeService;
  @Autowired
  private LoginUserService loginUserService;
  @Autowired
  private AttendanceShiftPlanService attendanceShiftPlanService;
  @Autowired
  private UserFeignVoService userFeignVoService;

  /**
   * 分页查询数据
   *
   * @param pageable 分页对象
   * @param dto      查询条件对象
   * @return
   */
  @Override
  public Page<AttendanceShiftApplication> findByConditions(Pageable pageable, ShiftApplicationConditionDto dto) {
    ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    if (Objects.isNull(dto)) {
      dto = new ShiftApplicationConditionDto();
    }
    return this.attendanceShiftApplicationRepository.findByConditions(pageable, dto);
  }

  /**
   * 通过主键查询单条数据
   *
   * @param id 主键
   * @return 单条数据
   */
  @Override
  public AttendanceShiftApplication findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.attendanceShiftApplicationRepository.getById(id);
  }

  /**
   * 新增数据
   *
   * @param attendanceShiftApplicationDto 实体对象
   * @return 新增结果
   */
  @Transactional
  @Override
  public AttendanceShiftApplication create(AttendanceShiftApplicationDto attendanceShiftApplicationDto) {

    AttendanceShiftApplication attendanceShiftApplication =
        nebulaToolkitService.copyObjectByBlankList(attendanceShiftApplicationDto, AttendanceShiftApplication.class, HashSet.class, ArrayList.class);
    AttendanceShiftApplication current = this.createForm(attendanceShiftApplication);
    this.createBusinessValidate(attendanceShiftApplicationDto);
    this.attendanceShiftApplicationRepository.saveOrUpdate(current);
    List<AttendanceShiftPlanDto> shiftPlanDtos = attendanceShiftApplicationDto.getShiftPlans();
    shiftPlanDtos.stream().forEach(o->o.setEffectiveDate(o.getSchedulingDate().compareTo(DateUtils.truncate(current.getApplyDate(),Calendar.DATE)) >= 0));
    Validate.isTrue(!CollectionUtils.isEmpty(shiftPlanDtos),"排班计划详情不能为空");
    this.attendanceShiftPlanService.createBatch(current.getApplyCode(),shiftPlanDtos);
    return current;
  }

  @Override
  public AttendanceShiftApplication createForm(AttendanceShiftApplication attendanceShiftApplication) {

    LoginUserDetailsForCPS loginDetails = loginUserService.getLoginDetails(LoginUserDetailsForCPS.class);
    Validate.notNull(loginDetails,"获取当前登录信息失败");
    attendanceShiftApplication.setUserAccount(loginDetails.getConsumerCode());
    attendanceShiftApplication.setUserType(loginDetails.getUsertype());
    attendanceShiftApplication.setUserCode(loginDetails.getAccount());
    attendanceShiftApplication.setUserName(loginDetails.getConsumerName());
    // 获取组织信息
    List<UserVo> userVos = userFeignVoService.findByUserNames(Arrays.asList(loginDetails.getConsumerCode()));
    Validate.isTrue(!CollectionUtils.isEmpty(userVos),"没有找到当前登录用户信息");
    UserVo userVo = userVos.get(0);
    attendanceShiftApplication.setOrgCode(userVo.getOrgCode());
    attendanceShiftApplication.setOrgName(userVo.getOrgName());

    this.createValidate(attendanceShiftApplication);

    Date now = new Date();
    attendanceShiftApplication.setCreateTime(now);
    attendanceShiftApplication.setModifyTime(now);
    String loginAccountName = loginUserService.getLoginAccountName();
    Validate.notBlank(loginAccountName,"未能获取到当前登录用户账号");
    attendanceShiftApplication.setCreateAccount(loginAccountName);
    attendanceShiftApplication.setModifyAccount(loginAccountName);
    attendanceShiftApplication.setTenantCode(TenantUtils.getTenantCode());
    attendanceShiftApplication.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    attendanceShiftApplication.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    attendanceShiftApplication.setAuditDate(null);
    attendanceShiftApplication.setRejectedReason(null);
    attendanceShiftApplication.setAuditStatus(ShiftApplicationAuditStatusEnum.NO_AUDIT.getDictCode());
    attendanceShiftApplication.setApplyPeriod(this.findApplyPeriodByDate(attendanceShiftApplication.getApplyStartDate()));
    attendanceShiftApplication.setApplyDate(now);

    String applyCode = generateCodeService.generateCode(SHIFT_APPLICATION_CODE_RULE, 1).get(0);
    Validate.notBlank(applyCode,"生成申请编码失败");
    AttendanceShiftApplication application = this.attendanceShiftApplicationRepository.findByApplyCode(applyCode);
    Validate.isTrue(Objects.isNull(application),"班次申请编码重复");
    attendanceShiftApplication.setApplyCode(applyCode);

    return attendanceShiftApplication;
  }

  /**
   * 修改新据
   *
   * @param attendanceShiftApplicationDto 实体对象
   * @return 修改结果
   */
  @Transactional
  @Override
  public AttendanceShiftApplication update(AttendanceShiftApplicationDto attendanceShiftApplicationDto) {
    AttendanceShiftApplication attendanceShiftApplication = nebulaToolkitService.copyObjectByBlankList(attendanceShiftApplicationDto, AttendanceShiftApplication.class, HashSet.class, ArrayList.class);

    AttendanceShiftApplication current = this.updateForm(attendanceShiftApplication);
    attendanceShiftPlanService.deleteByApplyCode(current.getApplyCode());
    this.updateBusinessValidate(attendanceShiftApplicationDto);
    this.attendanceShiftApplicationRepository.saveOrUpdate(current);

    List<AttendanceShiftPlanDto> shiftPlanDtos = attendanceShiftApplicationDto.getShiftPlans();
    Validate.isTrue(!CollectionUtils.isEmpty(shiftPlanDtos),"排班计划详情不能为空");
    shiftPlanDtos.stream().forEach(o->o.setEffectiveDate(o.getSchedulingDate().compareTo(DateUtils.truncate(current.getApplyDate(),Calendar.DATE)) >= 0));
    this.attendanceShiftPlanService.createBatch(current.getApplyCode(),shiftPlanDtos);
    return current;
  }

  @Override
  public AttendanceShiftApplication updateForm(AttendanceShiftApplication attendanceShiftApplication) {

    this.updateValidate(attendanceShiftApplication);

    String shiftApplicationId = attendanceShiftApplication.getId();
    String applyCode = attendanceShiftApplication.getApplyCode();
    AttendanceShiftApplication current = null;
    if(StringUtils.isNotBlank(shiftApplicationId)){
      current = attendanceShiftApplicationRepository.findById(shiftApplicationId);
    }else if(current == null && StringUtils.isNotBlank(applyCode)){
      current = attendanceShiftApplicationRepository.findByApplyCode(applyCode);
    }
    Validate.notNull(current,"没有找到原始对象信息");
    Validate.isTrue(!ShiftApplicationAuditStatusEnum.PASS.getDictCode().equals(current.getAuditStatus()),"该申请已经审核通过，不能修改");
    Date now = new Date();
    attendanceShiftApplication.setModifyTime(now);
    String loginAccountName = loginUserService.getLoginAccountName();
    Validate.notBlank(loginAccountName,"未能获取到当前登录用户账号");
    attendanceShiftApplication.setModifyAccount(loginAccountName);

    // 待修改的申请是待审核状态
    if(ShiftApplicationAuditStatusEnum.NO_AUDIT.getDictCode().equals(current.getAuditStatus())){
      // 如果是修改为审核通过
      if(ShiftApplicationAuditStatusEnum.PASS.getDictCode().equals(attendanceShiftApplication.getAuditStatus())){
        Validate.isTrue(ShiftApplicationAuditStatusEnum.NO_AUDIT.getDictCode().equals(current.getAuditStatus()),"只能通过待审核的申请");
        current.setAuditStatus(ShiftApplicationAuditStatusEnum.PASS.getDictCode());
        current.setAuditDate(now);
      }
      // 如果是修改为审核驳回
      if(ShiftApplicationAuditStatusEnum.REJECTED.getDictCode().equals(attendanceShiftApplication.getAuditStatus())){
        current.setAuditStatus(ShiftApplicationAuditStatusEnum.REJECTED.getDictCode());
        current.setRejectedReason(attendanceShiftApplication.getRejectedReason());
        current.setAuditDate(now);
      }
    }
    // 待修改的申请是已驳回状态
    else if(ShiftApplicationAuditStatusEnum.REJECTED.getDictCode().equals(current.getAuditStatus())){
      current.setAuditDate(null);
      current.setRejectedReason(null);
      current.setAuditStatus(ShiftApplicationAuditStatusEnum.NO_AUDIT.getDictCode());
    }else {
      throw new IllegalArgumentException("审核状态异常");
    }
    return current;
  }

  /**
   * 删除数据
   *
   * @param idList 主键结合
   * @return 删除结果
   */
  @Transactional
  @Override
  public void delete(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "删除数据时，主键集合不能为空！");
    this.attendanceShiftApplicationRepository.removeByIds(idList);
  }

  /**
   * 创建验证
   *
   * @param attendanceShiftApplication
   */
  private void createValidate(AttendanceShiftApplication attendanceShiftApplication) {
    Validate.notNull(attendanceShiftApplication, "新增时，对象信息不能为空！");
    attendanceShiftApplication.setId(null);
    Validate.notNull(attendanceShiftApplication.getApplyEndDate(), "新增数据时，申请排班结束时间不能为空！");
    Validate.notNull(attendanceShiftApplication.getApplyStartDate(), "新增数据时，申请排班开始时间不能为空！");
    Validate.notBlank(attendanceShiftApplication.getUserCode(), "新增数据时， 申请导购员编码 不能为空！");
    Validate.notBlank(attendanceShiftApplication.getUserName(), "新增数据时， 申请导购员名称 不能为空！");
    Validate.notBlank(attendanceShiftApplication.getUserAccount(),"新增数据时， 申请导购员账号 不能为空！");
    Validate.notBlank(attendanceShiftApplication.getContactPhone(),"新增数据时，申请导购员联系电话不能为空");
  }

  /**
   * 修改验证
   *
   * @param attendanceShiftApplication
   */
  private void updateValidate(AttendanceShiftApplication attendanceShiftApplication) {
    Validate.notNull(attendanceShiftApplication, "修改时，对象信息不能为空！");
    Validate.isTrue(StringUtils.isNotBlank(attendanceShiftApplication.getId()) || StringUtils.isNotBlank(attendanceShiftApplication.getApplyCode()), "修改数据时，id和编码不能同时为空");
    Validate.notNull(attendanceShiftApplication.getAuditStatus(), "修改数据时， 审核状态不能为空！");
  }

  /**
   * 新增业务校验
   * @param attendanceShiftApplicationDto
   */
  private void createBusinessValidate(AttendanceShiftApplicationDto attendanceShiftApplicationDto){
    /*
      总体规则
      1.每次排班申请周期为一周，本周只能申请本周和下周的排班
      2.申请的是本周的排班，如果
      2.同一导购员在一个周期内只能申请一次
     */

    List<AttendanceShiftPlanDto> shiftPlans = attendanceShiftApplicationDto.getShiftPlans();
    Validate.isTrue(!CollectionUtils.isEmpty(shiftPlans) && shiftPlans.size() <= 7,"排班申请详情不能为空且每次最多只能申请七天的排班");

    Calendar applyStartDate = DateUtils.toCalendar(attendanceShiftApplicationDto.getApplyStartDate());
    Calendar applyEndDate = DateUtils.toCalendar(attendanceShiftApplicationDto.getApplyEndDate());

    Validate.isTrue(DateUtils.truncatedCompareTo(applyStartDate,applyEndDate,Calendar.DATE) == -1,"申请排班的开始时间必须在结束时间之前");
    Validate.isTrue(applyStartDate.get(Calendar.DAY_OF_WEEK) == Calendar.MONDAY,"申请排班开始时间只能为周一");
    Validate.isTrue(applyEndDate.get(Calendar.DAY_OF_WEEK) == Calendar.SUNDAY,"申请排班结束时间只能为周末");

    // 判断申请的时间段是否为本周或下周
    Calendar thisWeekStart = Calendar.getInstance();
    thisWeekStart.set(Calendar.DAY_OF_WEEK,Calendar.MONDAY);

    Calendar thisWeekEnd = Calendar.getInstance();
    thisWeekEnd.setFirstDayOfWeek(Calendar.MONDAY);
    thisWeekEnd.set(Calendar.DAY_OF_WEEK,Calendar.SUNDAY);

    Calendar nextWeekStart = Calendar.getInstance();
    Date date = DateUtils.addWeeks(thisWeekStart.getTime(), 1);
    nextWeekStart.setTime(date);

    LoginUserDetailsForCPS loginDetails = loginUserService.getLoginDetails(LoginUserDetailsForCPS.class);
    Validate.notNull(loginDetails,"获取当前用户登录信息失败");
    String userAccount = loginDetails.getConsumerCode();
    if(DateUtils.isSameDay(thisWeekStart,applyStartDate)){
      Validate.isTrue(DateUtils.isSameDay(applyEndDate,thisWeekEnd),"申请本周排班时,申请结束时间只能是本周末");
      // 判断本周是否已经申请过
      AttendanceShiftApplication application = attendanceShiftApplicationRepository.findByUserAccountAndApplyPeriod(userAccount, this.findApplyPeriodByDate(thisWeekStart.getTime()));
      Validate.isTrue(Objects.isNull(application),"本周已经申请过");
    }else if(DateUtils.isSameDay(nextWeekStart,applyStartDate)){
      Date nextWeekEnd = DateUtils.addWeeks(thisWeekEnd.getTime(), 1);
      Validate.isTrue(DateUtils.isSameDay(applyEndDate.getTime(),nextWeekEnd),"申请下周排班时,申请结束时间只能是下周末");
      // 判断下周是否已经申请过
      AttendanceShiftApplication application = attendanceShiftApplicationRepository.findByUserAccountAndApplyPeriod(userAccount, this.findApplyPeriodByDate(nextWeekStart.getTime()));
      Validate.isTrue(Objects.isNull(application),"下周已经申请过");
    }else {
      throw new IllegalArgumentException("只能申请本周或者下周的排班");
    }

    // 校验排班详情的日期是否在申请的开始和结束区间内
    shiftPlans.stream().forEach(o->Validate.notNull(o.getSchedulingDate(),"排班详情排班日期不能为空"));
    Set<Date> set = shiftPlans.stream()
        .filter(o -> (applyStartDate.getTime().compareTo(o.getSchedulingDate()) <= 0 && applyEndDate.getTime().compareTo(o.getSchedulingDate()) >=0))
        .map(o -> DateUtils.truncate(o.getSchedulingDate(),Calendar.DATE))
        .collect(Collectors.toSet());
    Validate.isTrue(set.size() == shiftPlans.size(),"排班日期必须在开始时间和结束时间之间且不能重复");
  }

  /**
   * 修改前校验
   * @param attendanceShiftApplicationDto
   */
  private void updateBusinessValidate(AttendanceShiftApplicationDto attendanceShiftApplicationDto){

    List<AttendanceShiftPlanDto> shiftPlans = attendanceShiftApplicationDto.getShiftPlans();
    Validate.isTrue(!CollectionUtils.isEmpty(shiftPlans) && shiftPlans.size() == 7,"排班申请详情不能为空且每次只能申请七天的排班");

    String id = attendanceShiftApplicationDto.getId();
    AttendanceShiftApplication shiftApplication = this.findById(id);
    Date applyStartDate = shiftApplication.getApplyStartDate();
    Date applyEndDate = shiftApplication.getApplyEndDate();
    Date applyDate = shiftApplication.getApplyDate();
    // 校验排班详情的日期是否在申请的开始和结束区间内
    shiftPlans.stream().forEach(o->Validate.notNull(o.getSchedulingDate(),"排班详情排班日期不能为空"));
    Set<Date> set = shiftPlans.stream()
        .filter(o -> (applyStartDate.compareTo(o.getSchedulingDate()) <= 0 && applyEndDate.compareTo(o.getSchedulingDate()) >=0))
        .map(o -> DateUtils.truncate(o.getSchedulingDate(),Calendar.DATE))
        .collect(Collectors.toSet());
    Validate.isTrue(set.size() == shiftPlans.size(),"排班日期必须在开始时间和结束时间之间且不能重复");
  }

  /**
   * 根据申请开始日期，返回申请周期 年份+周数
   * @param date
   * @return
   */
  @Override
  public String findApplyPeriodByDate(Date date){
    Validate.notNull(date,"申请开始日期不能为空");

    String format = "%04d%02d";
    Calendar calendar = DateUtils.toCalendar(date);
    calendar.setFirstDayOfWeek(Calendar.MONDAY);
    calendar.setMinimalDaysInFirstWeek(7);
    calendar.set(Calendar.DAY_OF_WEEK,Calendar.MONDAY);
    String applyPeriod = String.format(format, calendar.get(Calendar.YEAR), calendar.get(Calendar.WEEK_OF_YEAR));
    return applyPeriod;
  }

  @Override
  public AttendanceShiftApplication findByUserAccountAndApplyPeriod(String userAccount, Date applyPeriodDate) {
    if(StringUtils.isBlank(userAccount) || Objects.isNull(applyPeriodDate)){
      return null;
    }
    String applyPeriod = this.findApplyPeriodByDate(applyPeriodDate);
    return attendanceShiftApplicationRepository.findByUserAccountAndApplyPeriod(userAccount,applyPeriod);
  }

  @Override
  @Transactional(rollbackFor = Exception.class)
  public void updateAuditStatusByIdBatch(AttendanceShiftApplicationApproveDto dto) {
    Validate.notNull(dto,"审核对象不能为空");
    List<String> ids = dto.getIds();
    String auditStatus = dto.getAuditStatus();
    Date now = new Date();
    String loginAccountName = loginUserService.getLoginAccountName();
    Validate.notBlank(loginAccountName,"获取当前登录账户信息失败");
    Validate.isTrue(!CollectionUtils.isEmpty(ids),"请选中要审核的对象");
    Validate.isTrue(StringUtils.isNotBlank(auditStatus),"审核状态不能为空");
    Validate.isTrue(ShiftApplicationAuditStatusEnum.REJECTED.getDictCode().equals(auditStatus)
        || ShiftApplicationAuditStatusEnum.PASS.getDictCode().equals(auditStatus),"只能选择审核通过和驳回");
    List<AttendanceShiftApplication> applications = attendanceShiftApplicationRepository.findByIds(ids);
    Validate.isTrue(ids.size() == applications.size(),"批量审批时,未找到对应数据，请检查");
    for (AttendanceShiftApplication application : applications) {
      Validate.isTrue(ShiftApplicationAuditStatusEnum.NO_AUDIT.getDictCode().equals(application.getAuditStatus()),String.format("编码为 %s的申请已经审核,不能重复审核",application.getApplyCode()));
      application.setModifyTime(now);
      application.setModifyAccount(loginAccountName);
      application.setAuditStatus(auditStatus);
      application.setAuditDate(now);
      if(ShiftApplicationAuditStatusEnum.REJECTED.getDictCode().equals(auditStatus)){
        application.setRejectedReason(dto.getRejectedReason());
      }
    }
    attendanceShiftApplicationRepository.saveOrUpdateBatch(applications);
  }

  @Override
  public AttendanceShiftApplication findByApplyCode(String applyCode) {
    if(StringUtils.isBlank(applyCode)){
      return null;
    }
    return attendanceShiftApplicationRepository.findByApplyCode(applyCode);
  }
}