package com.biz.crm.cps.business.attendance.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.model.LoginUserDetailsForCPS;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.cps.business.attendance.local.entity.AttendanceShift;
import com.biz.crm.cps.business.attendance.local.entity.AttendanceShiftApplication;
import com.biz.crm.cps.business.attendance.local.entity.AttendanceShiftPlan;
import com.biz.crm.cps.business.attendance.local.repository.AttendanceShiftPlanRepository;
import com.biz.crm.cps.business.attendance.local.repository.AttendanceShiftRepository;
import com.biz.crm.cps.business.attendance.local.service.AttendanceShiftApplicationService;
import com.biz.crm.cps.business.attendance.local.service.AttendanceShiftPlanService;
import com.biz.crm.cps.business.attendance.sdk.common.enums.CompensatoryLeaveTypeEnum;
import com.biz.crm.cps.business.attendance.sdk.common.enums.ShiftApplicationAuditStatusEnum;
import com.biz.crm.cps.business.attendance.sdk.common.enums.ShiftPlanTypeEnum;
import com.biz.crm.cps.business.attendance.sdk.common.enums.ShiftTypeEnum;
import com.biz.crm.cps.business.attendance.sdk.dto.AttendanceShiftPlanDto;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.time.DateUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.Calendar;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 排班申请计划详情(ShiftPlan)表服务实现类
 *
 * @author dy
 * @since 2022-02-16 15:35:27
 */
@Service("AttendanceShiftPlanService")
public class AttendanceShiftPlanServiceImpl implements AttendanceShiftPlanService {

  @Autowired
  private AttendanceShiftPlanRepository attendanceShiftPlanRepository;
  @Autowired
  private AttendanceShiftRepository attendanceShiftRepository;
  @Autowired
  private AttendanceShiftApplicationService attendanceShiftApplicationService;
  @Autowired
  private LoginUserService loginUserService;

  /**
   * 分页查询数据
   * @param pageable 分页对象
   * @param attendanceShiftPlan 实体对象
   * @return
   */
  @Override
  public Page<AttendanceShiftPlan> findByConditions(Pageable pageable, AttendanceShiftPlan attendanceShiftPlan) {
    ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    if (Objects.isNull(attendanceShiftPlan)) {
      attendanceShiftPlan = new AttendanceShiftPlan();
    }
    return this.attendanceShiftPlanRepository.findByConditions(pageable, attendanceShiftPlan);
  }
  
  /**
   * 通过主键查询单条数据
   * @param id 主键
   * @return 单条数据
   */
  @Override
  public AttendanceShiftPlan findById(String id) {
    if (StringUtils.isBlank(id)) {
	  return null;
	}
    return this.attendanceShiftPlanRepository.getById(id);
  }
  
  /**
   * 新增数据
   * @param attendanceShiftPlan 实体对象
   * @return 新增结果
   */
  @Transactional
  @Override
  public AttendanceShiftPlan create(AttendanceShiftPlan attendanceShiftPlan) {
    this.createValidate(attendanceShiftPlan);
    this.attendanceShiftPlanRepository.saveOrUpdate(attendanceShiftPlan);
    return attendanceShiftPlan;
  }
  
  /**
   * 修改新据
   * @param attendanceShiftPlan 实体对象
   * @return 修改结果
   */
  @Transactional
  @Override
  public AttendanceShiftPlan update(AttendanceShiftPlan attendanceShiftPlan) {
    this.updateValidate(attendanceShiftPlan);
    this.attendanceShiftPlanRepository.saveOrUpdate(attendanceShiftPlan);
    return attendanceShiftPlan;
  }
  
  /**
   * 删除数据
   * @param idList 主键结合
   * @return 删除结果
   */
  @Transactional
  @Override
  public void delete(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "删除数据时，主键集合不能为空！");
    this.attendanceShiftPlanRepository.removeByIds(idList);
  }

  @Override
  @Transactional(rollbackFor = Exception.class)
  public void createBatch(String applyCode, List<AttendanceShiftPlanDto> attendanceShiftPlans) {
    /*
    1. 校验数据完整性
    2. 判断是否是全天调休或者节假日
    3. 冗余班次数据到排班详情内
    4. 计算排班日期的上班下班时间,调休时间，调休时长
    5. 计算工时和调休时长
     */
    // 1
    this.createValidate(attendanceShiftPlans);
    this.deleteByApplyCode(applyCode);

    Date now = new Date();
    String loginAccountName = loginUserService.getLoginAccountName();
    String tenantCode = TenantUtils.getTenantCode();

    List<AttendanceShiftPlan> current = Lists.newArrayList();
    for (AttendanceShiftPlanDto attendanceShiftPlanDto : attendanceShiftPlans) {
      AttendanceShiftPlan attendanceShiftPlan = new AttendanceShiftPlan();
      attendanceShiftPlan.setApplyCode(applyCode);
      attendanceShiftPlan.setTenantCode(tenantCode);
      attendanceShiftPlan.setCreateAccount(loginAccountName);
      attendanceShiftPlan.setCreateTime(now);
      attendanceShiftPlan.setModifyAccount(loginAccountName);
      attendanceShiftPlan.setModifyTime(now);

      // 属性复制
      BeanUtils.copyProperties(attendanceShiftPlanDto,attendanceShiftPlan);

      Integer dayOfWeek = this.getDayOfWeek(attendanceShiftPlanDto.getSchedulingDate());
      attendanceShiftPlan.setWeek(dayOfWeek);

      // 如果是排班日期在申请日期之前
      if(!attendanceShiftPlanDto.getEffectiveDate()){
        attendanceShiftPlan.setCompensatoryLeaveType(CompensatoryLeaveTypeEnum.NOT_COMPENSATORY_LEAVE.getDictCode());
        attendanceShiftPlan.setShiftPlanType(ShiftPlanTypeEnum.EXPIRY_DATE.getDictCode());
        attendanceShiftPlan.setHoliday(false);
        current.add(attendanceShiftPlan);
        continue;
      }

      // 如果是节假日或者周末且没有申请排班的
      Calendar calendar = DateUtils.toCalendar(attendanceShiftPlanDto.getSchedulingDate());
      calendar.setFirstDayOfWeek(Calendar.MONDAY);
      int week = calendar.get(Calendar.DAY_OF_WEEK);
      if((attendanceShiftPlanDto.getHoliday() || week == Calendar.SATURDAY || week == Calendar.SUNDAY) && StringUtils.isBlank(attendanceShiftPlanDto.getShiftCode())){
        attendanceShiftPlan.setShiftPlanType(ShiftPlanTypeEnum.REST_DAY.getDictCode());
        attendanceShiftPlan.setCompensatoryLeaveType(CompensatoryLeaveTypeEnum.NOT_COMPENSATORY_LEAVE.getDictCode());
        attendanceShiftPlan.setTerminalCode(null);
        attendanceShiftPlan.setTerminalName(null);
        current.add(attendanceShiftPlan);
        continue;
      }
      String shiftCode = attendanceShiftPlanDto.getShiftCode();
      AttendanceShift attendanceShift = attendanceShiftRepository.findByShiftCode(shiftCode);
      Validate.notNull(attendanceShift,"不存在的班次");
      if(ShiftTypeEnum.DAY_COMPENSATORY_LEAVE.getDictCode().equals(attendanceShift.getShiftType())){
        attendanceShiftPlanDto.setCompensatoryLeaveType(CompensatoryLeaveTypeEnum.NOT_COMPENSATORY_LEAVE.getDictCode());
      }

      // 3
      attendanceShiftPlan.setShiftCode(shiftCode);
      attendanceShiftPlan.setShiftColor(attendanceShift.getShiftColor());
      attendanceShiftPlan.setShiftName(attendanceShift.getShiftName());
      attendanceShiftPlan.setShiftType(attendanceShift.getShiftType());
      attendanceShiftPlan.setShiftDescribe(attendanceShift.getShiftDescribe());
      attendanceShiftPlan.setAcrossDay(attendanceShift.getAcrossDay());
      attendanceShiftPlan.setShiftPlanType(attendanceShift.getShiftType());
      attendanceShiftPlan.setCompensatoryLeaveLength(BigDecimal.ZERO);
      attendanceShiftPlan.setNormalWorkHours(BigDecimal.ZERO);

      // 4
      Date workStartDate = attendanceShiftPlanDto.getSchedulingDate();
      long startMinutes = DateUtils.getFragmentInMinutes(attendanceShift.getWorkStartTime(), Calendar.DATE);
      workStartDate = DateUtils.addMinutes(workStartDate, (int) startMinutes);
      attendanceShiftPlan.setWorkStartDate(workStartDate);

      // 最晚签到时间
      if(attendanceShift.getLatestSignInTime() != null){
        Date latestSignInDate = attendanceShiftPlanDto.getSchedulingDate();
        long latestSignInMinutes = DateUtils.getFragmentInMinutes(attendanceShift.getLatestSignInTime(), Calendar.DATE);
        latestSignInDate = DateUtils.addMinutes(latestSignInDate, (int) latestSignInMinutes);
        attendanceShiftPlan.setLatestSignInDate(latestSignInDate);
      }

      // 下班时间
      Date workEndDate = attendanceShiftPlanDto.getSchedulingDate();
      long endMinutes = DateUtils.getFragmentInMinutes(attendanceShift.getWorkEndTime(), Calendar.DATE);
      workEndDate = DateUtils.addMinutes(workEndDate, (int) endMinutes);
      if(attendanceShift.getAcrossDay()){
        workEndDate = DateUtils.addDays(workEndDate,1);
      }
      attendanceShiftPlan.setWorkEndDate(workEndDate);

      // 最早签退时间
      if(attendanceShift.getEarliestSignBackTime()!= null){
        Date earliestSignBackDate = attendanceShiftPlanDto.getSchedulingDate();
        if(attendanceShift.getAcrossDay()){
          earliestSignBackDate = DateUtils.addDays(earliestSignBackDate,1);
        }
        long earliestSignBackMinutes = DateUtils.getFragmentInMinutes(attendanceShift.getEarliestSignBackTime(), Calendar.DATE);
        earliestSignBackDate = DateUtils.addMinutes(earliestSignBackDate, (int) earliestSignBackMinutes);
        attendanceShiftPlan.setEarliestSignBackDate(earliestSignBackDate);
      }

      if(!CompensatoryLeaveTypeEnum.NOT_COMPENSATORY_LEAVE.getDictCode().equals(attendanceShiftPlanDto.getCompensatoryLeaveType())) {
        // 调休开始时间
        Date compensatoryLeaveStartDate = attendanceShiftPlanDto.getCompensatoryLeaveStartDate();
        Date compensatoryLeaveEndDate = attendanceShiftPlanDto.getCompensatoryLeaveEndDate();

        Validate.isTrue(compensatoryLeaveStartDate.compareTo(workStartDate) >= 0, String.format("%tF 的调休时间必须大于等于上班时间",attendanceShiftPlan.getSchedulingDate()));
        Validate.isTrue(compensatoryLeaveEndDate.compareTo(workEndDate) <= 0, String.format("%tF 的调休结束时间必须小于等于下班时间",attendanceShiftPlan.getSchedulingDate()));
        Validate.isTrue(compensatoryLeaveStartDate.before(compensatoryLeaveEndDate),String.format("%tF 的调休开始时间必须在结束时间之前",attendanceShiftPlan.getSchedulingDate()));
        attendanceShiftPlan.setCompensatoryLeaveStartDate(compensatoryLeaveStartDate);
        attendanceShiftPlan.setCompensatoryLeaveEndDate(compensatoryLeaveEndDate);
        BigDecimal compensatoryLeaveLength = this.calculateWorkHours(compensatoryLeaveStartDate, compensatoryLeaveEndDate);
        attendanceShiftPlan.setCompensatoryLeaveLength(compensatoryLeaveLength);
      }

      // 5 计算工时 工时 = 班次时长 - 调休时长
      BigDecimal shiftWorkLength = this.calculateWorkHours(workStartDate, workEndDate);
      if(ShiftPlanTypeEnum.DAY_COMPENSATORY_LEAVE.getDictCode().equals(attendanceShift.getShiftType())){
        attendanceShiftPlan.setCompensatoryLeaveLength(shiftWorkLength);
        current.add(attendanceShiftPlan);
        continue;
      }
      attendanceShiftPlan.setNormalWorkHours(shiftWorkLength.subtract(attendanceShiftPlan.getCompensatoryLeaveLength()));
      current.add(attendanceShiftPlan);
    }
    this.businessValidate(current);
    attendanceShiftPlanRepository.saveOrUpdateBatch(current);
  }

  @Override
  @Transactional
  public void deleteByApplyCode(String applyCode) {
    if(StringUtils.isBlank(applyCode)){
      return;
    }
    attendanceShiftPlanRepository.deleteByApplyCode(applyCode);
  }

  @Override
  public AttendanceShiftPlan findByCurrentUserAndDate(Date date) {
    if(Objects.isNull(date)){
      return null;
    }
    LoginUserDetailsForCPS loginUser = loginUserService.getLoginDetails(LoginUserDetailsForCPS.class);
    Validate.notNull(loginUser,"未能获取到当前登录用户信息");
    String account = loginUser.getConsumerCode();
    AttendanceShiftApplication application = attendanceShiftApplicationService.findByUserAccountAndApplyPeriod(account, date);
    if(Objects.isNull(application) || !ShiftApplicationAuditStatusEnum.PASS.getDictCode().equals(application.getAuditStatus())){
      return null;
    }
    return this.attendanceShiftPlanRepository.findByApplyCodeAndSchedulingDate(application.getApplyCode(),DateUtils.truncate(date,Calendar.DATE));
  }

  @Override
  public AttendanceShiftPlan findByApplyCodeAndSchedulingDate(String applyCode, Date schedulingDate) {
    if(StringUtils.isBlank(applyCode) || Objects.isNull(schedulingDate)){
      return null;
    }
    return this.attendanceShiftPlanRepository.findByApplyCodeAndSchedulingDate(applyCode,DateUtils.truncate(schedulingDate,Calendar.DATE));
  }

  @Override
  public AttendanceShiftPlan findByUserAccountAndSchedulingDate(String userAccount, Date schedulingDate) {
    if(StringUtils.isBlank(userAccount) || Objects.isNull(schedulingDate)){
      return null;
    }
    return attendanceShiftPlanRepository.findByUserAccountAndSchedulingDate(userAccount,DateUtils.truncate(schedulingDate,Calendar.DATE));
  }

  @Override
  public List<AttendanceShiftPlan> findBySchedulingDateAndAuditStatus(Date schedulingDate, String auditStatus) {
    if(StringUtils.isBlank(auditStatus) || Objects.isNull(schedulingDate)){
      return Lists.newArrayList();
    }
    return attendanceShiftPlanRepository.findBySchedulingDateAndAuditStatus(auditStatus,DateUtils.truncate(schedulingDate,Calendar.DATE));
  }

  /**
   * 批量保存校验
   * @param attendanceShiftPlanDtos
   */
  private void createValidate(List<AttendanceShiftPlanDto> attendanceShiftPlanDtos){
    Validate.isTrue(!CollectionUtils.isEmpty(attendanceShiftPlanDtos),"排班申请详情不能为空！");
    for (AttendanceShiftPlanDto attendanceShiftPlanDto : attendanceShiftPlanDtos) {
      Validate.notNull(attendanceShiftPlanDto.getSchedulingDate(),"排班时间不能为空");
      Validate.notNull(attendanceShiftPlanDto.getEffectiveDate(),"是否为有效日期不能为空");
      if(!attendanceShiftPlanDto.getEffectiveDate()){
        continue;
      }
      // FIXME 目前没有稳定的节假日接口，所有记录都设置为false
      attendanceShiftPlanDto.setHoliday(false);
      Validate.notNull(attendanceShiftPlanDto.getHoliday(),"排班详情，是否是节假日或周末不能为空");
      // 如果是节假日或周末，且没有申请排班的
      Calendar calendar = DateUtils.toCalendar(attendanceShiftPlanDto.getSchedulingDate());
      calendar.setFirstDayOfWeek(Calendar.MONDAY);
      int dayOfWeek = calendar.get(Calendar.DAY_OF_WEEK);
      if((attendanceShiftPlanDto.getHoliday() || dayOfWeek == Calendar.SATURDAY || dayOfWeek == Calendar.SUNDAY) && StringUtils.isBlank(attendanceShiftPlanDto.getShiftCode())){
        continue;
      }
      Validate.notBlank(attendanceShiftPlanDto.getShiftCode(),"班次编码不能为空");
      AttendanceShift shift = attendanceShiftRepository.findByShiftCode(attendanceShiftPlanDto.getShiftCode());
      Validate.notNull(shift,"未知的班次对象");
      if(ShiftTypeEnum.DAY_COMPENSATORY_LEAVE.getDictCode().equals(shift.getShiftType())){
        continue;
      }

      Validate.notBlank(attendanceShiftPlanDto.getTerminalCode(),"终端编码不能为空");
      Validate.notBlank(attendanceShiftPlanDto.getTerminalName(),"终端名称不能为空");
      Validate.notBlank(attendanceShiftPlanDto.getTerminalType(),"终端类型不能为空");
      Validate.notBlank(attendanceShiftPlanDto.getCompensatoryLeaveType(),"排班详情，调休类型不能为空");
      // 如果有调休
      if(!CompensatoryLeaveTypeEnum.NOT_COMPENSATORY_LEAVE.getDictCode().equals(attendanceShiftPlanDto.getCompensatoryLeaveType())){
        Validate.notNull(attendanceShiftPlanDto.getCompensatoryLeaveStartDate(),"选择调休后，调休开始时间不能为空");
        Validate.notNull(attendanceShiftPlanDto.getCompensatoryLeaveEndDate(),"选择调休后，调休结束时间不能为空");

        if(CompensatoryLeaveTypeEnum.ACROSS_DAY_COMPENSATORY_LEAVE.getDictCode().equals(attendanceShiftPlanDto.getCompensatoryLeaveType())){
          Validate.isTrue(shift.getAcrossDay(),"只有跨天班次才能选择跨天调休");
        }
      }else {
        attendanceShiftPlanDto.setCompensatoryLeaveStartDate(null);
        attendanceShiftPlanDto.setCompensatoryLeaveEndDate(null);
      }
    }
  }

  /**
   * 业务校验
   * @param attendanceShiftPlans
   */
  private void businessValidate(List<AttendanceShiftPlan> attendanceShiftPlans){
    Validate.isTrue(!CollectionUtils.isEmpty(attendanceShiftPlans),"排班计划详情不能为空");
    List<AttendanceShiftPlan> plans = attendanceShiftPlans.stream()
        .sorted(Comparator.comparing(AttendanceShiftPlan::getSchedulingDate))
        .collect(Collectors.toList());

    // 校验间隔的两天上班时间是否有重合
    for (int i = 0; i < plans.size() - 1; i++) {
      AttendanceShiftPlan plan = plans.get(i);
      AttendanceShiftPlan nextDayPlan = plans.get(i + 1);
      if(Objects.nonNull(plan.getWorkEndDate()) && Objects.nonNull(nextDayPlan.getWorkStartDate())){
        Validate.isTrue(plan.getWorkEndDate().before(nextDayPlan.getWorkStartDate()),String.format("%tF与%tF的工作时间有冲突!请修改后重试",plan.getSchedulingDate(),nextDayPlan.getSchedulingDate()));
      }
    }

  }

  /**
   * 创建验证
   * @param attendanceShiftPlan
   */
  private void createValidate(AttendanceShiftPlan attendanceShiftPlan) {
    Validate.notNull(attendanceShiftPlan, "新增时，对象信息不能为空！");
    attendanceShiftPlan.setId(null);
    Validate.notBlank(attendanceShiftPlan.getShiftCode(), "新增数据时， 班次编码 不能为空！");
    Validate.notNull(attendanceShiftPlan.getShiftColor(), "新增数据时， 班次标识颜色 不能为空！");
    Validate.notBlank(attendanceShiftPlan.getShiftDescribe(), "新增数据时， 班次描述 不能为空！");
    Validate.notBlank(attendanceShiftPlan.getShiftName(), "新增数据时， 班次名称 不能为空！");
    Validate.notNull(attendanceShiftPlan.getShiftType(), "新增数据时， 班次类型 不能为空！");
   
  }
  
   /**
   * 修改验证
   * @param attendanceShiftPlan
   */
  private void updateValidate(AttendanceShiftPlan attendanceShiftPlan) {
    Validate.notNull(attendanceShiftPlan, "修改时，对象信息不能为空！");
    Validate.notBlank(attendanceShiftPlan.getId(), "新增数据时，不能为空！");
    Validate.notBlank(attendanceShiftPlan.getShiftCode(), "新增数据时， 班次编码 不能为空！");
    Validate.notNull(attendanceShiftPlan.getShiftColor(), "新增数据时， 班次标识颜色 不能为空！");
    Validate.notBlank(attendanceShiftPlan.getShiftDescribe(), "新增数据时， 班次描述 不能为空！");
    Validate.notBlank(attendanceShiftPlan.getShiftName(), "新增数据时， 班次名称 不能为空！");
    Validate.notNull(attendanceShiftPlan.getShiftType(), "新增数据时， 班次类型 不能为空！");
    
  }

  /**
   * 计算工时
   * @param start
   * @param end
   * @return
   */
  private BigDecimal calculateWorkHours(Date start,Date end){
    /*
    工时计算方式：
      上班签到时间为9:35= 9+35/60=9+0.58(四舍五入)=9.58h
      下班签退时间为18:30=18+30/60=18+0.5=18.50h
      工时时长为：18.50h-9.58h=8.92h
     */

    BigDecimal s = BigDecimal.valueOf(start.getTime());
    BigDecimal e = BigDecimal.valueOf(end.getTime());
    BigDecimal hourMinutes = BigDecimal.valueOf(60 * 60 * 1000);
    BigDecimal totalMinutes = e.subtract(s);
    return totalMinutes.divide(hourMinutes,2,BigDecimal.ROUND_HALF_UP);
  }

  /**
   * 获取指定日期是星期几
   * @param date
   * @return
   */
  private Integer getDayOfWeek(Date date){
    Calendar calendar = DateUtils.toCalendar(date);
    boolean isFirstSunday = (calendar.getFirstDayOfWeek() == Calendar.SUNDAY);
    int weekDay = calendar.get(Calendar.DAY_OF_WEEK);
    if(isFirstSunday){
      weekDay = weekDay - 1;
      if(weekDay == 0){
        weekDay = 7;
      }
    }
    return weekDay;
  }
}

