package com.biz.crm.cps.business.attendance.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.model.LoginUserDetailsForCPS;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.cps.business.attendance.local.entity.AttendanceShift;
import com.biz.crm.cps.business.attendance.local.entity.AttendanceShiftOrganizationRelationship;
import com.biz.crm.cps.business.attendance.local.entity.AttendanceShiftUserRelationship;
import com.biz.crm.cps.business.attendance.local.repository.AttendanceShiftRepository;
import com.biz.crm.cps.business.attendance.local.service.AttendanceShiftOrganizationRelationshipService;
import com.biz.crm.cps.business.attendance.local.service.AttendanceShiftService;
import com.biz.crm.cps.business.attendance.local.service.AttendanceShiftUserRelationshipService;
import com.biz.crm.cps.business.attendance.sdk.common.enums.ShiftTypeEnum;
import com.biz.crm.cps.business.attendance.sdk.common.enums.ShoppingGuideSelectMethodEnum;
import com.biz.crm.cps.business.attendance.sdk.dto.AttendanceShiftDto;
import com.biz.crm.cps.business.attendance.sdk.dto.AttendanceShiftOrganizationRelationshipDto;
import com.biz.crm.cps.business.attendance.sdk.dto.AttendanceShiftUserRelationshipDto;
import com.biz.crm.cps.business.attendance.sdk.dto.ShiftConditionDto;
import com.biz.crm.cps.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.cps.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.mdm.business.user.sdk.service.UserFeignVoService;
import com.biz.crm.mdm.business.user.sdk.vo.UserVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import static com.biz.crm.cps.business.attendance.sdk.common.constant.AttendanceCodeConstant.SHIFT_CODE_RULE;

/**
 * 上班班次管理(WorkShift)表服务实现类
 *
 * @author dy
 * @since 2022-02-16 15:40:38
 */
@Service("AttendanceShiftService")
public class AttendanceShiftServiceImpl implements AttendanceShiftService {

  @Autowired
  private AttendanceShiftRepository attendanceShiftRepository;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private LoginUserService loginUserService;
  @Autowired(required = false)
  private GenerateCodeService generateCodeService;
  @Autowired
  private AttendanceShiftUserRelationshipService attendanceShiftUserRelationshipService;
  @Autowired
  private AttendanceShiftOrganizationRelationshipService attendanceShiftOrganizationRelationshipService;
  @Autowired
  private UserFeignVoService userFeignVoService;

  /**
   * 分页查询数据
   * @param pageable 分页对象
   * @param dto 查询条件对象
   * @return
   */
  @Override
  public Page<AttendanceShift> findByConditions(Pageable pageable, ShiftConditionDto dto){
    ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    if (Objects.isNull(dto)) {
      dto = new ShiftConditionDto();
    }
    return this.attendanceShiftRepository.findByConditions(pageable, dto);
  }
  
  /**
   * 通过主键查询单条数据
   * @param id 主键
   * @return 单条数据
   */
  @Override
  public AttendanceShift findById(String id) {
    if (StringUtils.isBlank(id)) {
	  return null;
	}
    return this.attendanceShiftRepository.getById(id);
  }
  
  /**
   * 新增数据
   * @param attendanceShiftDto 实体对象
   * @return 新增结果
   */
  @Transactional
  @Override
  public AttendanceShift create(AttendanceShiftDto attendanceShiftDto) {

    Validate.notNull(attendanceShiftDto,"创建班次时，班次对象不能为空");
    AttendanceShift attendanceShift = nebulaToolkitService.copyObjectByBlankList(attendanceShiftDto, AttendanceShift.class, HashSet.class, ArrayList.class);

    AttendanceShift current = this.createForm(attendanceShift);
    attendanceShiftDto.setId(null);
    this.businessValidate(attendanceShiftDto);

    attendanceShiftRepository.save(current);
    attendanceShiftDto.setShiftCode(current.getShiftCode());
    this.saveShiftRelationship(attendanceShiftDto);
    return current;
  }

  @Override
  public AttendanceShift createForm(AttendanceShift attendanceShift) {

    this.createValidate(attendanceShift);
    Date now = new Date();
    attendanceShift.setCreateTime(now);
    attendanceShift.setModifyTime(now);
    String loginAccountName = loginUserService.getLoginAccountName();
    Validate.notBlank(loginAccountName,"未能获取到当前登录用户账号");
    attendanceShift.setCreateAccount(loginAccountName);
    attendanceShift.setModifyAccount(loginAccountName);
    attendanceShift.setShiftName(this.buildShiftName(attendanceShift));

    attendanceShift.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    attendanceShift.setTenantCode(TenantUtils.getTenantCode());
    String shiftCode = generateCodeService.generateCode(SHIFT_CODE_RULE, 1).get(0);
    Validate.notBlank(shiftCode,"生成班次编码失败！");
    AttendanceShift shift = this.attendanceShiftRepository.findByShiftCode(shiftCode);
    Validate.isTrue(Objects.isNull(shift),"班次编码重复");
    attendanceShift.setShiftCode(shiftCode);
    return attendanceShift;
  }

  /**
   * 保存班次的关联信息(选择的导购或者组织)
   * @param attendanceShiftDto
   */
  @Transactional(rollbackFor = Exception.class)
  public void saveShiftRelationship(AttendanceShiftDto attendanceShiftDto){

    /*
      1.删除该班次编码关联的所有导购或组织信息
      2.根据不同的导购选择方式，保存关联对象信息
     */
    String shiftCode = attendanceShiftDto.getShiftCode();
    Validate.notBlank(shiftCode,"班次编码不能为空");
    attendanceShiftOrganizationRelationshipService.deleteByShiftCode(shiftCode);
    attendanceShiftUserRelationshipService.deleteByShiftCode(shiftCode);

    if(ShoppingGuideSelectMethodEnum.PART.getDictCode().equals(attendanceShiftDto.getShoppingGuideSelectMethod())){
      // 选择按组织机构筛选用户
      Set<AttendanceShiftOrganizationRelationshipDto> organizationRelationshipDtos = attendanceShiftDto.getShiftOrganizationRelationships();
      Validate.isTrue(!CollectionUtils.isEmpty(organizationRelationshipDtos),"班次关联的组织对象不能为空");
      Set<AttendanceShiftOrganizationRelationship> shiftOrganizationRelationshipSet =
        (Set<AttendanceShiftOrganizationRelationship>)nebulaToolkitService.copyCollectionByBlankList(organizationRelationshipDtos, AttendanceShiftOrganizationRelationshipDto.class, AttendanceShiftOrganizationRelationship.class, HashSet.class, ArrayList.class);
      shiftOrganizationRelationshipSet.forEach(o->o.setShiftCode(shiftCode));
      attendanceShiftOrganizationRelationshipService.createBatch(shiftCode,shiftOrganizationRelationshipSet);
    }else if(ShoppingGuideSelectMethodEnum.SPECIFIED.getDictCode().equals(attendanceShiftDto.getShoppingGuideSelectMethod())){
      // 指定导购
      Set<AttendanceShiftUserRelationshipDto> shiftUserRelationshipDtos = attendanceShiftDto.getShiftUserRelationships();
      Validate.isTrue(!CollectionUtils.isEmpty(shiftUserRelationshipDtos),"班次关联的导购对象不能为空");
      Set<AttendanceShiftUserRelationship> shiftUserRelationshipSet =
          (Set<AttendanceShiftUserRelationship>)nebulaToolkitService.copyCollectionByBlankList(shiftUserRelationshipDtos, AttendanceShiftUserRelationshipDto.class, AttendanceShiftUserRelationship.class, HashSet.class, ArrayList.class);
      shiftUserRelationshipSet.forEach(u->u.setShiftCode(shiftCode));
      attendanceShiftUserRelationshipService.createBatch(shiftCode,shiftUserRelationshipSet);
    }else if(ShoppingGuideSelectMethodEnum.ALL.getDictCode().equals(attendanceShiftDto.getShoppingGuideSelectMethod())){
      // 全部导购
      return;
    }else {
      throw new IllegalArgumentException("参数错误");
    }
  }

  /**
   * 修改新据
   * @param attendanceShiftDto 实体对象
   * @return 修改结果
   */
  @Transactional
  @Override
  public AttendanceShift update(AttendanceShiftDto attendanceShiftDto) {
    AttendanceShift attendanceShift = nebulaToolkitService.copyObjectByBlankList(attendanceShiftDto, AttendanceShift.class, HashSet.class, ArrayList.class);
    AttendanceShift current = this.updateFrom(attendanceShift);
    this.attendanceShiftRepository.saveOrUpdate(current);

    attendanceShiftDto.setId(current.getId());
    this.businessValidate(attendanceShiftDto);

    attendanceShiftDto.setShiftCode(current.getShiftCode());
    this.saveShiftRelationship(attendanceShiftDto);
    return current;
  }

  @Override
  public AttendanceShift updateFrom(AttendanceShift attendanceShift) {

    this.updateValidate(attendanceShift);
    String shiftId = attendanceShift.getId();
    String shiftCode = attendanceShift.getShiftCode();
    AttendanceShift current = null;
    if(StringUtils.isNotBlank(shiftId)){
      current = attendanceShiftRepository.findById(shiftId);
    }else if(current == null && StringUtils.isNotBlank(shiftCode)){
      current = attendanceShiftRepository.findByShiftCode(shiftCode);
    }
    Validate.notNull(current,"没有找到原始对象信息");

    BeanUtils.copyProperties(attendanceShift,current,"id","tenantCode", "delFlag",  "createAccount", "createTime","shiftCode","remark");
    Date now = new Date();
    current.setModifyTime(now);
    String loginAccountName = loginUserService.getLoginAccountName();
    Validate.notBlank(loginAccountName,"未能获取到当前登录用户账号");
    current.setModifyAccount(loginAccountName);
    current.setShiftName(this.buildShiftName(current));

    return current;
  }

  /**
   * 删除数据
   * @param idList 主键结合
   * @return 删除结果
   */
  @Transactional
  @Override
  public void delete(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "删除数据时，主键集合不能为空！");
    this.attendanceShiftRepository.removeByIds(idList);
  }

  @Override
  public List<AttendanceShift> findByCurrentUser() {

    LoginUserDetailsForCPS loginUser = loginUserService.getLoginDetails(LoginUserDetailsForCPS.class);
    Validate.notNull(loginUser,"获取当前登录用户信息失败");
    String account = loginUser.getConsumerCode();
    List<AttendanceShift> shiftList = this.findByUserAccount(account);
    Validate.isTrue(!CollectionUtils.isEmpty(shiftList),"当前用户暂无可用班次");
    return shiftList;
  }

  @Override
  public List<AttendanceShift> findByUserAccount(String userAccount) {

    if(StringUtils.isBlank(userAccount)){
      return Lists.newArrayList();
    }

    List<AttendanceShift> result = Lists.newArrayList();
    List<AttendanceShift> attendanceShifts = this.attendanceShiftRepository.findByShoppingGuideSelectMethodAndEnable(ShoppingGuideSelectMethodEnum.ALL.getDictCode());
    result.addAll(attendanceShifts);

    List<AttendanceShift> specifiedShifts = this.attendanceShiftRepository.findByShoppingGuideSelectMethodAndEnable(ShoppingGuideSelectMethodEnum.SPECIFIED.getDictCode());
    for (AttendanceShift attendanceShift : specifiedShifts) {
      List<AttendanceShiftUserRelationship> userRels = attendanceShiftUserRelationshipService.findByShiftCode(attendanceShift.getShiftCode());
      Set<String> userSet = userRels.stream().map(AttendanceShiftUserRelationship::getUserAccount).collect(Collectors.toSet());
      if(userSet.contains(userAccount)){
        result.add(attendanceShift);
      }
    }

    List<AttendanceShift> partShifts = this.attendanceShiftRepository.findByShoppingGuideSelectMethodAndEnable(ShoppingGuideSelectMethodEnum.PART.getDictCode());
    if(!CollectionUtils.isEmpty(partShifts)){
      List<UserVo> userVos = userFeignVoService.findByUserNames(Arrays.asList(userAccount));
      Validate.isTrue(!CollectionUtils.isEmpty(userVos),"未能找到对应的用户信息");
      UserVo userVo = userVos.get(0);
      Validate.notNull(userVo,"未能找到对应的用户信息");
      Set<String> orgCodes = Sets.newHashSet();
      if(!CollectionUtils.isEmpty(userVo.getPositionList())){
        userVo.getPositionList().stream().forEach(p->orgCodes.addAll(p.getOrgCodes()));
      }
      for (AttendanceShift attendanceShift : partShifts) {
        List<AttendanceShiftOrganizationRelationship> orgRels = attendanceShiftOrganizationRelationshipService.findByShiftCode(attendanceShift.getShiftCode());
        Set<String> orgSet = orgRels.stream().map(AttendanceShiftOrganizationRelationship::getOrgCode).collect(Collectors.toSet());
        if(!Sets.intersection(orgCodes,orgSet).isEmpty()){
          result.add(attendanceShift);
        }
      }
    }
    return result;
  }

  /**
   * 创建验证
   * @param attendanceShift
   */
  private void createValidate(AttendanceShift attendanceShift) {
    Validate.notNull(attendanceShift, "新增时，对象信息不能为空！");
    attendanceShift.setId(null);
    Validate.notBlank(attendanceShift.getEnableStatus(),"新增数据时，班次启用状态不能为空!");
    Validate.notBlank(attendanceShift.getShiftColor(), "新增数据时， 班次标识颜色 不能为空！");
    Validate.notBlank(attendanceShift.getShiftName(), "新增数据时， 班次名称 不能为空！");
    Validate.notBlank(attendanceShift.getShiftType(), "新增数据时， 班次类型 不能为空！");
    Validate.notNull(attendanceShift.getAcrossDay(),"新增数据时，是否跨天班次不能为空！");
    Validate.notBlank(attendanceShift.getShoppingGuideSelectMethod(), "新增数据时， 导购选择方式 不能为空！");
    Validate.notNull(attendanceShift.getWorkEndTime(), "新增数据时， 工作结束时间 不能为空！");
    Validate.notNull(attendanceShift.getWorkStartTime(), "新增数据时， 工作开始时间 不能为空！");
  }
  
   /**
   * 修改验证
   * @param attendanceShift
   */
  private void updateValidate(AttendanceShift attendanceShift) {
    Validate.notNull(attendanceShift, "修改数据时，对象信息不能为空！");
    Validate.isTrue(StringUtils.isNotBlank(attendanceShift.getId()) || StringUtils.isNotBlank(attendanceShift.getShiftCode()),"修改数据时，id或编码不能同时为空！");
    Validate.notBlank(attendanceShift.getShiftColor(), "修改数据时， 班次标识颜色 不能为空！");
    Validate.notBlank(attendanceShift.getEnableStatus(),"新增数据时，班次启用状态不能为空!");
    Validate.notBlank(attendanceShift.getShiftName(), "修改数据时， 班次名称 不能为空！");
    Validate.notBlank(attendanceShift.getShiftType(), "修改数据时， 班次类型 不能为空！");
    Validate.notNull(attendanceShift.getAcrossDay(),"修改数据时，是否跨天班次不能为空！");
    Validate.notBlank(attendanceShift.getShoppingGuideSelectMethod(), "修改数据时， 导购选择方式 不能为空！");
    Validate.notNull(attendanceShift.getWorkEndTime(), "修改数据时， 工作结束时间 不能为空！");
    Validate.notNull(attendanceShift.getWorkStartTime(), "修改数据时， 工作开始时间 不能为空！");
  }

  /**
   * 业务性校验
   * @param attendanceShiftDto
   */
  private void businessValidate(AttendanceShiftDto attendanceShiftDto){
    /*
      总体校验逻辑
      1.当用户选择跨天班次后，该班次的结束时间在开始时间的第二天；当用户取消选择的时候，该班次的结束时间和开始时间在同一天
      2.在没有勾选跨天班次的情况下，开始时间的工作时间小于结束时间的工作时间
      3.在勾选了跨天班次的情况下，开始时间的工作时间不和结束时间的工作时间做校验
      4.最晚签到时间一定大于上班时间，最早签退时间一定小于下班时间
      5.同一班次类型下，所选导购员不能重复
        5.1 如果一个班次类型下有一个选择了 全部导购 的班次，那么不能再创建此班次类型的班次
        5.2 如果待创建的班次选择 部分导购 且已经创建的该类型班次里面有选择部分导购的，判断所选组织机构是否有重合
        5.3 如果待创建的班次选择 部分导购 且已经创建的该类型班次里面有选择指定导购的，判断指定导购所属组织是否与已经创建的有重合
        5.4 如果待创建的班次选择 指定导购 且已经创建的该类型的班次里面有选择指定导购的，判断所选导购是否有重合
        5.5 如果待创建的班次选择 指定导购 且已经创建的该类型的班次里面有选择部分导购的，判断这部分导购
     */

    // 2
    Validate.isTrue(attendanceShiftDto.getAcrossDay() || attendanceShiftDto.getWorkStartTime().before(attendanceShiftDto.getWorkEndTime()),"非跨天班次下，工作开始时间必须小于工作结束时间");
    // 4
    Validate.isTrue(attendanceShiftDto.getLatestSignInTime() == null || attendanceShiftDto.getWorkStartTime().before(attendanceShiftDto.getLatestSignInTime()),"最晚签到时间一定大于上班时间");
    Validate.isTrue(attendanceShiftDto.getEarliestSignBackTime() == null || attendanceShiftDto.getWorkEndTime().after(attendanceShiftDto.getEarliestSignBackTime()),"最早签退时间一定小于下班时间");
    // 5
    List<AttendanceShift> attendanceShifts = Lists.newArrayList();
    if(StringUtils.isNotBlank(attendanceShiftDto.getId())){
      attendanceShifts = attendanceShiftRepository.findByShiftTypeAndEnableAndIdNot(attendanceShiftDto.getShiftType(),attendanceShiftDto.getId());
    }else {
      attendanceShifts = attendanceShiftRepository.findByShiftTypeAndEnable(attendanceShiftDto.getShiftType());
    }
    if(CollectionUtils.isEmpty(attendanceShifts) || EnableStatusEnum.DISABLE.getCode().equals(attendanceShiftDto.getEnableStatus())){
      return;
    }
    String createSelectMethod = attendanceShiftDto.getShoppingGuideSelectMethod();
    Validate.isTrue(!ShoppingGuideSelectMethodEnum.ALL.getDictCode().equals(createSelectMethod),"该班次类型已经创建，不能选择全部导购");
    Set<String> userAccounts = Sets.newHashSet();
    Set<String> orgCodes = Sets.newHashSet();
    if(ShoppingGuideSelectMethodEnum.PART.getDictCode().equals(createSelectMethod)){
      Validate.isTrue(!CollectionUtils.isEmpty(attendanceShiftDto.getShiftOrganizationRelationships()),"选择部分导购时，选择的组织机构不能为空");
      orgCodes = attendanceShiftDto.getShiftOrganizationRelationships().stream().map(AttendanceShiftOrganizationRelationshipDto::getOrgCode).collect(Collectors.toSet());
    }
    if(ShoppingGuideSelectMethodEnum.SPECIFIED.getDictCode().equals(createSelectMethod)){
      Validate.isTrue(!CollectionUtils.isEmpty(attendanceShiftDto.getShiftUserRelationships()),"选择指定导购时，选择的导购用户不能为空");
      userAccounts = attendanceShiftDto.getShiftUserRelationships().stream().map(AttendanceShiftUserRelationshipDto::getUserAccount).collect(Collectors.toSet());
    }
    for (AttendanceShift attendanceShift : attendanceShifts) {
      String selectMethod = attendanceShift.getShoppingGuideSelectMethod();
      // 5.1
      Validate.isTrue(!ShoppingGuideSelectMethodEnum.ALL.getDictCode().equals(selectMethod),"该班次类型已经创建，且对所有导购有效");
      if(ShoppingGuideSelectMethodEnum.PART.getDictCode().equals(createSelectMethod)){
        if(ShoppingGuideSelectMethodEnum.PART.getDictCode().equals(selectMethod)){
          // 5.2
          List<AttendanceShiftOrganizationRelationship> orgRels = attendanceShiftOrganizationRelationshipService.findByShiftCode(attendanceShift.getShiftCode());
          Set<String> currentOrgCodeSets = orgRels.stream().map(AttendanceShiftOrganizationRelationship::getOrgCode).collect(Collectors.toSet());
          Validate.isTrue(Sets.intersection(orgCodes, currentOrgCodeSets).isEmpty(),String.format("选择的组织机构与已经创建的 %s 班次选择的组织机构有重复",attendanceShift.getShiftCode()));
        }
        // 5.3
        if (ShoppingGuideSelectMethodEnum.SPECIFIED.getDictCode().equals(selectMethod)){
          List<AttendanceShiftUserRelationship> userRels = attendanceShiftUserRelationshipService.findByShiftCode(attendanceShift.getShiftCode());
          List<String> accounts = new ArrayList<>();
          userRels.stream().forEach(u->accounts.add(u.getUserAccount()));
          List<UserVo> userVos = this.userFeignVoService.findByUserNames(accounts);
          Set<String> orgCodeSet = Sets.newHashSet();
          for (UserVo userVo : userVos) {
            if(CollectionUtils.isEmpty(userVo.getPositionList())){
              continue;
            }
            userVo.getPositionList().stream().filter(p->!CollectionUtils.isEmpty(p.getOrgCodes())).forEach(p->orgCodeSet.addAll(p.getOrgCodes()));
          }
          Validate.isTrue(Sets.intersection(orgCodeSet,orgCodes).isEmpty(),String.format("选择的组织机构与已经创建的 %s 班次选择的导购用户所属的组织机构有重复",attendanceShift.getShiftCode()));
        }
      }
      if(ShoppingGuideSelectMethodEnum.SPECIFIED.getDictCode().equals(createSelectMethod)){
        // 5.4
        if(ShoppingGuideSelectMethodEnum.SPECIFIED.getDictCode().equals(selectMethod)){
          List<AttendanceShiftUserRelationship> userRels = attendanceShiftUserRelationshipService.findByShiftCode(attendanceShift.getShiftCode());
          Set<String> currentUserAccountSets = userRels.stream().map(AttendanceShiftUserRelationship::getUserAccount).collect(Collectors.toSet());
          Validate.isTrue(Sets.intersection(currentUserAccountSets,userAccounts).isEmpty(),String.format("选择的导购用户与已经创建的 %s 班次选择的导购用户有重复",attendanceShift.getShiftCode()));
        }
        // 5.5
        if(ShoppingGuideSelectMethodEnum.PART.getDictCode().equals(selectMethod)){
          List<AttendanceShiftOrganizationRelationship> orgRels = attendanceShiftOrganizationRelationshipService.findByShiftCode(attendanceShift.getShiftCode());
          Set<String> currentOrgCodeSets = orgRels.stream().map(AttendanceShiftOrganizationRelationship::getOrgCode).collect(Collectors.toSet());
          List<UserVo> userVos = this.userFeignVoService.findByUserNames(new ArrayList<>(userAccounts));
          Set<String> orgCodeSet = Sets.newHashSet();
          for (UserVo userVo : userVos) {
            if(CollectionUtils.isEmpty(userVo.getPositionList())){
              continue;
            }
            userVo.getPositionList().stream().filter(o->!CollectionUtils.isEmpty(o.getOrgCodes())).forEach(p->orgCodeSet.addAll(p.getOrgCodes()));
          }
          Validate.isTrue(Sets.intersection(orgCodeSet,currentOrgCodeSets).isEmpty(),String.format("选择的导购用户所属的组织机构与已经创建的 %s 班次选择的组织机构有重复",attendanceShift.getShiftCode()));
        }
      }
    }
  }

  /**
   * 拼接构造班次名称 班次类型名称 + 工作时间段 + (跨天)
   * @param attendanceShift
   * @return
   */
  private String buildShiftName(AttendanceShift attendanceShift){

    StringBuilder sb = new StringBuilder();
    SimpleDateFormat format = new SimpleDateFormat("HH:mm");
    String shiftType = attendanceShift.getShiftType();
    Date workStartTime = attendanceShift.getWorkStartTime();
    Date workEndTime = attendanceShift.getWorkEndTime();
    ShiftTypeEnum shiftTypeEnum = ShiftTypeEnum.getByDictCode(shiftType);
    Validate.notNull(shiftTypeEnum,"未知的班次类型");

    sb.append(shiftTypeEnum.getValue());
    sb.append(" ");
    sb.append(format.format(workStartTime));
    sb.append("-");
    sb.append(format.format(workEndTime));
    sb.append(" ");
    if(attendanceShift.getAcrossDay()){
      sb.append("(跨天)");
    }
    return sb.toString();
  }
}

