package com.biz.crm.cps.business.capital.local.service.internal;


import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.cps.business.capital.local.entity.CapitalRecharge;
import com.biz.crm.cps.business.capital.local.repository.CapitalRechargeRepository;
import com.biz.crm.cps.business.capital.local.service.CapitalRechargeService;
import com.biz.crm.cps.business.capital.sdk.common.constant.CapitalCodeConstant;
import com.biz.crm.cps.business.capital.sdk.common.enums.DataSourceEnum;
import com.biz.crm.cps.business.capital.sdk.common.enums.PushStatusEnum;
import com.biz.crm.cps.business.capital.sdk.common.enums.RechargeAuditEnum;
import com.biz.crm.cps.business.capital.sdk.common.enums.RechargeStatusEnum;
import com.biz.crm.cps.business.capital.sdk.dto.RechargeQueryDto;
import com.biz.crm.cps.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.cps.business.common.sdk.enums.EnableStatusEnum;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.Date;
import java.util.List;
import java.util.Objects;

/**
 * 充值信息(CapitalRecharge)表服务实现类
 *
 * @author makejava
 * @since 2022-02-18 16:40:38
 */
@Service("capitalRechargeService")
public class CapitalRechargeServiceImpl implements CapitalRechargeService {

  @Autowired
  private CapitalRechargeRepository capitalRechargeRepository;
  @Autowired
  private GenerateCodeService generateCodeService;
  @Autowired
  private LoginUserService loginUserService;

  /**
   * 分页查询数据
   *
   * @param pageable 分页对象
   * @param dto      查询dto
   * @return
   */
  @Override
  public Page<CapitalRecharge> findByConditions(Pageable pageable, RechargeQueryDto dto) {
    ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    if (Objects.isNull(dto)) {
      dto = new RechargeQueryDto();
    }
    return this.capitalRechargeRepository.findByConditions(pageable, dto);
  }

  /**
   * 通过主键查询单条数据
   *
   * @param id 主键
   * @return 单条数据
   */
  @Override
  public CapitalRecharge findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.capitalRechargeRepository.getById(id);
  }

  /**
   * 新增数据
   *
   * @param capitalRecharge 充值信息
   * @return 新增结果
   */
  @Transactional
  @Override
  public CapitalRecharge create(CapitalRecharge capitalRecharge) {
    Validate.notNull(capitalRecharge, "新增时，对象信息不能为空！");
    this.createForm(capitalRecharge);
    this.createValidate(capitalRecharge);
    this.capitalRechargeRepository.saveOrUpdate(capitalRecharge);
    return capitalRecharge;
  }

  @Transactional
  @Override
  public CapitalRecharge createForm(CapitalRecharge capitalRecharge){
    Validate.notNull(capitalRecharge, "新增时，对象信息不能为空！");
    if (Objects.nonNull(this.generateCodeService)) {
      String rechargeCode = this.generateCodeService.generateCode(StringUtils.join(CapitalCodeConstant.CAPITAL_RECHARGE_CODE, DateFormatUtils.format(new Date(), "yyyyMMdd")), 1).get(0);
      Validate.isTrue(null == this.capitalRechargeRepository.findByRechargeCodeAndTenantCode(rechargeCode, TenantUtils.getTenantCode()),
          "新增充值信息时,充值信息编号重复");
      capitalRecharge.setRechargeCode(rechargeCode);
    }
    capitalRecharge.setTenantCode(TenantUtils.getTenantCode());
    Date date = new Date();
    String loginAccountName = loginUserService.getLoginAccountName();
    capitalRecharge.setCreateTime(date);
    capitalRecharge.setCreateAccount(loginAccountName);
    capitalRecharge.setModifyTime(date);
    capitalRecharge.setModifyAccount(loginAccountName);
    capitalRecharge.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    capitalRecharge.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    capitalRecharge.setRechargeStatus(RechargeStatusEnum.WAIT.getDictCode());
    capitalRecharge.setAuditStatus(RechargeAuditEnum.WAIT.getDictCode());
    capitalRecharge.setPushStatus(PushStatusEnum.PUSH_WAIT.getDictCode());
    if (null == capitalRecharge.getDataSource()){
      capitalRecharge.setDataSource(DataSourceEnum.CPS_ADD.getDictCode());
    }
    return capitalRecharge;
  }

  /**
   * 修改新据
   *
   * @param capitalRecharge 实体对象
   * @return 修改结果
   */
  @Transactional
  @Override
  public CapitalRecharge update(CapitalRecharge capitalRecharge) {
    this.updateValidate(capitalRecharge);
    this.capitalRechargeRepository.saveOrUpdate(capitalRecharge);
    return capitalRecharge;
  }

  /**
   * 删除数据
   *
   * @param idList 主键结合
   * @return 删除结果
   */
  @Transactional
  @Override
  public void delete(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "删除数据时，主键集合不能为空！");
    this.capitalRechargeRepository.removeByIds(idList);
  }

  @Override
  @Transactional
  public List<CapitalRecharge> createBatch(List<CapitalRecharge> capitalRecharges) {
    Validate.notEmpty(capitalRecharges,"批量保存数据时,数据集合不能为空");
    for (CapitalRecharge capitalRecharge : capitalRecharges) {
      this.createForm(capitalRecharge);
      this.createValidate(capitalRecharge);
    }
    this.capitalRechargeRepository.saveBatch(capitalRecharges);
    return capitalRecharges;
  }

  @Override
  public CapitalRecharge findByRechargeCode(String rechargeCode) {
    if (StringUtils.isBlank(rechargeCode)){
      return null;
    }
    return this.capitalRechargeRepository.findByRechargeCodeAndTenantCode(rechargeCode,TenantUtils.getTenantCode());
  }

  @Override
  @Transactional
  public void updatePushStatusByRechargeCodes(List<String> rechargeCodes) {
    Validate.notEmpty(rechargeCodes, "根据充值编码更改推送状态时，充值编码不能为空！");
    List<CapitalRecharge> capitalRecharges = capitalRechargeRepository.findByRechargeCodesAndTenantCode(rechargeCodes, TenantUtils.getTenantCode());
    Validate.notEmpty(capitalRecharges, "根据充值编码更改推送状态时，未查询到充值信息");
    for (CapitalRecharge capitalRecharge : capitalRecharges) {
      capitalRecharge.setPushStatus(PushStatusEnum.PUSH_SUCCESS.getDictCode());
      capitalRecharge.setPushTime(new Date());
    }
    capitalRechargeRepository.saveOrUpdateBatch(capitalRecharges);
  }

  /**
   * 创建验证
   *
   * @param capitalRecharge
   */
  private void createValidate(CapitalRecharge capitalRecharge) {
    capitalRecharge.setId(null);
    Validate.notBlank(capitalRecharge.getCreateAccount(), "新增数据时，创建人账号不能为空！");
    Validate.notNull(capitalRecharge.getCreateTime(), "新增数据时，创建时间不能为空！");
    Validate.notBlank(capitalRecharge.getModifyAccount(), "新增数据时，更新人账号不能为空！");
    Validate.notNull(capitalRecharge.getRechargeAmount(), "新增数据时， 充值金额 不能为空！");
    Validate.isTrue(capitalRecharge.getRechargeAmount().compareTo(BigDecimal.ZERO) > 0,"新增数据时， 充值金额 必须大于0");
    Validate.notBlank(capitalRecharge.getRechargeCode(), "新增数据时， 充值编码 不能为空！");
    Validate.notBlank(capitalRecharge.getRechargeSource(), "新增数据时， 充值来源 不能为空！");
    Validate.notBlank(capitalRecharge.getRechargeStatus(), "新增数据时， 充值状态 不能为空！");
    Validate.notBlank(capitalRecharge.getRechargeTipCode(), "新增数据时， 充值提示编码 不能为空！");
    Validate.notBlank(capitalRecharge.getAccountCode(), "新增数据时， 充值账号编码 不能为空！");
    Validate.notBlank(capitalRecharge.getAccountName(), "新增数据时， 充值账号名称 不能为空！");
  }

  /**
   * 修改验证
   *
   * @param capitalRecharge
   */
  private void updateValidate(CapitalRecharge capitalRecharge) {
    Validate.notNull(capitalRecharge, "修改时，对象信息不能为空！");
    Validate.notBlank(capitalRecharge.getId(), "新增数据时，不能为空！");
    Validate.notBlank(capitalRecharge.getCreateAccount(), "新增数据时，创建人账号不能为空！");
    Validate.notNull(capitalRecharge.getCreateTime(), "新增数据时，创建时间不能为空！");
    Validate.notBlank(capitalRecharge.getModifyAccount(), "新增数据时，更新人账号不能为空！");
    Validate.notNull(capitalRecharge.getRechargeAmount(), "新增数据时， 充值金额 不能为空！");
    Validate.isTrue(capitalRecharge.getRechargeAmount().compareTo(BigDecimal.ZERO) > 0,"新增数据时， 充值金额 必须大于0");
    Validate.notBlank(capitalRecharge.getRechargeCode(), "新增数据时， 充值编码 不能为空！");
    Validate.notBlank(capitalRecharge.getRechargeSource(), "新增数据时， 充值来源 不能为空！");
    Validate.notBlank(capitalRecharge.getRechargeStatus(), "新增数据时， 充值状态 不能为空！");
    Validate.notBlank(capitalRecharge.getRechargeTipCode(), "新增数据时， 充值提示编码 不能为空！");
    Validate.notBlank(capitalRecharge.getAccountCode(), "新增数据时， 充值账号编码 不能为空！");
    Validate.notBlank(capitalRecharge.getAccountName(), "新增数据时， 充值账号名称 不能为空！");
  }
}

