package com.biz.crm.cps.business.capital.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.model.LoginUserDetailsForCPS;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.cps.business.capital.local.entity.CapitalRecharge;
import com.biz.crm.cps.business.capital.local.repository.CapitalRechargeVoRepository;
import com.biz.crm.cps.business.capital.local.service.CapitalRechargeService;
import com.biz.crm.cps.business.capital.sdk.dto.CapitalRechargeDto;
import com.biz.crm.cps.business.capital.sdk.dto.CapitalRechargeInfoDto;
import com.biz.crm.cps.business.capital.sdk.dto.CpsRaiseRechargeAuditDto;
import com.biz.crm.cps.business.capital.sdk.dto.CpsRaiseRechargeDto;
import com.biz.crm.cps.business.capital.sdk.dto.RechargeAccountDto;
import com.biz.crm.cps.business.capital.sdk.dto.RechargeDto;
import com.biz.crm.cps.business.capital.sdk.dto.RechargeQueryDto;
import com.biz.crm.cps.business.capital.sdk.dto.UserAccountRechargeDto;
import com.biz.crm.cps.business.capital.sdk.service.CapitalRechargeVoService;
import com.biz.crm.cps.business.capital.sdk.service.observer.RechargeServiceObserver;
import com.biz.crm.cps.business.capital.sdk.vo.CapitalRechargeVo;
import com.biz.crm.cps.external.tax.raise.sdk.dto.base.MerchantAccountBatchDto;
import com.biz.crm.cps.external.tax.raise.sdk.dto.base.MerchantAccountDto;
import com.biz.crm.cps.external.tax.raise.sdk.service.recharge.TaxRaiseRechargeAccountVoService;
import com.biz.crm.cps.external.tax.raise.sdk.vo.recharge.TaxRaiseRechargeAccountVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 充值信息服务
 *
 * @Author: wjx
 * @Date: 2022/2/21 16:58
 */
@Service("capitalRechargeVoService")
public class CapitalRechargeVoServiceImpl implements CapitalRechargeVoService {

  @Autowired
  private CapitalRechargeService capitalRechargeService;
  @Autowired
  private CapitalRechargeVoRepository capitalRechargeVoRepository;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private LoginUserService loginUserService;
  @Autowired(required = false)
  private List<RechargeServiceObserver> rechargeServiceObserverList;
  @Autowired
  private TaxRaiseRechargeAccountVoService taxRaiseRechargeAccountVoService;

  @Override
  public Page<CapitalRechargeVo> findByConditions(Pageable pageable, RechargeQueryDto dto) {
    ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    if (Objects.isNull(dto)) {
      dto = new RechargeQueryDto();
    }
    //获取商户id
    Page<CapitalRechargeVo> page = this.capitalRechargeVoRepository.findByConditions(pageable, dto);
    //赋值账户详情信息
    if (!CollectionUtils.isEmpty(page.getRecords())){
      List<CapitalRechargeVo> capitalRechargeVos =(List<CapitalRechargeVo>) nebulaToolkitService.copyCollectionByBlankList(page.getRecords(), CapitalRechargeVo.class, CapitalRechargeVo.class, LinkedHashSet.class, ArrayList.class);
      List<String> merchantUserAccounts = capitalRechargeVos.stream().map(CapitalRechargeVo::getMerchantUserAccount).distinct().collect(Collectors.toList());
      List<TaxRaiseRechargeAccountVo> accountVos = this.findAccountByAccountCodes(merchantUserAccounts);
      if (!CollectionUtils.isEmpty(accountVos)){
        Map<String, TaxRaiseRechargeAccountVo> map = accountVos.stream()
            .collect(Collectors.toMap(TaxRaiseRechargeAccountVo::getMerchantUserAccount, Function.identity()));
        for (CapitalRechargeVo capitalRechargeVo : capitalRechargeVos) {
          if (map.containsKey(capitalRechargeVo.getMerchantUserAccount())){
            TaxRaiseRechargeAccountVo accountVo = map.get(capitalRechargeVo.getMerchantUserAccount());
            this.handleCapitalRechargeVo(capitalRechargeVo,accountVo);
          }
        }
      }
      page.setRecords(capitalRechargeVos);
    }
    return page;
  }

  @Override
  public Page<CapitalRechargeVo> findCurrentUserByConditions(Pageable pageable, RechargeQueryDto dto) {
    ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    if (Objects.isNull(dto)) {
      dto = new RechargeQueryDto();
    }
    LoginUserDetailsForCPS loginUserDetailsForCPS = loginUserService.getLoginDetails(LoginUserDetailsForCPS.class);
    Validate.notNull(loginUserDetailsForCPS, "未获取到登录信息");
    dto.setAccountCode(loginUserDetailsForCPS.getAccount());
    Page<CapitalRechargeVo> page = this.capitalRechargeVoRepository.findByConditions(pageable, dto);
    return page;
  }

  @Override
  public CapitalRechargeVo findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    CapitalRechargeVo capitalRechargeVo = new CapitalRechargeVo();
    CapitalRecharge capitalRecharge = this.capitalRechargeService.findById(id);
    if (null == capitalRecharge){
      return capitalRechargeVo;
    }
    capitalRechargeVo = this.nebulaToolkitService.copyObjectByBlankList(capitalRecharge, CapitalRechargeVo.class, LinkedHashSet.class, ArrayList.class);
    //查询税筹充值详细信息
    MerchantAccountDto accountDto = new MerchantAccountDto();
    accountDto.setMerchantUserAccount(capitalRecharge.getMerchantUserAccount());
    TaxRaiseRechargeAccountVo accountVo = taxRaiseRechargeAccountVoService.findByMerchantAccountDto(accountDto);
    if (null != accountVo){
      this.handleCapitalRechargeVo(capitalRechargeVo,accountVo);
    }
    return capitalRechargeVo;
  }

  @Override
  public void create(RechargeDto rechargeDto) {
    this.createValidateDto(rechargeDto);
    List<CapitalRecharge> capitalRecharges = Lists.newArrayList();
    //查询用户账户
    List<String> merchantUserAccounts = rechargeDto.getRechargeAccounts().stream().map(RechargeAccountDto::getMerchantUserAccount).collect(Collectors.toList());
    List<TaxRaiseRechargeAccountVo> accountVos = this.findAccountByAccountCodes(merchantUserAccounts);
    Validate.notEmpty(accountVos,"根据用户账户集合未查询到用户账户");
    //充值账户的名称是充值合同中的企业名称
    Map<String, String> accountNameMap = accountVos.stream()
        .collect(Collectors.toMap(TaxRaiseRechargeAccountVo::getMerchantUserAccount, TaxRaiseRechargeAccountVo::getCompanyName));
    for (RechargeAccountDto rechargeAccount : rechargeDto.getRechargeAccounts()) {
      CapitalRecharge capitalRecharge = this.nebulaToolkitService.copyObjectByBlankList(rechargeDto, CapitalRecharge.class, LinkedHashSet.class, ArrayList.class);
      capitalRecharge.setMerchantUserAccount(rechargeAccount.getMerchantUserAccount());
      capitalRecharge.setAccountCode(rechargeAccount.getMerchantUserAccount());
      capitalRecharge.setRechargeAmount(rechargeAccount.getRechargeAmount());
      capitalRecharge.setAccountName(accountNameMap.get(rechargeAccount.getMerchantUserAccount()));
      capitalRecharges.add(capitalRecharge);
    }
    List<CapitalRecharge> saveCapitalRecharges = capitalRechargeService.createBatch(capitalRecharges);
    List<CapitalRechargeDto> capitalRechargeDtos = (List<CapitalRechargeDto>)this.nebulaToolkitService
        .copyCollectionByBlankList(saveCapitalRecharges, CapitalRecharge.class, CapitalRechargeDto.class, HashSet.class, LinkedList.class);
    CapitalRechargeInfoDto capitalRechargeInfoDto = new CapitalRechargeInfoDto();
    capitalRechargeInfoDto.setRechargeKey(rechargeDto.getRechargeKey());
    capitalRechargeInfoDto.setCapitalRechargeDtos(capitalRechargeDtos);
    for (RechargeServiceObserver rechargeServiceObserver : rechargeServiceObserverList) {
      rechargeServiceObserver.onRecharge(capitalRechargeInfoDto);
    }
  }

  @Override
  public CapitalRechargeVo findByRechargeCode(String rechargeCode) {
    if (StringUtils.isBlank(rechargeCode)) {
      return null;
    }
    CapitalRecharge capitalRecharge = capitalRechargeService.findByRechargeCode(rechargeCode);
    return this.nebulaToolkitService.copyObjectByBlankList(capitalRecharge, CapitalRechargeVo.class, LinkedHashSet.class, ArrayList.class);
  }

  @Override
  @Transactional
  public void updatePushStatusByRechargeCodes(List<String> rechargeCodes) {
    Validate.notEmpty(rechargeCodes, "根据充值编码更改推送状态时，充值编码不能为空！");
    this.capitalRechargeService.updatePushStatusByRechargeCodes(rechargeCodes);
  }

  @Override
  @Transactional
  public void createByCurrentUser(UserAccountRechargeDto dto) {
    this.createValidateUserAccountRechargeDto(dto);
    RechargeDto rechargeDto = nebulaToolkitService.copyObjectByBlankList(dto, RechargeDto.class, LinkedHashSet.class, ArrayList.class);
    List<RechargeAccountDto> rechargeAccounts = Lists.newArrayList();
    LoginUserDetailsForCPS loginUserDetailsForCPS = loginUserService.getLoginDetails(LoginUserDetailsForCPS.class);
    Validate.notNull(loginUserDetailsForCPS, "未获取到登录信息");
    RechargeAccountDto rechargeAccountDto = new RechargeAccountDto();
    rechargeAccountDto.setAccountCode(loginUserDetailsForCPS.getAccount());
    rechargeAccountDto.setMerchantUserAccount(loginUserDetailsForCPS.getAccount());
    rechargeAccountDto.setRechargeAmount(dto.getRechargeAmount());
    rechargeAccounts.add(rechargeAccountDto);
    rechargeDto.setRechargeAccounts(rechargeAccounts);
    this.create(rechargeDto);
  }

  @Override
  @Transactional
  public void updateAuditStatusByAuditDto(CpsRaiseRechargeAuditDto dto) {
    Validate.notNull(dto,"税筹平台充值审核状态同步时,请求参数不能为空");
    Validate.isTrue(StringUtils.isNotBlank(dto.getRechargeCode()) || (StringUtils.isNotBlank(dto.getPushCode())),
        "税筹平台充值审核状态同步时,充值编号不能为空");
    Validate.notBlank(dto.getAuditStatus(),"税筹平台充值审核状态同步时,审核状态不能为空");
    Validate.notBlank(dto.getRechargeStatus(),"税筹平台充值审核状态同步时,充值状态不能为空");
    if (StringUtils.isNotBlank(dto.getPushCode())){
      capitalRechargeVoRepository.updateStatusByPushCode(dto.getPushCode(),dto.getAuditStatus(),dto.getRechargeStatus());
    }
    if (StringUtils.isNotBlank(dto.getRechargeCode())){
      capitalRechargeVoRepository.updateStatusByRechargeCode(dto.getRechargeCode(),dto.getAuditStatus(),dto.getRechargeStatus());
    }
  }

  @Override
  @Transactional
  public void createBatch(CpsRaiseRechargeDto dto) {
    Validate.notEmpty(dto.getRechargeDtos(),"批量新增充值信息时,充值信息集合不能为空");
    List<CapitalRecharge> capitalRecharges = (List<CapitalRecharge>)nebulaToolkitService
        .copyCollectionByBlankList(dto.getRechargeDtos(), CapitalRechargeDto.class, CapitalRecharge.class, LinkedHashSet.class, ArrayList.class);
    List<String> merchantUserAccounts = capitalRecharges.stream().map(CapitalRecharge::getMerchantUserAccount).distinct().collect(Collectors.toList());
    List<TaxRaiseRechargeAccountVo> accountVos = this.findAccountByAccountCodes(merchantUserAccounts);
    Validate.notEmpty(accountVos,"根据用户账户集合未查询到用户账户");
    //充值账户的名称是充值合同中的企业名称
    Map<String, String> accountNameMap = accountVos.stream()
        .collect(Collectors.toMap(TaxRaiseRechargeAccountVo::getMerchantUserAccount, TaxRaiseRechargeAccountVo::getCompanyName));
    for (CapitalRecharge capitalRecharge : capitalRecharges) {
      capitalRecharge.setAccountName(accountNameMap.get(capitalRecharge.getMerchantUserAccount()));
      capitalRecharge.setPushCode(capitalRecharge.getRechargeCode());
    }
    capitalRechargeService.createBatch(capitalRecharges);
  }

  /**
   * 创建验证dto
   *
   * @param rechargeDto
   */
  private void createValidateDto(RechargeDto rechargeDto) {
    Validate.notNull(rechargeDto, "新增时，对象信息不能为空！");
    Validate.notBlank(rechargeDto.getRechargeSource(), "新增数据时， 充值来源 不能为空！");
    Validate.notBlank(rechargeDto.getRechargeTipCode(), "新增数据时， 充值提示编码 不能为空！");
    Validate.notEmpty(rechargeDto.getRechargeAccounts(), "新增数据时,充值账户信息不能为空");
    Validate.notBlank(rechargeDto.getRechargeKey(), "新增数据时， 充值类型key 不能为空！");
    for (RechargeAccountDto rechargeAccountDto : rechargeDto.getRechargeAccounts()) {
      Validate.notBlank(rechargeAccountDto.getAccountCode(), "新增数据时， 充值账户编码 不能为空！");
      Validate.notNull(rechargeAccountDto.getRechargeAmount(), "新增数据时， 充值金额 不能为空！");
      rechargeAccountDto.setMerchantUserAccount(rechargeAccountDto.getAccountCode());
    }
    Validate.notEmpty(this.rechargeServiceObserverList, "充值模块不存在");
  }

  /**
   * 创建验证dto
   *
   * @param dto
   */
  private void createValidateUserAccountRechargeDto(UserAccountRechargeDto dto) {
    Validate.notNull(dto, "新增时，对象信息不能为空！");
    Validate.notBlank(dto.getRechargeSource(), "新增数据时， 充值来源 不能为空！");
    Validate.notBlank(dto.getRechargeTipCode(), "新增数据时， 充值提示编码 不能为空！");
    Validate.notNull(dto.getRechargeAmount(), "新增数据时， 充值金额 不能为空！");
  }

  /**
   * 赋值充值信息账户详情信息
   * @param rechargeVo
   * @param accountVo
   */
  private void handleCapitalRechargeVo(CapitalRechargeVo rechargeVo,TaxRaiseRechargeAccountVo accountVo) {
    rechargeVo.setAccountName(accountVo.getCompanyName());
    rechargeVo.setCompanyName(accountVo.getCompanyName());
    rechargeVo.setSocialCreditCode(accountVo.getOrganization());
    rechargeVo.setBankCardNo(accountVo.getBankCardNo());
    rechargeVo.setRlegalPersonName(accountVo.getLegalName());
    rechargeVo.setCustomerId(accountVo.getCustomerId());
  }

  /**
   * 根据cps用户编码查询税筹用户信息
   * @param accountCodes
   * @return
   */
  private List<TaxRaiseRechargeAccountVo> findAccountByAccountCodes(List<String> accountCodes){
    MerchantAccountBatchDto merchantAccountBatchDto = new MerchantAccountBatchDto();
    merchantAccountBatchDto.setMerchantUserAccounts(accountCodes);
    return taxRaiseRechargeAccountVoService.findByMerchantAccountBatchDto(merchantAccountBatchDto);
  }
}
