package com.biz.crm.cps.business.cash.local.service.internal;

import com.biz.crm.business.common.sdk.model.LoginUserDetailsForCPS;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.cps.business.cash.local.entity.CashConfiguration;
import com.biz.crm.cps.business.cash.local.repository.CashConfigurationRepository;
import com.biz.crm.cps.business.cash.local.service.CashConfigurationService;
import com.biz.crm.cps.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.cps.business.common.sdk.enums.EnableStatusEnum;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.Date;

/**
 * @author hecheng
 * @description: 兑付配置实现
 * @date 2021/8/26 下午2:49
 */
@Service("CashConfigurationServiceImpl")
public class CashConfigurationServiceImpl implements CashConfigurationService {

  @Autowired(required = false)
  private LoginUserService loginUserService;

  @Autowired
  private CashConfigurationRepository cashConfigurationRepository;

  @Override
  public CashConfiguration findBySignatoryCodeAndSignatoryType(String signatoryCode, String signatoryType) {
    if (StringUtils.isAnyBlank(signatoryCode,signatoryType)) {
      return null;
    }
    return this.cashConfigurationRepository.findBySignatoryCodeAndSignatoryType(signatoryCode, signatoryType);
  }

  @Transactional
  @Override
  public CashConfiguration create(CashConfiguration cashConfig) {
    CashConfiguration current = this.createForm(cashConfig);
    this.cashConfigurationRepository.save(current);
    return current;
  }

  /**
   * 构建数据
   *
   * @param cashConfig
   * @return
   */
  private CashConfiguration createForm(CashConfiguration cashConfig) {
    /*
     * 对静态模型的保存操作过程为：
     * 1、如果当前模型对象不是主模型
     *  1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     *  1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     *  1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     * 2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *   2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *   2.3.2、以及验证每个分组的OneToMany明细信息
     * */
    Date now = new Date();
    String account = loginUserService.getLoginDetails(LoginUserDetailsForCPS.class).getAccount();
    cashConfig.setTenantCode(TenantUtils.getTenantCode());
    cashConfig.setCreateAccount(account);
    cashConfig.setCreateTime(now);
    cashConfig.setModifyAccount(account);
    cashConfig.setModifyTime(now);
    cashConfig.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    cashConfig.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    this.createValidation(cashConfig);
    return cashConfig;
  }

  /**
   * 在创建一个新的cashConfig模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(CashConfiguration cashConfig) {
    Validate.notNull(cashConfig, "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    cashConfig.setId(null);
    String signatoryCode = cashConfig.getSignatoryCode();
    String signatoryType = cashConfig.getSignatoryType();
    Validate.notBlank(cashConfig.getTenantCode(), "添加信息时，租户编号不能为空！");
    Validate.notBlank(signatoryCode, "添加信息时，签约者编码不能为空！");
    Validate.notBlank(signatoryType, "添加信息时，签约者类型不能为空！");
    Validate.notBlank(cashConfig.getSignatoryName(), "添加信息时，签约者名称不能为空！");
    Validate.notBlank(cashConfig.getCashMethodKey(), "添加信息时，发放类型标志不能为空！");
    Validate.notBlank(cashConfig.getCashMethodName(), "添加信息时，发放类型名称不能为空！");
    CashConfiguration old = this.cashConfigurationRepository.findBySignatoryCodeAndSignatoryType(signatoryCode, signatoryType);
    Validate.isTrue(old == null, "添加信息时，签约者已存在兑现服务合同");
  }
}
