package com.biz.crm.cps.business.cash.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.cps.business.cash.local.entity.CashRecord;
import com.biz.crm.cps.business.cash.local.repository.CashRecordRepository;
import com.biz.crm.cps.business.cash.local.service.CashRecordService;
import com.biz.crm.cps.business.cash.sdk.common.constant.CashRecordConstant;
import com.biz.crm.cps.business.cash.sdk.common.enums.CashPushStatusEnum;
import com.biz.crm.cps.business.cash.sdk.common.enums.CashStatusEnum;
import com.biz.crm.cps.business.cash.sdk.dto.CashPushDto;
import com.biz.crm.cps.business.cash.sdk.dto.CashRecordDto;
import com.biz.crm.cps.business.cash.sdk.register.CashPayRegister;
import com.biz.crm.cps.business.cash.sdk.vo.CashRecordVo;
import com.biz.crm.cps.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.cps.business.common.sdk.enums.EnableStatusEnum;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * @author hecheng
 * @description: 兑付流水service
 * @date 2021/8/31 下午7:28
 */
@Service("CashRecordServiceImpl")
@Slf4j
public class CashRecordServiceImpl implements CashRecordService {

  private Map<String,CashPayRegister> registerMap;
  @Autowired
  private CashRecordRepository cashRecordRepository;
  @Autowired(required = false)
  private LoginUserService loginUserService;
  @Autowired(required = false)
  private GenerateCodeService generateCodeService;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;

  public CashRecordServiceImpl(List<CashPayRegister> registers) {
    registerMap = Maps.newHashMap();
    if (CollectionUtils.isEmpty(registers)) {
      return;
    }
    registers.forEach(register -> {
      CashPayRegister cashPayRegister = registerMap.get(register.getKey());
      Validate.isTrue(Objects.isNull(cashPayRegister), String.format("存在相同兑现打款的注册器[%s]", register.getKey()));
      registerMap.put(register.getKey(), register);
    });
  }

  @Override
  @Transactional
  public CashRecord create(CashRecord cashRecord) {
    CashRecord current = this.createForm(cashRecord);
    this.cashRecordRepository.save(current);
    return current;
  }

  @Override
  public Page<CashRecord> findByConditions(Pageable pageable, CashRecordDto dto) {
    ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    if (dto == null) {
      dto = new CashRecordDto();
    }
    return this.cashRecordRepository.findByConditions(pageable, dto);
  }

  @Override
  @Transactional
  public void updateCashStatus(CashRecord cashRecord) {
    Validate.notNull(cashRecord, "兑付记录不能为空");
    Validate.notBlank(cashRecord.getCashCode(), "兑付编码不能为空");
    Validate.notBlank(cashRecord.getCashStatus(), "兑付状态不能为空");
    Validate.notBlank(cashRecord.getRemark(), "兑付备注不能为空");
    this.cashRecordRepository.updateCashStatus(cashRecord);
  }

  @Override
  @Transactional
  public void send(CashPushDto dto) {
    Validate.notNull(dto, "推送对象不能为空");
    Validate.isTrue(!CollectionUtils.isEmpty(dto.getIds()), "兑付流水ID不能为空");
    Validate.notBlank(dto.getPayKey(), "提款key值不能为空");
    CashPayRegister cashPayRegister = registerMap.get(dto.getPayKey());
    Validate.notNull(cashPayRegister, String.format("找不到[%s]对应注册器", dto.getPayKey()));
    List<CashRecord> cashRecords = this.cashRecordRepository.listByIds(dto.getIds());
    Validate.isTrue(!CollectionUtils.isEmpty(cashRecords) && cashRecords.size() == dto.getIds().size(), "兑付流水数据不匹配");
    cashRecords.forEach(cashRecord -> {
      Validate.isTrue(CashPushStatusEnum.WAIT_PUSH.getDictCode().equals(cashRecord.getPushStatus())
          , String.format("只能操作%s记录",CashPushStatusEnum.WAIT_PUSH.getValue()));
    });
    List<CashRecordVo> cashRecordVoList = (List<CashRecordVo>) this.nebulaToolkitService.copyCollectionByBlankList(cashRecords
        , CashRecord.class, CashRecordVo.class, HashSet.class, ArrayList.class);
    Map<String, String> payMap = Maps.newHashMap();
    String pushStatus = CashPushStatusEnum.PUSH_SUCCESS.getDictCode();
    try {
      payMap = cashPayRegister.pay(cashRecordVoList);
    } catch (Exception e) {
      log.error("推送失败",e);
      pushStatus = CashPushStatusEnum.PUSH_FAIL.getDictCode();
    }
    String account = loginUserService.getLoginAccountName();
    Date nowDate = new Date();
    for (CashRecord cashRecord : cashRecords) {
      cashRecord.setPushStatus(pushStatus);
      cashRecord.setPayBatchNo(payMap.get(cashRecord.getCashCode()));
      cashRecord.setPushDate(nowDate);
      cashRecord.setModifyAccount(account);
      cashRecord.setModifyTime(nowDate);
    }
    this.cashRecordRepository.updateBatchById(cashRecords);
  }

  /**
   * 根据id打款（单个或者批量）
   * @param ids
   */
  @Override
  public void pay(List<String> ids) {
    if (CollectionUtils.isEmpty(ids)) {
      return;
    }
    this.cashRecordRepository.updateCashStatusByIds(ids, CashStatusEnum.PAID.getDictCode());
  }

  /**
   * 根据参与者信息和时间范围统计金额
   * @param participatorCode 参与者编码
   * @param participatorFlag 参与者标识
   * @param startTime 开始时间
   * @param endTime 结束时间
   * @return
   */
  @Override
  public BigDecimal findAmountByParticipatorAndDateTime(String participatorCode, String participatorFlag, Date startTime, Date endTime) {
    if (StringUtils.isAnyBlank(participatorCode, participatorFlag)) {
      return BigDecimal.ZERO;
    }
    if (startTime == null || endTime == null) {
      return BigDecimal.ZERO;
    }
    BigDecimal bigDecimal = this.cashRecordRepository.findAmountByParticipatorAndDateTime(participatorCode, participatorFlag, startTime, endTime);
    return bigDecimal == null ? BigDecimal.ZERO : bigDecimal;
  }

  /**
   * 根据参与者信息和时间范围统计次数
   * @param participatorCode 参与者编码
   * @param participatorFlag 参与者标识
   * @param startTime 开始时间
   * @param endTime 结束时间
   * @return
   */
  @Override
  public Integer findCountByParticipatorAndDateTime(String participatorCode, String participatorFlag, Date startTime, Date endTime) {
    if (StringUtils.isAnyBlank(participatorCode, participatorFlag)) {
      return 0;
    }
    if (startTime == null || endTime == null) {
      return 0;
    }
    Integer integer = this.cashRecordRepository.findCountByParticipatorAndDateTime(participatorCode, participatorFlag, startTime, endTime);
    return integer == null ? 0 : integer;
  }

  /**
   * 构建数据
   *
   * @param cashRecord
   * @return
   */
  private CashRecord createForm(CashRecord cashRecord) {
    /*
     * 对静态模型的保存操作过程为：
     * 1、如果当前模型对象不是主模型
     *  1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     *  1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     *  1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     * 2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *   2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *   2.3.2、以及验证每个分组的OneToMany明细信息
     * */
    Date now = new Date();
    if (Objects.nonNull(this.generateCodeService)) {
      cashRecord.setCashCode(this.generateCodeService.generateCode(CashRecordConstant.CASHRECORD_CODE, 1).get(0));
    }
    String account = loginUserService.getLoginAccountName();
    cashRecord.setTenantCode(TenantUtils.getTenantCode());
    cashRecord.setCreateAccount(account);
    cashRecord.setCreateTime(now);
    cashRecord.setModifyAccount(account);
    cashRecord.setModifyTime(now);
    cashRecord.setReceiveDate(now);
    cashRecord.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    cashRecord.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    this.createValidation(cashRecord);
    if (StringUtils.isBlank(cashRecord.getCashStatus())) {
      cashRecord.setCashStatus(CashStatusEnum.NO_PAID.getDictCode());
    }
    if (StringUtils.isBlank(cashRecord.getPushStatus())) {
      cashRecord.setPushStatus(CashPushStatusEnum.WAIT_PUSH.getDictCode());
    }
    return cashRecord;
  }

  /**
   * 在创建一个新的cashRecord模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(CashRecord cashRecord) {
    Validate.notNull(cashRecord, "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    cashRecord.setId(null);
    Validate.notBlank(cashRecord.getTenantCode(), "添加信息时，租户编号不能为空！");
    Validate.notBlank(cashRecord.getCashCode(), "添加信息时，兑现编码不能为空！");
    Validate.notBlank(cashRecord.getCashMethodKey(), "添加信息时，兑现方式标志不能为空！");
    Validate.notBlank(cashRecord.getCashMethodName(), "添加信息时，兑现方式名称不能为空！");
    Validate.notBlank(cashRecord.getRewardKey(), "添加信息时，兑现奖励类型不能为空！");
    Validate.notBlank(cashRecord.getRewardName(), "添加信息时，兑现奖励名称不能为空！");
    Validate.notNull(cashRecord.getAmount(), "添加信息时，兑现金额不能为空！");
    Validate.notBlank(cashRecord.getParticipatorType(), "参与者类型不能为空;");
    Validate.notBlank(cashRecord.getParticipatorCode(), "参与者编码不能为空;");
    Validate.notBlank(cashRecord.getParticipatorName(), "参与者名称不能为空;");
    Validate.isTrue(cashRecord.getPayeeName() == null || cashRecord.getPayeeName().length() < 64, "收款人姓名，在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(cashRecord.getPayeeSex() == null || cashRecord.getPayeeSex().length() < 10, "收款人性别，在进行添加时填入值超过了限定长度(10)，请检查!");
    Validate.isTrue(cashRecord.getPayeePhone() == null || cashRecord.getPayeePhone().length() < 12, "收款人电话，在进行添加时填入值超过了限定长度(11)，请检查!");
    Validate.isTrue(cashRecord.getPayeeBank() == null || cashRecord.getPayeeBank().length() < 64, "收款人账户，在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(cashRecord.getBackgroundIdCardPath() == null || cashRecord.getBackgroundIdCardPath().length() < 255, "收款人身份证正面，在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(cashRecord.getHeadPhotoIdCardPath() == null || cashRecord.getHeadPhotoIdCardPath().length() < 255, "收款人身份证背面，在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(cashRecord.getParticipatorType().length() < 64, "参与者类型，在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(cashRecord.getParticipatorCode().length() < 64, "参与者编码，在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(cashRecord.getParticipatorName().length() < 64, "参与者名称，在进行添加时填入值超过了限定长度(64)，请检查!");

  }

}
