package com.biz.crm.cps.business.consumer.local.controller;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.cps.business.common.local.model.Result;
import com.biz.crm.cps.business.consumer.local.entity.Consumer;
import com.biz.crm.cps.business.consumer.local.service.ConsumerService;
import com.biz.crm.cps.business.consumer.sdk.dto.ConsumerDto;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.util.List;

/**
 * 分利消费者http接口
 * @author songjingen
 */
@Api(tags = "分利消费者信息")
@Slf4j
@RestController
@RequestMapping("/v1/consumer/consumer")
public class  ConsumerController {

  @Resource
  private ConsumerService consumerService;

  /**
   * 只是（分页）查询分利消费者的基本信息
   * 查询消费者数据时，无要求
   * 查询黑名单数据时，需要赋值consumer.blacklistStatus = "009"
   * @param dto
   * @return
   */
  @ApiOperation(value = "只是（分页）查询分利消费者的基本信息, 查询消费者数据时，无要求；查询黑名单数据时，需要赋值consumer.blacklistStatus = '009'")
  @GetMapping(value = "/findByConditions")
  public Result<Page<Consumer>> findByConditions(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                 @ApiParam(name = "consumer", value = "分利消费者对象") ConsumerDto dto) {
    try {
      Page<Consumer> listByPage = consumerService.findByConditions(pageable, dto);
      return Result.ok(listByPage);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 相关的创建过程，http接口。请注意该创建过程除了可以创建consumer中的基本信息以外，还可以对consumer中属于OneToMany关联的明细信息一同进行创建注意：基于（Consumer）模型的创建操作传入的consumerJSON对象，其主键信息不能有值，服务端将会自动为其赋予相关值。另外，创建操作成功后，系统将返回该对象的基本信息（不包括任何关联信息）
   * */
  @ApiOperation(value = "相关的创建过程，http接口。请注意该创建过程除了可以创建consumer中的基本信息以外，还可以对consumer中属于OneToMany关联的明细信息一同进行创建注意：基于（Consumer）模型的创建操作传入的consumerJSON对象，其主键信息不能有值，服务端将会自动为其赋予相关值。另外，创建操作成功后，系统将返回该对象的基本信息（不包括任何关联信息）")
  @PostMapping
  public Result<Consumer> create(@RequestBody @ApiParam(name="consumer" , value="相关的创建过程，http接口。请注意该创建过程除了可以创建consumer中的基本信息以外，还可以对consumer中属于OneToMany关联的明细信息一同进行创建注意：基于（Consumer）模型的创建操作传入的consumerJSON对象，其主键信息不能有值，服务端将会自动为其赋予相关值。另外，创建操作成功后，系统将返回该对象的基本信息（不包括任何关联信息）") Consumer consumer) {
    try {
      Consumer current = this.consumerService.create(consumer);
      return Result.ok(current);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 分利消费者详情信息，根据分利消费者主键ID，查询成功后返回分利消费者的基本信息、标签信息
   * @param id
   * @return 查询成功后返回分利消费者的基本信息、标签信息
   */
  @ApiOperation(value = "根据分利消费者主键ID，查询成功后返回分利消费者的基本信息、标签信息")
  @GetMapping(value = "/{id}")
  public Result<Consumer> findDetailsById(@PathVariable @ApiParam(name = "id", value = "分利消费者主键ID") String id) {
    try {
      Consumer consumer = consumerService.findDetailsById(id);
      return Result.ok(consumer);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 分利消费者详情信息，根据分利消费者编码，查询成功后返回分利消费者的基本信息、标签信息
   * @param externalId
   * @return 查询成功后返回分利消费者的基本信息、标签信息
   */
  @ApiOperation(value = "根据分利消费者编码，查询成功后返回分利消费者的基本信息、标签信息")
  @GetMapping(value = "findDetailsByExternalId")
  public Result<Consumer> findDetailsByExternalId(@RequestParam @ApiParam(name = "externalId", value = "分利消费者编码") String externalId) {
    try {
      Consumer consumer = consumerService.findDetailsByExternalId(externalId);
      return Result.ok(consumer);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }


  /**
   * 移除黑名单(单个或批量)
   * @param ids
   * @return
   */
  @ApiOperation(value = "移除黑名单(单个或批量)")
  @PatchMapping("/disable")
  public Result<?> disable(@RequestBody @ApiParam(name = "ids", value = "string数组类型的主键") List<String> ids) {
    try {
      consumerService.disable(ids);
      return Result.ok();
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 加入黑名单(单个或批量)
   * @param ids
   * @return
   */
  @ApiOperation(value = "加入黑名单(单个或批量)")
  @PatchMapping("/enable")
  public Result<?> enable(@RequestParam @ApiParam(name = "ids", value = "string数组类型的主键") List<String> ids,
                       @RequestParam @ApiParam(name = "blacklistReason", value = "加入黑名单原因") String blacklistReason) {
    try {
      consumerService.enable(ids, blacklistReason);
      return Result.ok();
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

}
