package com.biz.crm.cps.business.consumer.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.cps.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.cps.business.consumer.local.entity.Consumer;
import com.biz.crm.cps.business.consumer.local.entity.ConsumerTerminalMapping;
import com.biz.crm.cps.business.consumer.local.entity.Tag;
import com.biz.crm.cps.business.consumer.local.repository.ConsumerRepository;
import com.biz.crm.cps.business.consumer.local.service.ConsumerService;
import com.biz.crm.cps.business.consumer.local.service.ConsumerTerminalMappingService;
import com.biz.crm.cps.business.consumer.sdk.common.constant.ConsumerCodeConstant;
import com.biz.crm.cps.business.consumer.sdk.dto.ConsumerDto;
import com.biz.crm.cps.business.consumer.sdk.event.ConsumerEventListener;
import com.biz.crm.cps.business.consumer.sdk.vo.ConsumerVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;

/**
 * 分利消费者业务接口实现类
 * @author songjingen
 */
@Service
public class ConsumerServiceImpl implements ConsumerService {

  @Autowired
  private ConsumerRepository consumerRepository;

  @Autowired(required = false)
  private LoginUserService loginUserService;

  @Autowired(required = false)
  private List<ConsumerEventListener> consumerEventListeners;

  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Autowired(required = false)
  private GenerateCodeService generateCodeService;

  @Autowired
  private ConsumerTerminalMappingService consumerTerminalMappingService;


  /**
   * 创建新的Consumer
   * @param consumer
   * @return
   */
  @Transactional
  @Override
  public Consumer create(Consumer consumer) {
    this.createValidation(consumer);
    //验证数据是否存在
    Consumer byExternalId = this.consumerRepository.findByExternalId(consumer.getExternalId());
    if (byExternalId != null) {
      throw new RuntimeException("新增操作时，当前消费者已存在！");
    }

    if (Objects.nonNull(generateCodeService)) {
      consumer.setConsumerCode(generateCodeService.generateCode(ConsumerCodeConstant.CONSUMER_CODE, 1).get(0));
    } else {
      consumer.setConsumerCode(String.valueOf(System.currentTimeMillis()));
    }
    consumer.setBlacklistStatus(EnableStatusEnum.DISABLE.getCode());
    consumer.setCreateTime(new Date());
    consumer.setCreateAccount(loginUserService.getLoginAccountName());
    consumer.setTenantCode(TenantUtils.getTenantCode());
    consumerRepository.save(consumer);
    //增加消费者和终端关联数据
    if (StringUtils.isNotBlank(consumer.getTerminalCode())) {
      ConsumerTerminalMapping mapping = new ConsumerTerminalMapping();
      mapping.setConsumerId(consumer.getId());
      mapping.setTerminalCode(consumer.getTerminalCode());
      this.consumerTerminalMappingService.create(mapping);
    }
    // 返回最终处理的结果，里面带有详细的关联信息
    return consumer;
  }

  /**
   *
   * 根据分利消费者主键ID，查询成功后返回分利消费者的基本信息、标签信息、关联终端信息
   * @param id
   * @return Consumer
   */
  @Override
  public Consumer findDetailsById(String id) {
    /*
     * 0、首要条件是必须要id不为空
     * 1、查询基本信息
     * 2、查询标签信息
     * 3、查询终端信息
     * 4、将标签和终端信息赋值到基本信息的tags、terminals的集合中
     */
    if (StringUtils.isBlank(id)) {
      return null;
    }
    Consumer consumer = consumerRepository.getById(id);
    Set<Tag> tags = consumerRepository.findByConsumerId(id);
    consumer.setTags(tags);
    return consumer;
  }

  /**
   *
   * 根据分利消费者编码，查询成功后返回分利消费者的基本信息、标签信息、关联终端信息
   * @param externalId
   * @return Consumer
   */
  @Override
  public Consumer findDetailsByExternalId(String externalId) {
    /*
     * 0、首要条件是必须要externalId不为空
     * 1、查询基本信息
     * 2、查询标签信息
     * 3、查询终端信息
     * 4、将标签和终端信息赋值到基本信息的tags、terminals的集合中
     */
    if (StringUtils.isBlank(externalId)) {
      return null;
    }
    Consumer consumer = consumerRepository.findByExternalId(externalId);
    if (Objects.isNull(consumer)) {
      return null;
    }
    Set<Tag> tags = consumerRepository.findByConsumerId(consumer.getId());
    consumer.setTags(tags);
    return consumer;
  }

  /**
   * 根据外部编码查询消费者信息
   * @param externalId
   * @return
   */
  @Override
  public Consumer findByExternalId(String externalId) {
    if (StringUtils.isBlank(externalId)) {
      return null;
    }
    return this.consumerRepository.findByExternalId(externalId);
  }

  /**
   * 根据消费者编码查询消费者信息
   * @param consumerCode
   * @return
   */
  @Override
  public Consumer findByConsumerCode(String consumerCode) {
    if (StringUtils.isBlank(consumerCode)) {
      return null;
    }
    return this.consumerRepository.findByConsumerCode(consumerCode);
  }

  /**
   * 根据终端编码查询消费者集合
   * @param terminalCode
   * @return
   */
  @Override
  public List<Consumer> findByTerminalCode(String terminalCode) {
    if (StringUtils.isBlank(terminalCode)) {
      return null;
    }
    return this.consumerRepository.findByTerminalCode(terminalCode);
  }


  /**
   * 只是（分页）查询分利消费者的基本信息，不会包括OneToMany/ManyToMany/ManyToOne的关联信息
   *
   * @param pageable
   * @param dto
   * @return
   */
  @Override
  public Page<Consumer> findByConditions(Pageable pageable, ConsumerDto dto) {
    ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    if(dto == null) {
      dto = new ConsumerDto();
    }
    return consumerRepository.findByConditions(pageable, dto);
  }

  /**
   * 加入黑名单(单个或批量)
   * @param ids
   * @param blacklistReason
   */
  @Transactional
  @Override
  public void enable(List<String> ids, String blacklistReason) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids),"请传入要操作的数据");
    consumerRepository.updateBlacklistStatusByIds(ids, blacklistReason, loginUserService.getLoginAccountName(), EnableStatusEnum.ENABLE);
    if (!CollectionUtils.isEmpty(consumerEventListeners)) {
      List<Consumer> terminals = this.consumerRepository.findByIds(ids);
      List<ConsumerVo> voList = (List<ConsumerVo>) this.nebulaToolkitService.copyCollectionByWhiteList(terminals, Consumer.class, ConsumerVo.class, HashSet.class, ArrayList.class);
      for (ConsumerEventListener event : consumerEventListeners) {
        event.onEnable(voList);
      }
    }
  }

  /**
   * 移除黑名单(单个或批量)
   * @param ids
   */
  @Transactional
  @Override
  public void disable(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids),"请传入要操作的数据");
    consumerRepository.updateBlacklistStatusByIds(ids, null, loginUserService.getLoginAccountName(), EnableStatusEnum.DISABLE);
    if (!CollectionUtils.isEmpty(consumerEventListeners)) {
      List<Consumer> terminals = this.consumerRepository.findByIds(ids);
      List<ConsumerVo> voList = (List<ConsumerVo>) this.nebulaToolkitService.copyCollectionByWhiteList(terminals, Consumer.class, ConsumerVo.class, HashSet.class, ArrayList.class);
      for (ConsumerEventListener event : consumerEventListeners) {
        event.onDisable(voList);
      }
    }
  }

  /**
   * 新增前验证数据的有效性
   * @param consumer
   */
  private void createValidation(Consumer consumer) {
    Validate.notNull(consumer, "进行当前操作时，信息对象必须传入!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(consumer.getId()), "添加信息时，当前信息的数据编号（主键）不能有值！");
    consumer.setId(null);
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.notBlank(consumer.getExternalId(), "添加信息时，分利消费者外部ID不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况）
    Validate.isTrue(consumer.getExternalId() == null || consumer.getExternalId().length() < 255, "分利消费者外部ID，在进行添加时填入值超过了限定长度(255)，请检查!");

  }
}
