package com.biz.crm.cps.business.policy.display.ladder.local.service.internal;

import com.biz.crm.business.common.sdk.model.LoginUserDetailsForCPS;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.cps.bisiness.policy.display.sdk.common.constant.DisplayConstant;
import com.biz.crm.cps.bisiness.policy.display.sdk.common.enums.DisplayTaskStatusNewEnum;
import com.biz.crm.cps.business.policy.display.ladder.local.entity.DisplayCalculateTask;
import com.biz.crm.cps.business.policy.display.ladder.local.entity.DisplayTaskScanCode;
import com.biz.crm.cps.business.policy.display.ladder.local.model.DisplayPolicyConfigModelVo;
import com.biz.crm.cps.business.policy.display.ladder.local.repository.DisplayCalculateTaskRepository;
import com.biz.crm.cps.business.policy.display.ladder.local.service.DisplayCalculateTaskService;
import com.biz.crm.cps.business.policy.display.ladder.local.service.DisplayPolicyService;
import com.biz.crm.cps.business.policy.display.ladder.local.service.DisplayTaskScanCodeService;
import com.biz.crm.cps.business.product.sdk.service.MaterialVoService;
import com.biz.crm.cps.external.barcode.sdk.common.enums.BarCodeTypeEnum;
import com.biz.crm.cps.external.barcode.sdk.service.BarCodeVoService;
import com.biz.crm.cps.external.barcode.sdk.vo.BarCodeVo;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 陈列分利任务接口实现
 *
 * @author sunx
 * @date 2021/12/16
 */
@Service
public class DisplayCalculateTaskServiceImpl implements DisplayCalculateTaskService {

  @Autowired private DisplayCalculateTaskRepository displayCalculateTaskRepository;

  @Autowired(required = false)
  private LoginUserService loginUserService;

  @Autowired private BarCodeVoService barCodeVoService;

  @Autowired private DisplayPolicyService displayPolicyService;

  @Autowired private MaterialVoService materialVoService;

  @Autowired private DisplayTaskScanCodeService displayTaskScanCodeService;

  /** 获取配置的计算销量的基准（box-标箱，bottle-标盒） */
  @Value("${cps.barcode.sale-standard-unit:}")
  private String saleStandardUnit;

  @Override
  @Transactional
  public void updateScanInfoByProductCodeAndBarCode(String productCode, String barCode) {
    Date now = new Date();
    LoginUserDetailsForCPS loginUser = this.loginUserService.getLoginDetails(LoginUserDetailsForCPS.class);
    Validate.notNull(loginUser, "未获取到操作用户信息");
    String terminalCode = loginUser.getConsumerCode();
    Validate.notBlank(terminalCode, "用户对应的终端编码不能为空");
    BarCodeVo barCodeVo = barCodeVoService.findByBarCode(barCode);
    Validate.notNull(barCodeVo, "未获取到条码对应的配置信息");

    List<DisplayCalculateTask> taskList =
        this.displayCalculateTaskRepository.findCurrentRunningTaskByTerminalCode(
            terminalCode, DisplayTaskStatusNewEnum.RUNNING.getDictCode());

    if (CollectionUtils.isEmpty(taskList)) {
      return;
    }
    // 绑定销量目标陈列分利任务
    List<DisplayCalculateTask> list =
        taskList.stream()
            .filter(a -> DisplayConstant.ZERO.equals(a.getBindSaleStatus()))
            .collect(Collectors.toList());
    if (CollectionUtils.isEmpty(list)) {
      return;
    }
    BigDecimal countSales = BigDecimal.ZERO;
    if (saleStandardUnit.equals(BarCodeTypeEnum.BOX.getKey())) {
      countSales = countSales.add(barCodeVo.getStandardUnitBoxValue());
    } else if (saleStandardUnit.equals(BarCodeTypeEnum.BOTTLE.getKey())) {
      countSales = countSales.add(barCodeVo.getStandardUnitBottleValue());
    }
    Set<String> displayPolicyIds =
        list.stream()
            .filter(a -> StringUtils.isNotBlank(a.getDisplayPolicyId()))
            .map(DisplayCalculateTask::getDisplayPolicyId)
            .collect(Collectors.toSet());
    if (CollectionUtils.isEmpty(displayPolicyIds)) {
      return;
    }
    List<DisplayPolicyConfigModelVo> displayPolicyConfigModelVoList =
        this.displayPolicyService.findDisplayPolicyConfigModelVoByPolicyIds(displayPolicyIds);
    if (CollectionUtils.isEmpty(displayPolicyConfigModelVoList)) {
      return;
    }
    Set<String> policyIdSet =
        this.findValidatePolicyIdSet(displayPolicyConfigModelVoList, productCode);
    if (CollectionUtils.isEmpty(policyIdSet)) {
      return;
    }
    List<DisplayCalculateTask> needUpdateList = Lists.newArrayList();
    List<DisplayTaskScanCode> scanCodeList = Lists.newArrayList();
    for (DisplayCalculateTask displayCalculateTask : list) {
      if (policyIdSet.contains(displayCalculateTask.getDisplayPolicyId())) {
        displayCalculateTask.setSaleActual(
            Optional.ofNullable(displayCalculateTask.getSaleActual())
                .orElse(BigDecimal.ZERO)
                .add(countSales));
        needUpdateList.add(displayCalculateTask);
        DisplayTaskScanCode scanCode = new DisplayTaskScanCode();
        scanCode.setAmount(countSales);
        scanCode.setBusinessCode(displayCalculateTask.getBusinessCode());
        scanCode.setBarCode(barCode);
        scanCode.setProductCode(productCode);
        scanCodeList.add(scanCode);
      }
    }
    if (!CollectionUtils.isEmpty(needUpdateList)) {
      this.displayCalculateTaskRepository.saveOrUpdateBatch(needUpdateList);
    }
    if (!CollectionUtils.isEmpty(scanCodeList)) {
      this.displayTaskScanCodeService.saveBatch(scanCodeList);
    }
  }

  @Override
  public List<DisplayCalculateTask> findByBusinessCodes(List<String> businessCodeList) {
    if(CollectionUtils.isEmpty(businessCodeList)){
      return Lists.newLinkedList();
    }
    return this.displayCalculateTaskRepository.findByBusinessCodes(businessCodeList);
  }

  /**
   * 获取当前扫码商品服务政策维度的政策id集合
   *
   * @param displayPolicyConfigModelVoList
   * @param productCode
   * @return
   */
  private Set<String> findValidatePolicyIdSet(
      List<DisplayPolicyConfigModelVo> displayPolicyConfigModelVoList, String productCode) {
    Set<String> set = Sets.newHashSet();
    if (CollectionUtils.isEmpty(displayPolicyConfigModelVoList)
        || StringUtils.isBlank(productCode)) {
      return set;
    }
    Map<String, List<DisplayPolicyConfigModelVo>> map =
        displayPolicyConfigModelVoList.stream()
            .collect(Collectors.groupingBy(DisplayPolicyConfigModelVo::getPolicyId));
    for (Entry<String, List<DisplayPolicyConfigModelVo>> item : map.entrySet()) {
      List<DisplayPolicyConfigModelVo> list = item.getValue();
      if (CollectionUtils.isEmpty(list)) {
        continue;
      }
      String dimensionFlag = list.get(0).getDimensionFlag();
      List<String> codes =
          list.stream()
              .filter(a -> StringUtils.isNotBlank(a.getSpecialCode()))
              .map(DisplayPolicyConfigModelVo::getSpecialCode)
              .collect(Collectors.toList());
      if (StringUtils.isBlank(dimensionFlag) || CollectionUtils.isEmpty(codes)) {
        continue;
      }
      List<String> curList =
          materialVoService.findMaterialCodeByDimensionCodesAndDimensionType(codes, dimensionFlag);
      if (CollectionUtils.isEmpty(curList) || !curList.contains(productCode)) {
        continue;
      }
      set.add(item.getKey());
    }
    return set;
  }
}
