package com.biz.crm.cps.business.policy.display.ladder.local.service.internal;

import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.cps.bisiness.policy.display.sdk.common.constant.DisplayConstant;
import com.biz.crm.cps.bisiness.policy.display.sdk.dto.DisplayPolicyDto;
import com.biz.crm.cps.business.agreement.sdk.dto.AgreementPolicyDto;
import com.biz.crm.cps.business.agreement.sdk.dto.ProfitAgreementTemplatePaginationDto;
import com.biz.crm.cps.business.agreement.sdk.service.AgreementLadderVoService;
import com.biz.crm.cps.business.agreement.sdk.service.AgreementTemplateVoService;
import com.biz.crm.cps.business.agreement.sdk.vo.AgreementLadderVo;
import com.biz.crm.cps.business.agreement.sdk.vo.AgreementTemplateVo;
import com.biz.crm.cps.business.agreement.sdk.vo.TemplateOrgRelationshipVo;
import com.biz.crm.cps.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.cps.business.policy.display.ladder.local.entity.DisplayPolicy;
import com.biz.crm.cps.business.policy.display.ladder.local.entity.DisplayPolicyExpression;
import com.biz.crm.cps.business.policy.display.ladder.local.entity.DisplayPolicyRange;
import com.biz.crm.cps.business.policy.display.ladder.local.entity.DisplayPolicySaleTarget;
import com.biz.crm.cps.business.policy.display.ladder.local.entity.DisplayPolicyUploadRule;
import com.biz.crm.cps.business.policy.display.ladder.local.model.DisplayPolicyConfigModelVo;
import com.biz.crm.cps.business.policy.display.ladder.local.repository.DisplayPolicyConfigurationRepository;
import com.biz.crm.cps.business.policy.display.ladder.local.repository.DisplayPolicyExpressionRepository;
import com.biz.crm.cps.business.policy.display.ladder.local.repository.DisplayPolicyRangeRepository;
import com.biz.crm.cps.business.policy.display.ladder.local.repository.DisplayPolicyRepository;
import com.biz.crm.cps.business.policy.display.ladder.local.repository.DisplayPolicySaleTargetRepository;
import com.biz.crm.cps.business.policy.display.ladder.local.repository.DisplayPolicyUploadRuleRepository;
import com.biz.crm.cps.business.policy.display.ladder.local.repository.DisplayTaskUploadDetailRepository;
import com.biz.crm.cps.business.policy.display.ladder.local.repository.DisplayTaskUploadRepository;
import com.biz.crm.cps.business.policy.display.ladder.local.service.DisplayPolicyService;
import com.biz.crm.cps.business.policy.display.ladder.local.service.observer.DisplayPolicyMountRegisterImpl;
import com.biz.crm.cps.business.product.sdk.common.constant.MaterialDimensionConstant;
import com.biz.crm.cps.business.product.sdk.service.MaterialVoService;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;


/**
 * @author HuaHongQiang
 * @date 2021/8/27 14:07
 * 陈列政策接口实现类
 */
@Service
public class DisplayPolicyServiceImpl implements DisplayPolicyService {

  @Autowired(required = false)
  private LoginUserService loginUserService;
  @Autowired
  private DisplayPolicyRepository policyRepository;

  @Autowired
  private DisplayPolicyConfigurationRepository configurationRepository;

  @Autowired
  private DisplayPolicyRangeRepository rangeRepository;

  @Autowired
  private DisplayPolicyExpressionRepository expressionRepository;

  @Autowired
  private DisplayPolicySaleTargetRepository saleTargetRepository;

  @Autowired
  private DisplayPolicyUploadRuleRepository uploadRuleRepository;

  @Autowired
  private DisplayTaskUploadRepository displayTaskUploadRepository;

  @Autowired
  private MaterialVoService materialVoService;

  @Autowired
  private AgreementLadderVoService agreementLadderVoService;

  @Autowired
  private AgreementTemplateVoService agreementTemplateVoService;

  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Autowired
  private DisplayPolicyMountRegisterImpl displayPolicyMountRegister;

  @Autowired
  private DisplayTaskUploadDetailRepository displayTaskUploadDetailRepository;

  @Override
  @Transactional
  public DisplayPolicy create(AgreementPolicyDto agreementPolicyDto) {
    if (agreementPolicyDto == null) {
      return null;
    }
    DisplayPolicy displayPolicy = nebulaToolkitService.copyObjectByWhiteList(agreementPolicyDto, DisplayPolicy.class, HashSet.class, ArrayList.class,
            "saleTarget", "uploadRules", "displayPolicyConfigurations", "displayPolicyConfigurations.displayPolicyRanges", "displayPolicyConfigurations.displayPolicyExpressions", "displaySampleGraphs");
    return this.createFrom(displayPolicy);
  }

  @Override
  @Transactional
  public DisplayPolicy createFrom(DisplayPolicy displayPolicy) {
    //校验政策数据
    createValidation(displayPolicy);
    //设置创建信息
    Date now = new Date();
    displayPolicy.setCreateTime(now);
    displayPolicy.setModifyTime(now);
    displayPolicy.setCreateAccount(loginUserService.getLoginAccountName());
    displayPolicy.setModifyAccount(loginUserService.getLoginAccountName());
    displayPolicy.setUploadTimes(displayPolicy.getUploadRules().size());
    List<String> dimensionCodes = Lists.newArrayList();
    //保存陈列政策和销量目标
    this.policyRepository.save(displayPolicy);
    DisplayPolicySaleTarget saleTarget = displayPolicy.getSaleTarget();
    saleTarget.setDisplayPolicyId(displayPolicy.getId());
    this.saleTargetRepository.save(displayPolicy.getSaleTarget());
    //保存陈列示例图片
    if (!CollectionUtils.isEmpty(displayPolicy.getDisplaySampleGraphs())) {
      displayPolicy.getDisplaySampleGraphs().forEach(sampleGraph -> {
        sampleGraph.setBusinessCode(displayPolicy.getId());
      });
      this.displayTaskUploadDetailRepository.saveBatch(displayPolicy.getDisplaySampleGraphs());
    }
    //保存陈列上传规则
    displayPolicy.getUploadRules().forEach(displayPolicyUploadRule -> {
      displayPolicyUploadRule.setDisplayPolicyId(displayPolicy.getId());
    });
    this.uploadRuleRepository.saveBatch(displayPolicy.getUploadRules());
    //保存陈列配置行
    displayPolicy.getDisplayPolicyConfigurations().forEach(policyConfiguration -> {
      policyConfiguration.setDisplayPolicyId(displayPolicy.getId());
      AgreementLadderVo agreementLadderVo = this.agreementLadderVoService.findByLadderCode(policyConfiguration.getAgreementLadderCode());
      if (ObjectUtils.allNotNull(agreementLadderVo)) {
        policyConfiguration.setDisplayNumber(agreementLadderVo.getDisplayNumber());
      }
      this.configurationRepository.save(policyConfiguration);
      //保存配置行下关联的产品维度
      policyConfiguration.getDisplayPolicyRanges().forEach(displayPolicyRange -> {
        dimensionCodes.add(displayPolicyRange.getSpecialCode());
        displayPolicyRange.setConfigurationId(policyConfiguration.getId());
      });
      this.rangeRepository.saveBatch(policyConfiguration.getDisplayPolicyRanges());
      //保存配置行下关联的分利表达式
      policyConfiguration.getDisplayPolicyExpressions().forEach(policyExpression -> {
        policyExpression.setConfigurationId(policyConfiguration.getId());
      });
      this.expressionRepository.saveBatch(policyConfiguration.getDisplayPolicyExpressions());
    });
    CompletableFuture.runAsync(() -> {
      materialVoService.sync(dimensionCodes, displayPolicy.getDimensionFlag());
    });
    return displayPolicy;
  }

  @Override
  public DisplayPolicy findByTemplateCode(String templateCode) {
    return policyRepository.findDetailByTemplateCode(templateCode);
  }

  /**
   * 创建陈列政策参数校验
   *
   * @param displayPolicy 陈列政策创建实体
   */
  private void createValidation(DisplayPolicy displayPolicy) {
    Validate.notNull(displayPolicy, "添加信息时，陈列政策不能为空！");
    displayPolicy.setId(null);
    if (StringUtils.isBlank(displayPolicy.getTenantCode())) {
      displayPolicy.setTenantCode(TenantUtils.getTenantCode());
    }
    Set<DisplayPolicyUploadRule> uploadRules = displayPolicy.getUploadRules();
    Validate.isTrue(!CollectionUtils.isEmpty(displayPolicy.getDisplaySampleGraphs()), "添加信息时，陈列示例图不能为空！");
    Validate.isTrue(!CollectionUtils.isEmpty(displayPolicy.getDisplayPolicyConfigurations()), "添加信息时，政策配置行不能为空！");
    Validate.isTrue(!CollectionUtils.isEmpty(uploadRules), "添加信息时，陈列照上传规则不能为空！");
    Validate.notBlank(displayPolicy.getDimensionFlag(), "产品维度标记不能为空！");
    Validate.notBlank(displayPolicy.getDimensionName(), "产品维度名称不能为空！");
    //校验政策行对应的产品边界和分利表达式
    displayPolicy.getDisplayPolicyConfigurations().forEach(displayPolicyConfiguration -> {
      if (!displayPolicy.getDimensionFlag().equals(DisplayConstant.DIMENSION_ALL)) {
        Validate.notBlank(displayPolicyConfiguration.getAgreementLadderCode(),"阶梯编码不能为空");
        Validate.isTrue(!CollectionUtils.isEmpty(displayPolicyConfiguration.getDisplayPolicyRanges()), "政策关联产品范围不能为空！");
        displayPolicyConfiguration.getDisplayPolicyRanges().forEach(this::displayRangeValidation);
      }
      Validate.isTrue(!CollectionUtils.isEmpty(displayPolicyConfiguration.getDisplayPolicyExpressions()), "政策关联分利表达式不能为空！");
      displayPolicyConfiguration.getDisplayPolicyExpressions().forEach(this::displayPolicyExpressionValidation);
    });
    //校验政策对应的陈列照上传规则
    uploadRules.forEach(this::displayUploadRulesValidation);
    //校验陈列照上传规则不能出现合集
    if (uploadRules.size() == 1) {
      return;
    }
    List<DisplayPolicyUploadRule> displayPolicyUploadRuleList = new ArrayList<>(displayPolicy.getUploadRules());
    displayPolicyUploadRuleList = displayPolicyUploadRuleList.stream().sorted(Comparator.comparing(DisplayPolicyUploadRule::getOrderNum)).collect(Collectors.toList());
    for (int i = 1; i < displayPolicyUploadRuleList.size(); i++) {
      Validate.isTrue(displayPolicyUploadRuleList.get(i).getStartDay() > displayPolicyUploadRuleList.get(i - 1).getStartDay(), "陈列图片后续任务上传开始时间必须大于前一个任务开始时间！");
      Validate.isTrue(displayPolicyUploadRuleList.get(i).getEndDay() >= displayPolicyUploadRuleList.get(i - 1).getEndDay(), "陈列图片后续任务上传结束时间必须大于前一个任务结束时间！");
    }
  }

  /**
   * 陈列产品边界校验
   *
   * @param displayPolicyRange 陈列产品范围实体
   */
  private void displayRangeValidation(DisplayPolicyRange displayPolicyRange) {
    Validate.notNull(displayPolicyRange, "传入信息对象不能为空！");
    displayPolicyRange.setId(null);
    Validate.notBlank(displayPolicyRange.getSpecialCode(), "添加信息时，产品维度编码不能为空！");
    Validate.notBlank(displayPolicyRange.getSpecialName(), "添加信息时，产品维度名称不能为空！");
  }

  /**
   * 陈列分利表达式校验
   *
   * @param displayPolicyExpression 陈列分利表达式校验
   */
  private void displayPolicyExpressionValidation(DisplayPolicyExpression displayPolicyExpression) {
    Validate.notNull(displayPolicyExpression, "传入信息对象不能为空!");
    displayPolicyExpression.setId(null);
    Validate.notBlank(displayPolicyExpression.getRewardMethodFlag(), "添加信息时，奖励方式标志不能为空！");
    Validate.notBlank(displayPolicyExpression.getRewardMethodName(), "添加信息时，奖励方式名称不能为空！");
    Validate.notNull(displayPolicyExpression.getDisplayNumber(), "添加信息时，陈列面达标个数不能为空！");
//    Validate.isTrue(displayPolicyExpression.getDisplayNumber() > 0, "陈列面达标个数必须大于0！");
    Validate.notNull(displayPolicyExpression.getRewardData(), "添加信息时，奖励金额不能为空！");
    Validate.isTrue(displayPolicyExpression.getRewardData().compareTo(BigDecimal.ZERO) > 0, "奖励数据必须大于0！");
  }

  /**
   * 陈列照上传规则校验
   *
   * @param uploadRule 陈列上传规则实体
   */
  private void displayUploadRulesValidation(DisplayPolicyUploadRule uploadRule) {
    Validate.notNull(uploadRule, "传入信息对象不能为空！");
    uploadRule.setId(null);
    Validate.notNull(uploadRule.getOrderNum(), "添加信息时，陈列照上传规则次数标记不能为空！");
    Validate.isTrue(uploadRule.getOrderNum() > 0, "陈列照上传规则次数标记必须大于0！");
    Validate.notNull(uploadRule.getStartDay(), "添加信息时，陈列照上传规则开始时间不能为空！");
    Validate.isTrue(uploadRule.getStartDay() > 0 && uploadRule.getStartDay() < 32,
            "陈列照上传规则开始时间必须大于0且小于32！");
    Validate.notNull(uploadRule.getEndDay(), "添加信息时，陈列照上传规则结束时间不能为空！");
    Validate.isTrue(uploadRule.getEndDay() >= uploadRule.getStartDay()
            && uploadRule.getEndDay() < 32, "陈列照上传规则结束时间必须大于陈列照上传规则开始时间且小于32！");
  }

  /**
   * 陈列政策范围重复校验
   * step1:获取当前有效期时间内，同组织的包含陈列任务的模板信息
   * step2:获取创建政策的物料编码集合
   * step3:查询交集模板对应的政策列表
   * step4:循环政策列表，获取物料列表并与创建政策的物料编码取交集，若交集存在则抛出异常
   *
   * @param agreementTemplateVo 陈列产品范围实体
   */
  @Override
  public void validatePolicyScope(AgreementTemplateVo agreementTemplateVo, AgreementPolicyDto agreementPolicyDto) {
    ProfitAgreementTemplatePaginationDto templatePaginationDto = new ProfitAgreementTemplatePaginationDto();
    if (!CollectionUtils.isEmpty(agreementTemplateVo.getTemplateOrgRelationships())) {
      //如果协议模板设定了组织范围，则取组织范围放入查询参数
      List<String> orgCodes = agreementTemplateVo.getTemplateOrgRelationships().stream().map(TemplateOrgRelationshipVo::getOrgCode).collect(Collectors.toList());
      templatePaginationDto.setOrgCodes(orgCodes);
    }
    templatePaginationDto.setPolicyCode(this.displayPolicyMountRegister.getKey());
    templatePaginationDto.setEffectiveScopeStartTime(agreementTemplateVo.getEffectiveStartTime());
    templatePaginationDto.setEffectiveScopeEndTime(agreementTemplateVo.getEffectiveEndTime());
    templatePaginationDto.setStatus(EnableStatusEnum.ENABLE.getCode());
    //根据组织列表，生效时间查询已存在陈列政策的协议模板
    List<AgreementTemplateVo> templateVos = this.agreementTemplateVoService.findByConditions(templatePaginationDto);
    if (CollectionUtils.isEmpty(templateVos)) {
      //如果没有找到交集模板，则校验通过
      return;
    }
    DisplayPolicy displayPolicy = nebulaToolkitService.copyObjectByWhiteList(agreementPolicyDto, DisplayPolicy.class, HashSet.class, ArrayList.class, "displayPolicyConfigurations", "displayPolicyConfigurations.displayPolicyRanges", "displayPolicyConfigurations.displayPolicyExpressions");
    //通过交集模板列表查询对应的陈列政策
    List<String> templateCodes = templateVos.stream().map(AgreementTemplateVo::getTemplateCode).collect(Collectors.toList());
    DisplayPolicyDto displayPolicyDto = new DisplayPolicyDto();
    displayPolicyDto.setTemplateCodes(templateCodes);
    List<DisplayPolicy> quantifyPolicies = this.policyRepository.findByConditions(displayPolicyDto);
    if (CollectionUtils.isEmpty(quantifyPolicies)) {
      //如果没有找到陈列政策，则校验通过
      return;
    }
    //获取本次创建政策阶梯编码-是否包含全部产品-物料编码集合
    Map<String, Map<Boolean, List<String>>> ladderRangeMap = this.policyLadderValidation(displayPolicy);
    //循环交集模板列表，获取对应包含的政策阶梯编码-是否包含全部产品-物料编码集合，并进行验证
    quantifyPolicies.forEach(policy -> {
      Map<String, Map<Boolean, List<String>>> oldLadderRangeMap = this.policyLadderValidation(policy);
      oldLadderRangeMap.forEach((key, value) -> {
        if (ladderRangeMap.containsKey(key)) {
          Map<Boolean, List<String>> rangeMap = ladderRangeMap.get(key);
          Validate.isTrue(rangeMap.containsKey(false) && value.containsKey(false), "多个政策配置共用同一个协议阶梯并包含全部产品！");
          //求此配合的物料编码集合和本次上传的同阶梯编码的物料集合交集
          value.get(false).retainAll(rangeMap.get(false));
          Validate.isTrue(CollectionUtils.isEmpty(value.get(false)), "多个政策配置共用同一个协议阶梯并出现物料重复");
        }
      });
    });
  }

  /**
   * 修改陈列政策
   * @param deserialize
   * @param before
   */
  @Transactional
  @Override
  public void update(AgreementPolicyDto deserialize, Boolean before) {
      if (before) {
        this.policyRepository.deleteByTemplateCode(deserialize.getTemplateCode());
        this.create(deserialize);
      }
  }

  @Override
  public List<DisplayPolicyConfigModelVo> findDisplayPolicyConfigModelVoByPolicyIds(
      Set<String> policyIds) {
    if(CollectionUtils.isEmpty(policyIds)){
      return Lists.newLinkedList();
    }
    return this.policyRepository.findDisplayPolicyConfigModelVoByPolicyIds(policyIds);
  }

  @Override
  public List<String> findPolicyUploadSamplePicUrlListByPolicyId(String policyId) {
    if(StringUtils.isBlank(policyId)) {
      return Lists.newLinkedList();
    }
    return this.policyRepository.findPolicyUploadSamplePicUrlListByPolicyId(policyId);
  }

  /**
   * 获取政策配置的物料编码列表
   * 此方法陈列政策验重使用，前提为本次创建模板已经在数据库中查询到组织-时间有交集的模板信息
   * 故此方法内部会校验产品维度不能为全部，当产品维度为全部时抛出异常
   * 返回政策下所有配置的物料编码集合
   *
   * @param displayPolicy 陈列政策
   */
  private List<String> policyRangeValidation(DisplayPolicy displayPolicy) {
    List<String> materialCodes = Lists.newArrayList();
    Validate.isTrue(!CollectionUtils.isEmpty(displayPolicy.getDisplayPolicyConfigurations()), "陈列政策必须包含配置！");
    displayPolicy.getDisplayPolicyConfigurations().forEach(configuration -> {
      Validate.isTrue(!CollectionUtils.isEmpty(configuration.getDisplayPolicyExpressions()), "陈列政策配置必须配置产品维度范围！");
      List<String> materialCodeList = this.materialVoService.findMaterialCodeByDimensionCodesAndDimensionType(configuration.getDisplayPolicyRanges().stream().map(DisplayPolicyRange::getSpecialCode).collect(Collectors.toList()), displayPolicy.getDimensionFlag());
      if (!CollectionUtils.isEmpty(materialCodeList)) {
        materialCodes.addAll(materialCodeList);
      }
    });
    return materialCodes;
  }

  /**
   * 获取包量政策的协议阶梯编码-是否全部产品-产品维度列表
   *
   * @param quantifyPolicy 陈列政策
   * @return 阶梯编码-是否包含全部产品-物料编码集合
   */
  private Map<String, Map<Boolean, List<String>>> policyLadderValidation(DisplayPolicy quantifyPolicy) {
    //ladderRangeMap为政策配置的 阶梯编码-是否包含全部产品-物料编码集合
    Map<String, Map<Boolean, List<String>>> ladderRangeMap = Maps.newHashMap();
    quantifyPolicy.getDisplayPolicyConfigurations().forEach(quantifyConfiguration -> {
      Validate.notEmpty(quantifyConfiguration.getAgreementLadderCode(), "创建包量政策时阶梯编码不能为空！");
      Map<Boolean, List<String>> rangeMap = Maps.newHashMap();
      if (MaterialDimensionConstant.DIMENSION_ALL.equals(quantifyPolicy.getDimensionFlag())) {
        //如果配置为全部商品，则包含全部产品为true，物料编码集合为空数组
        rangeMap.put(true, Lists.newArrayList());
      } else {
        //如果配置不为全部商品，则包含全部产品为false，查询物料编码集合
        rangeMap.put(false, this.policyRangeValidation(quantifyPolicy));
      }
      if (ladderRangeMap.containsKey(quantifyConfiguration.getAgreementLadderCode())) {
        //如果一个政策内出现多个配置，共用一个协议阶梯的情况，需要验证多个配置的产品维度
        Map<Boolean, List<String>> oldRangeMap = ladderRangeMap.get(quantifyConfiguration.getAgreementLadderCode());
        Validate.isTrue(oldRangeMap.containsKey(false) && rangeMap.containsKey(false), "多个政策配置共用同一个协议阶梯并包含全部产品！");
        //新建物料编码集合来进行校验
        List<String> oldRanges = Lists.newArrayList();
        oldRanges.addAll(oldRangeMap.get(false));
        oldRanges.retainAll(rangeMap.get(false));
        Validate.isTrue(CollectionUtils.isEmpty(oldRanges), "多个政策配置共用同一个协议阶梯并出现物料重复");
        //校验通过则叠加物料编码集合
        rangeMap.get(false).addAll(oldRangeMap.get(false));
      }
      ladderRangeMap.put(quantifyConfiguration.getAgreementLadderCode(), rangeMap);
    });
    return ladderRangeMap;
  }

}
