package com.biz.crm.cps.business.policy.quantify.fiscal.local.service.builder;

import com.biz.crm.cps.bisiness.policy.quantify.sdk.constant.QuantifyConstant;
import com.biz.crm.cps.bisiness.policy.quantify.sdk.constant.QuantifyTaskStatusEnum;
import com.biz.crm.cps.business.agreement.sdk.vo.AgreementLadderVo;
import com.biz.crm.cps.business.agreement.sdk.vo.AgreementVo;
import com.biz.crm.cps.business.policy.quantify.fiscal.local.entity.QuantifyFiscal;
import com.biz.crm.cps.business.policy.quantify.fiscal.local.entity.QuantifyTask;
import com.biz.crm.cps.business.policy.quantify.fiscal.local.repository.QuantifyFiscalRepository;
import com.biz.crm.cps.business.policy.quantify.fiscal.local.utils.QuantifyUtils;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author jerry7
 * @date 2021/8/25 15:09
 * 按半年度实现包量任务的组装
 */
@Component
public class QuantifyTaskHalfYearBuilder implements QuantifyTaskBuilder {

  @Autowired
  private QuantifyFiscalRepository quantifyFiscalRepository;

  @Override
  public String quantifyTaskCycle() {
    return QuantifyConstant.HALF_YEAR;
  }

  @Override
  public List<QuantifyTask> assemblyQuantifyTask(AgreementVo agreementVo, AgreementLadderVo agreementLadderVo) {
    List<QuantifyTask> taskList = Lists.newArrayList();
    if (!ObjectUtils.allNotNull(agreementVo, agreementLadderVo)) {
      return taskList;
    }
    Date effectiveTime = agreementVo.getEffectiveStartTime();
    Date now = new Date();
    QuantifyTaskStatusEnum statusEnum = QuantifyTaskStatusEnum.STATUS_WAIT;
    if (effectiveTime.before(now)) {
      effectiveTime = now;
      //当协议生效时间小于当前时间，即签署协议时协议已生效，则当前第一个任务为进行中，其余任务为待执行
      statusEnum = QuantifyTaskStatusEnum.STATUS_DOING;
    }
    while (effectiveTime.before(agreementVo.getEffectiveEndTime())) {
      Calendar calendar = Calendar.getInstance();
      calendar.setTime(effectiveTime);
      //获取当前月份
      int nowMonth = calendar.get(Calendar.MONTH) + 1;
      //获取当前年份
      int nowYear = calendar.get(Calendar.YEAR);
      //获取当前时间到下一个半年度结算点的月份
      int addMonths = nowMonth % 6;
      //构造包量任务
      QuantifyTask quantifyTask = new QuantifyTask();
      quantifyTask.setTaskStartTime(effectiveTime);
      if (addMonths == 0) {
        //当前月份为当前半年度最后一个月，则下一个循环起点为下个月第一个时间点
        effectiveTime = DateUtils.ceiling(effectiveTime, Calendar.MONTH);
      } else {
        //当前月份不为当前半年度最后一个月，则下一个循环起点应补充到下一个半年度的第一个月份
        effectiveTime = DateUtils.addMonths(DateUtils.ceiling(effectiveTime, Calendar.MONTH), 6 - addMonths);
      }
      quantifyTask.setTaskEndTime(DateUtils.addSeconds(effectiveTime, -1));
      int halfYear = nowMonth / 6;
      Date targetStartTime = quantifyTask.getTaskStartTime();
      if (halfYear > 0) {
        targetStartTime = DateUtils.setMonths(targetStartTime,6);
        quantifyTask.setTaskName(nowYear + "年下半年包量任务");
      } else {
        targetStartTime = DateUtils.setMonths(targetStartTime,0);
        quantifyTask.setTaskName(nowYear + "年上半年包量任务");
      }
      quantifyTask.setTaskStatus(statusEnum.getCode());
      quantifyTask.setSaleTarget(QuantifyUtils.getScaleByTimeInterval(targetStartTime, quantifyTask.getTaskEndTime(), agreementLadderVo));
      taskList.add(quantifyTask);
      //同一个终端同一商品维度在同一时间段内只能存在一个包量任务，故生成包量任务后将后续任务状态置为待执行
      statusEnum = QuantifyTaskStatusEnum.STATUS_WAIT;
    }
    return taskList;
  }

  @Override
  public List<QuantifyTask> assemblyQuantifyTaskFiscal(AgreementVo agreementVo, AgreementLadderVo agreementLadderVo, String quantifyPolicyId) {
    List<QuantifyTask> taskList = Lists.newArrayList();
    if (!ObjectUtils.allNotNull(agreementVo, agreementLadderVo)) {
      return taskList;
    }
    List<QuantifyFiscal> quantifyFiscals = quantifyFiscalRepository.findByQuantifyPolicyId(quantifyPolicyId);
    Map<Integer, String> fiscalMap = new HashMap<>();
    for(QuantifyFiscal quantifyFiscal : quantifyFiscals){
      fiscalMap.put(quantifyFiscal.getFiscalMonth(), quantifyFiscal.getNatureYearMonth());
    }
    SimpleDateFormat format = new SimpleDateFormat("yyyy-MM");
    Date fiscal = new Date();
    try {
      String yearMonth = fiscalMap.get(1);
      fiscal = format.parse(yearMonth);
    } catch (ParseException e) {
      e.printStackTrace();
    }
    Date effectiveTime = agreementVo.getEffectiveStartTime();
    QuantifyTaskStatusEnum statusEnum = QuantifyTaskStatusEnum.STATUS_WAIT;
    if (effectiveTime.before(fiscal)) {
      effectiveTime = fiscal;
      //当协议生效时间小于当前时间，即签署协议时协议已生效，则当前第一个任务为进行中，其余任务为待执行
      statusEnum = QuantifyTaskStatusEnum.STATUS_DOING;
    }
    while (effectiveTime.before(agreementVo.getEffectiveEndTime())) {
      Calendar calendar = Calendar.getInstance();
      calendar.setTime(effectiveTime);
      //获取当前月份
      int nowMonth = calendar.get(Calendar.MONTH) + 1;
      //获取当前年份
      int nowYear = calendar.get(Calendar.YEAR);
      //获取当前时间到下一个半年度结算点的月份
      Calendar end = Calendar.getInstance();
      end.setTime(fiscal);
      int s = end.get(Calendar.MONTH);
      int addMonths = 0;
      if(nowMonth != s){
        addMonths = (nowMonth - s) % 6;
      }
      //构造包量任务
      QuantifyTask quantifyTask = new QuantifyTask();
      quantifyTask.setTaskStartTime(effectiveTime);
      if (addMonths == 0) {
        //当前月份为当前半年度最后一个月，则下一个循环起点为下个月第一个时间点
        effectiveTime = DateUtils.ceiling(effectiveTime, Calendar.MONTH);
      } else {
        //当前月份不为当前半年度最后一个月，则下一个循环起点应补充到下一个半年度的第一个月份
        effectiveTime = DateUtils.addMonths(DateUtils.ceiling(effectiveTime, Calendar.MONTH), 6 - addMonths);
      }
      quantifyTask.setTaskEndTime(DateUtils.addSeconds(effectiveTime, -1));
      int halfYear = (nowMonth - s) / 6;
      Date targetStartTime = quantifyTask.getTaskStartTime();
      if (halfYear > 0) {
        targetStartTime = DateUtils.setMonths(targetStartTime,6);
        quantifyTask.setTaskName(nowYear + "年下半年包量任务");
      } else {
        targetStartTime = DateUtils.setMonths(targetStartTime,0);
        quantifyTask.setTaskName(nowYear + "年上半年包量任务");
      }
      quantifyTask.setTaskStatus(statusEnum.getCode());
      quantifyTask.setSaleTarget(QuantifyUtils.getScaleByTimeInterval(targetStartTime, quantifyTask.getTaskEndTime(), agreementLadderVo));
      taskList.add(quantifyTask);
      //同一个终端同一商品维度在同一时间段内只能存在一个包量任务，故生成包量任务后将后续任务状态置为待执行
      statusEnum = QuantifyTaskStatusEnum.STATUS_WAIT;
    }
    return taskList;
  }
}
