package com.biz.crm.cps.business.policy.quantify.fiscal.local.service.internal;

import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.cps.bisiness.policy.quantify.sdk.dto.QuantifyPolicyDto;
import com.biz.crm.cps.bisiness.policy.quantify.sdk.event.QuantifyFiscalEventListener;
import com.biz.crm.cps.bisiness.policy.quantify.sdk.vo.QuantifyFiscalVo;
import com.biz.crm.cps.business.agreement.sdk.dto.AgreementPolicyDto;
import com.biz.crm.cps.business.agreement.sdk.dto.ProfitAgreementTemplatePaginationDto;
import com.biz.crm.cps.business.agreement.sdk.service.AgreementTemplateVoService;
import com.biz.crm.cps.business.agreement.sdk.vo.AgreementTemplateVo;
import com.biz.crm.cps.business.agreement.sdk.vo.TemplateOrgRelationshipVo;
import com.biz.crm.cps.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.cps.business.policy.quantify.fiscal.local.entity.QuantifyExpression;
import com.biz.crm.cps.business.policy.quantify.fiscal.local.entity.QuantifyFiscal;
import com.biz.crm.cps.business.policy.quantify.fiscal.local.entity.QuantifyPolicy;
import com.biz.crm.cps.business.policy.quantify.fiscal.local.entity.QuantifyRange;
import com.biz.crm.cps.business.policy.quantify.fiscal.local.repository.QuantifyConfigurationRepository;
import com.biz.crm.cps.business.policy.quantify.fiscal.local.repository.QuantifyExpressionRepository;
import com.biz.crm.cps.business.policy.quantify.fiscal.local.repository.QuantifyFiscalRepository;
import com.biz.crm.cps.business.policy.quantify.fiscal.local.repository.QuantifyPolicyRepository;
import com.biz.crm.cps.business.policy.quantify.fiscal.local.repository.QuantifyRangeRepository;
import com.biz.crm.cps.business.policy.quantify.fiscal.local.service.QuantifyPolicyService;
import com.biz.crm.cps.business.policy.quantify.fiscal.local.service.observer.QuantifyPolicyMountRegisterImpl;
import com.biz.crm.cps.business.policy.quantify.fiscal.local.service.observer.QuantifyPolicyPojoObserverRegister;
import com.biz.crm.cps.business.product.sdk.common.constant.MaterialDimensionConstant;
import com.biz.crm.cps.business.product.sdk.service.MaterialVoService;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

/**
 * @author jerry7
 * @date 2021/8/9 19:50
 * 包量政策接口实现类
 */
@Service
public class QuantifyPolicyServiceImpl implements QuantifyPolicyService {

  @Autowired
  private QuantifyPolicyPojoObserverRegister policyPojoObserver;

  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Autowired
  private QuantifyPolicyRepository quantifyPolicyRepository;

  @Autowired
  private QuantifyConfigurationRepository quantifyConfigurationRepository;

  @Autowired
  private QuantifyRangeRepository quantifyRangeRepository;

  @Autowired
  private QuantifyExpressionRepository quantifyExpressionRepository;

  @Autowired(required = false)
  private LoginUserService loginUserService;

  @Autowired
  private AgreementTemplateVoService agreementTemplateVoService;

  @Autowired
  @Qualifier("QuantifyPolicyMountRegisterImpl")
  private QuantifyPolicyMountRegisterImpl quantifyPolicyMountRegister;

  @Autowired
  private QuantifyFiscalRepository quantifyFiscalRepository;

  @Autowired
  private MaterialVoService materialVoService;

  @Autowired(required = false)
  private List<QuantifyFiscalEventListener> quantifyFiscalEventListeners;

  @Override
  @Transactional
  public QuantifyPolicy create(AgreementPolicyDto agreementPolicyDto) {
    if (agreementPolicyDto == null) {
      return null;
    }
    QuantifyPolicy quantifyPolicy = nebulaToolkitService.copyObjectByWhiteList(agreementPolicyDto, QuantifyPolicy.class, HashSet.class, ArrayList.class, "quantifyConfigurations", "quantifyConfigurations.quantifyRanges", "quantifyConfigurations.quantifyExpressions");
    return this.createFrom(quantifyPolicy);
  }

  @Override
  @Transactional
  public QuantifyPolicy createFrom(QuantifyPolicy quantifyPolicy) {
    //校验政策数据
    createValidation(quantifyPolicy);
    Date now = new Date();
    quantifyPolicy.setCreateTime(now);
    quantifyPolicy.setModifyTime(now);
    quantifyPolicy.setCreateAccount(loginUserService.getLoginAccountName());
    quantifyPolicy.setModifyAccount(loginUserService.getLoginAccountName());
    //保存包量政策
    this.quantifyPolicyRepository.save(quantifyPolicy);
    //保存财年
    if(quantifyPolicy.getRewardCycleType().equals("2")) {
      List<QuantifyFiscalVo> vos = new ArrayList<>();
      if (!CollectionUtils.isEmpty(this.quantifyFiscalEventListeners)) {
        for (QuantifyFiscalEventListener quantifyFiscalEventListener : this.quantifyFiscalEventListeners) {
          vos = quantifyFiscalEventListener.onFindFiscal(quantifyPolicy.getFiscal());
        }
      }
      //TODO 测试使用，记得删除-----开始
      QuantifyFiscalVo vo1 = new QuantifyFiscalVo();
      vo1.setMonth(1);
      vo1.setYearMonth("2021-02");
      vos.add(vo1);
      QuantifyFiscalVo vo2 = new QuantifyFiscalVo();
      vo2.setMonth(2);
      vo2.setYearMonth("2021-03");
      vos.add(vo2);
      QuantifyFiscalVo vo3 = new QuantifyFiscalVo();
      vo3.setMonth(3);
      vo3.setYearMonth("2021-04");
      vos.add(vo3);
      QuantifyFiscalVo vo4 = new QuantifyFiscalVo();
      vo4.setMonth(4);
      vo4.setYearMonth("2021-05");
      vos.add(vo4);
      QuantifyFiscalVo vo5 = new QuantifyFiscalVo();
      vo5.setMonth(5);
      vo5.setYearMonth("2021-06");
      vos.add(vo5);
      QuantifyFiscalVo vo6 = new QuantifyFiscalVo();
      vo6.setMonth(6);
      vo6.setYearMonth("2021-07");
      vos.add(vo6);
      QuantifyFiscalVo vo7 = new QuantifyFiscalVo();
      vo7.setMonth(7);
      vo7.setYearMonth("2021-08");
      vos.add(vo7);
      QuantifyFiscalVo vo8 = new QuantifyFiscalVo();
      vo8.setMonth(8);
      vo8.setYearMonth("2021-09");
      vos.add(vo8);
      QuantifyFiscalVo vo9 = new QuantifyFiscalVo();
      vo9.setMonth(9);
      vo9.setYearMonth("2021-10");
      vos.add(vo9);
      QuantifyFiscalVo vo10 = new QuantifyFiscalVo();
      vo10.setMonth(10);
      vo10.setYearMonth("2021-11");
      vos.add(vo10);
      QuantifyFiscalVo vo11 = new QuantifyFiscalVo();
      vo11.setMonth(11);
      vo11.setYearMonth("2021-12");
      vos.add(vo11);
      QuantifyFiscalVo vo12 = new QuantifyFiscalVo();
      vo12.setMonth(12);
      vo12.setYearMonth("2022-01");
      vos.add(vo12);
      //TODO 测试使用，记得删除-----结束
      if (!CollectionUtils.isEmpty(vos)) {
        Collection<QuantifyFiscal> quantifyFiscals = nebulaToolkitService.copyCollectionByWhiteList(vos, QuantifyFiscalVo.class, QuantifyFiscal.class, HashSet.class, ArrayList.class);
        quantifyFiscals.forEach(quantifyFiscal -> quantifyFiscal.setQuantifyPolicyId(quantifyPolicy.getId()));
        quantifyFiscalRepository.saveBatch(quantifyFiscals);
      }
    }
    //产品维度编码，用于同步物料数据
    List<String> dimensionCodes = Lists.newArrayList();
    //保存包量配置行
    quantifyPolicy.getQuantifyConfigurations().forEach(quantifyConfiguration -> {
      quantifyConfiguration.setQuantifyPolicyId(quantifyPolicy.getId());
      this.quantifyConfigurationRepository.save(quantifyConfiguration);
      //保存配置行下关联的分利表达式
      quantifyConfiguration.getQuantifyExpressions().forEach(quantifyExpression -> quantifyExpression.setConfigurationId(quantifyConfiguration.getId()));
      this.quantifyExpressionRepository.saveBatch(quantifyConfiguration.getQuantifyExpressions());
      //当产品维度设置不为全部时保存配置行下关联的产品维度
      if (!MaterialDimensionConstant.DIMENSION_ALL.equals(quantifyPolicy.getDimensionFlag())) {
        quantifyConfiguration.getQuantifyRanges().forEach(quantifyRange -> {
          dimensionCodes.add(quantifyRange.getSpecialCode());
          quantifyRange.setConfigurationId(quantifyConfiguration.getId());
        });
        this.quantifyRangeRepository.saveBatch(quantifyConfiguration.getQuantifyRanges());
      }
    });
    CompletableFuture.runAsync(() -> {
      materialVoService.sync(dimensionCodes, quantifyPolicy.getDimensionFlag());
    });
    return quantifyPolicy;
  }

  @Override
  public QuantifyPolicy findDetailByTemplateCode(String templateCode) {
    if (StringUtils.isBlank(templateCode)) {
      return null;
    }
    return this.quantifyPolicyRepository.findDetailByTemplateCode(templateCode);
  }

  @Override
  public QuantifyPolicy findById(String id) {
    return quantifyPolicyRepository.findDetailById(id);
  }

  @Override
  public QuantifyPolicy findByTemplateCode(String templateCode) {
    return quantifyPolicyRepository.findDetailByTemplateCode(templateCode);
  }

  /**
   * 创建包量政策参数校验
   *
   * @param entity 包量政策创建实体
   */
  private void createValidation(QuantifyPolicy entity) {
    Validate.notNull(entity, "进行当前操作时，信息对象必须传入!!");
    Validate.isTrue(StringUtils.isBlank(entity.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    entity.setId(null);
    if (StringUtils.isBlank(entity.getTenantCode())) {
      entity.setTenantCode(TenantUtils.getTenantCode());
    }
    Validate.notBlank(entity.getDimensionFlag(), "产品维度标记不能为空");
    Validate.notBlank(entity.getDimensionName(), "产品维度名称不能为空");
    Validate.notBlank(entity.getRewardCycleType(), "结算类型不能为空");
    if(entity.getRewardCycleType().equals("2")){
      Validate.notBlank(entity.getFiscal(), "选择财年时，财年信息不能为空");
    }
    Validate.isTrue(!CollectionUtils.isEmpty(entity.getQuantifyConfigurations()), "政策配置行不能为空！");
    //校验政策行对应的产品边界和分利表达式
    entity.getQuantifyConfigurations().forEach(quantifyConfiguration -> {
      if (!MaterialDimensionConstant.DIMENSION_ALL.equals(entity.getDimensionFlag())) {
        //如果配置的产品维度不是所有产品，则需要校验产品维度数据不能为空
        Validate.isTrue(!CollectionUtils.isEmpty(quantifyConfiguration.getQuantifyRanges()), "政策关联产品范围不能为空！");
      }
      Validate.notBlank(quantifyConfiguration.getAgreementLadderCode(), "政策关联协议阶梯编码不能为空！");
      Validate.notBlank(quantifyConfiguration.getOnlyHighestLevel(), "请确认是否只使用最高比例分利！");
      Validate.isTrue(!CollectionUtils.isEmpty(quantifyConfiguration.getQuantifyExpressions()), "政策关联分利表达式不能为空！");
      quantifyConfiguration.getQuantifyRanges().forEach(this::quantifyRangeValidation);
      quantifyConfiguration.getQuantifyExpressions().forEach(this::quantifyExpressionValidation);
      //校验同一分利比例下，分利方式不能重复
      Map<BigDecimal, List<String>> expressionMap = new HashMap<>();
      quantifyConfiguration.getQuantifyExpressions().forEach(quantifyExpression -> {
        List<String> rewardMethods = expressionMap.get(quantifyExpression.getReachRate());
        if (CollectionUtils.isEmpty(rewardMethods)) {
          rewardMethods = new ArrayList<>();
        }
        Validate.isTrue(!rewardMethods.contains(quantifyExpression.getRewardMethodFlag() + "," + quantifyExpression.getCalculateCycle()), "同一分利比例下同一种分利方式只能存在一个");
        rewardMethods.add(quantifyExpression.getRewardMethodFlag() + "," + quantifyExpression.getCalculateCycle());
        expressionMap.put(quantifyExpression.getReachRate(), rewardMethods);
      });
    });
  }

  /**
   * 包量分利表达式校验
   *
   * @param entity 包量分利表达式校验
   */
  private void quantifyExpressionValidation(QuantifyExpression entity) {
    Validate.notNull(entity, "进行当前操作时，包量分利表达式信息对象必须传入!!");
    Validate.isTrue(StringUtils.isBlank(entity.getId()), "添加信息时，包量分利表达式信息的数据编号（主键）不能有值！");
    entity.setId(null);
    Validate.notBlank(entity.getRewardMethodFlag(), "添加信息时，包量分利表达式奖励方式标志不能为空！");
    Validate.notBlank(entity.getRewardMethodName(), "添加信息时，包量分利表达式奖励方式名称不能为空！");
    Validate.notBlank(entity.getAwardConditionFlag(), "添加信息时，包量分利表达式奖励条件标志不能为空！");
    Validate.notBlank(entity.getAwardConditionName(), "添加信息时，包量分利表达式奖励条件名称不能为空！");
    Validate.notBlank(entity.getCalculateCycle(), "添加信息时，包量分利表达式结算周期结算不能为空！");
    Validate.notNull(entity.getReachRate(), "添加信息时，包量分利表达式阶梯达成比例不能为空！");
    Validate.isTrue(entity.getReachRate().compareTo(BigDecimal.ZERO) > 0, "包量分利表达式阶梯达成比例必须大于0！");
    Validate.notNull(entity.getRewardData(), "添加信息时，包量分利表达式奖励金额不能为空！");
    Validate.isTrue(entity.getRewardData().compareTo(BigDecimal.ZERO) > 0, "包量分利表达式奖励金额必须大于0！");
  }

  /**
   * 包量产品范围校验
   *
   * @param entity 包量产品范围实体
   */
  private void quantifyRangeValidation(QuantifyRange entity) {
    Validate.notNull(entity, "进行当前操作时，信息对象必须传入!!");
    Validate.isTrue(StringUtils.isBlank(entity.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    entity.setId(null);
    Validate.notBlank(entity.getSpecialCode(), "添加信息时，产品维度编码不能为空！");
    Validate.notBlank(entity.getSpecialName(), "添加信息时，产品维度名称不能为空！");
  }

  /**
   * 包量政策范围重复校验
   * step1:获取当前有效期时间内，同组织的包含包量任务的模板信息
   * step2:获取创建政策的物料编码集合
   * step3:查询交集模板对应的政策列表
   * step4:循环政策列表，获取物料列表并与创建政策的物料编码取交集，若交集存在则抛出异常
   *
   * @param agreementTemplateVo 包量产品范围实体
   */
  @Override
  public void validatePolicyScope(AgreementTemplateVo agreementTemplateVo, AgreementPolicyDto agreementPolicyDto) {
    ProfitAgreementTemplatePaginationDto templatePaginationDto = new ProfitAgreementTemplatePaginationDto();
    if (!CollectionUtils.isEmpty(agreementTemplateVo.getTemplateOrgRelationships())) {
      //如果协议模板设定了组织范围，则取组织范围放入查询参数
      List<String> orgCodes = agreementTemplateVo.getTemplateOrgRelationships().stream().map(TemplateOrgRelationshipVo::getOrgCode).collect(Collectors.toList());
      templatePaginationDto.setOrgCodes(orgCodes);
    }
    templatePaginationDto.setPolicyCode(this.quantifyPolicyMountRegister.getKey());
    templatePaginationDto.setEffectiveScopeStartTime(agreementTemplateVo.getEffectiveStartTime());
    templatePaginationDto.setEffectiveScopeEndTime(agreementTemplateVo.getEffectiveEndTime());
    templatePaginationDto.setStatus(EnableStatusEnum.ENABLE.getCode());
    //根据组织列表，生效时间查询已存在包量政策的协议模板
    List<AgreementTemplateVo> templateVos = this.agreementTemplateVoService.findByConditions(templatePaginationDto);
    if (CollectionUtils.isEmpty(templateVos)) {
      //如果没有找到交集模板，则校验通过
      return;
    }
    QuantifyPolicy quantifyPolicy = nebulaToolkitService.copyObjectByWhiteList(agreementPolicyDto, QuantifyPolicy.class, HashSet.class, ArrayList.class, "quantifyConfigurations", "quantifyConfigurations.quantifyRanges", "quantifyConfigurations.quantifyExpressions");
    //通过交集模板列表查询对应的包量政策
    List<String> templateCodes = templateVos.stream().map(AgreementTemplateVo::getTemplateCode).collect(Collectors.toList());
    QuantifyPolicyDto quantifyPolicyDto = new QuantifyPolicyDto();
    quantifyPolicyDto.setTemplateCodes(templateCodes);
    List<QuantifyPolicy> quantifyPolicies = this.quantifyPolicyRepository.findByConditions(quantifyPolicyDto);
    if (CollectionUtils.isEmpty(quantifyPolicies)) {
      //如果没有找到包量政策，则校验通过
      return;
    }
    //获取本次创建政策阶梯编码-是否包含全部产品-物料编码集合
    Map<String, Map<Boolean, List<String>>> ladderRangeMap = this.policyLadderValidation(quantifyPolicy);
    //循环交集模板列表，获取对应包含的政策阶梯编码-是否包含全部产品-物料编码集合，并进行验证
    quantifyPolicies.forEach(policy -> {
      Map<String, Map<Boolean, List<String>>> oldLadderRangeMap = this.policyLadderValidation(policy);
      oldLadderRangeMap.forEach((key, value) -> {
        if (ladderRangeMap.containsKey(key)) {
          Map<Boolean, List<String>> rangeMap = ladderRangeMap.get(key);
          Validate.isTrue(rangeMap.containsKey(false) && value.containsKey(false), "多个政策配置共用同一个协议阶梯并包含全部产品！");
          //求此配合的物料编码集合和本次上传的同阶梯编码的物料集合交集
          value.get(false).retainAll(rangeMap.get(false));
          Validate.isTrue(CollectionUtils.isEmpty(value.get(false)), "多个政策配置共用同一个协议阶梯并出现物料重复");
        }
      });
    });
  }

  /**
   * 修改包量信息
   * @param deserialize
   * @param before
   */
  @Transactional
  @Override
  public void update(AgreementPolicyDto deserialize, Boolean before) {
    if (before) {
      this.quantifyPolicyRepository.deleteByTemplateCode(deserialize.getTemplateCode());
      this.create(deserialize);
    }
  }

  /**
   * 获取政策配置的物料编码列表
   * 此方法包量政策验重使用，前提为本次创建模板已经在数据库中查询到组织-时间有交集的模板信息
   * 返回政策下所有配置的物料编码集合
   *
   * @param quantifyPolicy 包量政策
   */
  private List<String> policyRangeValidation(QuantifyPolicy quantifyPolicy) {
    List<String> materialCodes = Lists.newArrayList();
    Validate.isTrue(!CollectionUtils.isEmpty(quantifyPolicy.getQuantifyConfigurations()), "包量政策必须包含配置！");
    quantifyPolicy.getQuantifyConfigurations().forEach(configuration -> {
      Validate.isTrue(!CollectionUtils.isEmpty(configuration.getQuantifyRanges()), "包量政策配置必须配置产品维度范围！");
      List<String> materialCodeList = this.materialVoService.findMaterialCodeByDimensionCodesAndDimensionType(configuration.getQuantifyRanges().stream().map(QuantifyRange::getSpecialCode).collect(Collectors.toList()), quantifyPolicy.getDimensionFlag());
      if (!CollectionUtils.isEmpty(materialCodeList)) {
        materialCodes.addAll(materialCodeList);
      }
    });
    return materialCodes;
  }

  @Override
  public List<QuantifyPolicy> findByConditions(QuantifyPolicyDto quantifyPolicyDto) {
    if (Objects.isNull(quantifyPolicyDto)) {
      return null;
    }
    return this.quantifyPolicyRepository.findByConditions(quantifyPolicyDto);
  }

  /**
   * 获取包量政策的协议阶梯编码-是否全部产品-产品维度列表
   *
   * @param quantifyPolicy 包量政策
   * @return 阶梯编码-是否包含全部产品-物料编码集合
   */
  private Map<String, Map<Boolean, List<String>>> policyLadderValidation(QuantifyPolicy quantifyPolicy) {
    //ladderRangeMap为政策配置的 阶梯编码-是否包含全部产品-物料编码集合
    Map<String, Map<Boolean, List<String>>> ladderRangeMap = Maps.newHashMap();
    quantifyPolicy.getQuantifyConfigurations().forEach(quantifyConfiguration -> {
      Validate.notEmpty(quantifyConfiguration.getAgreementLadderCode(), "创建包量政策时阶梯编码不能为空！");
      Map<Boolean, List<String>> rangeMap = Maps.newHashMap();
      if (MaterialDimensionConstant.DIMENSION_ALL.equals(quantifyPolicy.getDimensionFlag())) {
        //如果配置为全部商品，则包含全部产品为true，物料编码集合为空数组
        rangeMap.put(true, Lists.newArrayList());
      } else {
        //如果配置不为全部商品，则包含全部产品为false，查询物料编码集合
        rangeMap.put(false, this.policyRangeValidation(quantifyPolicy));
      }
      if (ladderRangeMap.containsKey(quantifyConfiguration.getAgreementLadderCode())) {
        //如果一个政策内出现多个配置，共用一个协议阶梯的情况，需要验证多个配置的产品维度
        Map<Boolean, List<String>> oldRangeMap = ladderRangeMap.get(quantifyConfiguration.getAgreementLadderCode());
        Validate.isTrue(oldRangeMap.containsKey(false) && rangeMap.containsKey(false), "多个政策配置共用同一个协议阶梯并包含全部产品！");
        //新建物料编码集合来进行校验
        List<String> oldRanges = Lists.newArrayList();
        oldRanges.addAll(oldRangeMap.get(false));
        oldRanges.retainAll(rangeMap.get(false));
        Validate.isTrue(CollectionUtils.isEmpty(oldRanges), "多个政策配置共用同一个协议阶梯并出现物料重复");
        //校验通过则叠加物料编码集合
        rangeMap.get(false).addAll(oldRangeMap.get(false));
      }
      ladderRangeMap.put(quantifyConfiguration.getAgreementLadderCode(), rangeMap);
    });
    return ladderRangeMap;
  }
}
