package com.biz.crm.cps.business.policy.quantify.fiscal.local.service.internal;

import cn.hutool.json.JSONUtil;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.model.LoginUserDetailsForCPS;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.cps.bisiness.policy.quantify.sdk.constant.ProfitStatusEnum;
import com.biz.crm.cps.bisiness.policy.quantify.sdk.constant.QuantifyConstant;
import com.biz.crm.cps.bisiness.policy.quantify.sdk.constant.QuantifyTaskStatusEnum;
import com.biz.crm.cps.bisiness.policy.quantify.sdk.dto.QuantifyTaskDto;
import com.biz.crm.cps.business.agreement.sdk.common.enums.AgreementStatusEnum;
import com.biz.crm.cps.business.agreement.sdk.common.enums.AutoSignEnum;
import com.biz.crm.cps.business.agreement.sdk.common.enums.SignStatusEnum;
import com.biz.crm.cps.business.agreement.sdk.dto.AgreementDto;
import com.biz.crm.cps.business.agreement.sdk.dto.LoginUserAgreementDto;
import com.biz.crm.cps.business.agreement.sdk.dto.ProfitAgreementTemplatePaginationDto;
import com.biz.crm.cps.business.agreement.sdk.service.AgreementLadderVoService;
import com.biz.crm.cps.business.agreement.sdk.service.AgreementTemplateVoService;
import com.biz.crm.cps.business.agreement.sdk.service.AgreementVoService;
import com.biz.crm.cps.business.agreement.sdk.vo.AgreementLadderVo;
import com.biz.crm.cps.business.agreement.sdk.vo.AgreementTemplateVo;
import com.biz.crm.cps.business.agreement.sdk.vo.AgreementVo;
import com.biz.crm.cps.business.participator.sdk.common.enums.ParticipatorTypeEnum;
import com.biz.crm.cps.business.policy.quantify.fiscal.local.entity.QuantifyConfiguration;
import com.biz.crm.cps.business.policy.quantify.fiscal.local.entity.QuantifyExpression;
import com.biz.crm.cps.business.policy.quantify.fiscal.local.entity.QuantifyPolicy;
import com.biz.crm.cps.business.policy.quantify.fiscal.local.entity.QuantifyRange;
import com.biz.crm.cps.business.policy.quantify.fiscal.local.entity.QuantifyStatistics;
import com.biz.crm.cps.business.policy.quantify.fiscal.local.entity.QuantifyTask;
import com.biz.crm.cps.business.policy.quantify.fiscal.local.entity.QuantifyTaskDetail;
import com.biz.crm.cps.business.policy.quantify.fiscal.local.entity.QuantifyTaskExpression;
import com.biz.crm.cps.business.policy.quantify.fiscal.local.repository.QuantifyStatisticsRepository;
import com.biz.crm.cps.business.policy.quantify.fiscal.local.repository.QuantifyTaskDetailRepository;
import com.biz.crm.cps.business.policy.quantify.fiscal.local.repository.QuantifyTaskExpressionRepository;
import com.biz.crm.cps.business.policy.quantify.fiscal.local.repository.QuantifyTaskRepository;
import com.biz.crm.cps.business.policy.quantify.fiscal.local.service.QuantifyPolicyService;
import com.biz.crm.cps.business.policy.quantify.fiscal.local.service.QuantifyTaskService;
import com.biz.crm.cps.business.policy.quantify.fiscal.local.service.builder.QuantifyTaskBuilder;
import com.biz.crm.cps.business.policy.quantify.fiscal.local.service.observer.QuantifyPolicyMountRegisterImpl;
import com.biz.crm.cps.business.policy.quantify.fiscal.local.utils.SchedulerUtils;
import com.biz.crm.cps.business.policy.sdk.dto.PolicyRewardConditionDto;
import com.biz.crm.cps.business.policy.sdk.service.observer.PolicyRewardServiceObserver;
import com.biz.crm.cps.business.product.sdk.common.constant.MaterialDimensionConstant;
import com.biz.crm.cps.business.product.sdk.service.MaterialVoService;
import com.biz.crm.cps.external.barcode.sdk.common.enums.BarCodeTypeEnum;
import com.biz.crm.cps.external.barcode.sdk.common.enums.ScanCodeExceptionEnum;
import com.biz.crm.cps.external.barcode.sdk.dto.ScanCodeRecordPageDto;
import com.biz.crm.cps.external.barcode.sdk.service.BarCodeVoService;
import com.biz.crm.cps.external.barcode.sdk.service.ScanCodeRecordVoService;
import com.biz.crm.cps.external.barcode.sdk.vo.BarCodeVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.task.annotations.DynamicTaskService;
import com.bizunited.nebula.task.service.DynamicTaskSchedulerVoService;
import com.bizunited.nebula.task.vo.DynamicTaskSchedulerVo;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * @author jerry7
 * @date 2021/8/12 15:34 包量任务接口实现
 */
@Service("quantifyTaskServiceImpl")
@Slf4j
public class QuantifyTaskServiceImpl implements QuantifyTaskService {

  @Autowired private AgreementVoService agreementVoService;

  @Autowired private QuantifyPolicyService quantifyPolicyService;

  @Autowired private QuantifyTaskRepository quantifyTaskRepository;

  @Autowired(required = false)
  private GenerateCodeService generateCodeService;

  @Autowired private MaterialVoService materialVoService;

  @Autowired private ScanCodeRecordVoService scanCodeRecordVoService;

  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Autowired private List<PolicyRewardServiceObserver> policyRewardServiceObservers;

  @Autowired private DynamicTaskSchedulerVoService dynamicTaskSchedulerVoService;

  @Autowired private QuantifyTaskDetailRepository quantifyTaskDetailRepository;

  @Autowired private QuantifyPolicyMountRegisterImpl quantifyPolicyMountRegister;

  @Autowired private List<QuantifyTaskBuilder> quantifyTaskBuilderList;

  @Autowired private AgreementLadderVoService agreementLadderVoService;

  @Autowired private QuantifyTaskExpressionRepository quantifyTaskExpressionRepository;

  @Autowired private AgreementTemplateVoService agreementTemplateVoService;

  @Autowired(required = false)
  private LoginUserService loginUserService;

  @Autowired private BarCodeVoService barCodeVoService;

  @Autowired private QuantifyStatisticsRepository quantifyStatisticsRepository;

  /** 获取配置的计算销量的基准（box-标箱，bottle-标盒） */
  @Value("${cps.barcode.sale-standard-unit:}")
  private String saleStandardUnit;

  /**
   * step1:根据协议id查询协议信息(包含模板信息) step2:根据协议模板查询对应的包量政策 step3:根据时间段和包量结算周期创建对应的包量任务 step4:保存包量任务实体
   *
   * @param agreementId 协议主键id
   * @return 创建结果
   */
  @Override
  @Transactional
  public List<QuantifyTask> create(String agreementId) {
    AgreementDto agreementDto = new AgreementDto();
    agreementDto.setAgreementCode(agreementId);
    Page<AgreementVo> agreementVo =
        agreementVoService.findByConditions(PageRequest.of(0, 50), agreementDto);
    if (agreementVo != null && !CollectionUtils.isEmpty(agreementVo.getRecords())) {
      return this.create(agreementVo.getRecords().get(0));
    } else {
      return Lists.newArrayList();
    }
  }

  @Override
  @Transactional
  public List<QuantifyTask> create(AgreementVo agreementVo) {
    // 根据协议模板查询对应的包量政策
    QuantifyPolicy quantifyPolicy =
        quantifyPolicyService.findByTemplateCode(agreementVo.getTemplateCode());
    Validate.notNull(quantifyPolicy, "协议模板编码" + agreementVo.getAgreementCode() + "未找到对应的包量政策！");
    List<QuantifyTask> quantifyTasks = Lists.newArrayList();
    quantifyPolicy
        .getQuantifyConfigurations()
        .forEach(
            quantifyConfiguration -> {
              List<QuantifyTask> quantifyTaskList =
                  this.buildQuantifyTask(agreementVo, quantifyConfiguration, quantifyPolicy);
              if (!CollectionUtils.isEmpty(quantifyTaskList)) {
                quantifyTasks.addAll(quantifyTaskList);
              }
            });
    this.quantifyTaskRepository.saveBatch(quantifyTasks);
    quantifyTasks.forEach(
        quantifyTask ->
            this.quantifyTaskExpressionRepository.saveBatch(
                quantifyTask.getQuantifyTaskExpressions()));
    // 生成定时任务
    quantifyTasks.forEach(
        task -> {
          DynamicTaskSchedulerVo taskSchedulerVo = new DynamicTaskSchedulerVo();
          // 自然年
          if (quantifyPolicy.getRewardCycleType().equals("1")) {
            taskSchedulerVo =
                dynamicTaskSchedulerVoService.createIgnorePrefix(
                    task.getBusinessCode(),
                    "quantifyTaskServiceImpl",
                    "triggerQuantityTask",
                    SchedulerUtils.getSchedulerCron(task.getTaskType()),
                    agreementVo.getEffectiveEndTime(),
                    task.getTaskName(),
                    task.getBusinessCode());
          }
          // 财年
          if (quantifyPolicy.getRewardCycleType().equals("2")) {
            // TODO 财年处理
            //        taskSchedulerVo =
            // dynamicTaskSchedulerVoService.createIgnorePrefix(dynamicTaskSchedulerVo, null);
          }
          dynamicTaskSchedulerVoService.start(new String[] {taskSchedulerVo.getTaskCode()});
        });
    return quantifyTasks;
  }

  /**
   * 创建包量任务列表
   *
   * @param agreementVo 协议vo
   * @param quantifyConfiguration 包量协议配置
   * @return 包量任务列表
   */
  private List<QuantifyTask> buildQuantifyTask(
      AgreementVo agreementVo,
      QuantifyConfiguration quantifyConfiguration,
      QuantifyPolicy quantifyPolicy) {
    List<QuantifyTask> quantifyTaskList = Lists.newArrayList();
    // 构造结算周期-结算表达式map
    Map<String, List<QuantifyExpression>> quantifyExpressMap =
        convertMapByCalculateCycle(quantifyConfiguration);
    if (quantifyExpressMap.size() == 0) {
      return quantifyTaskList;
    }
    // 获取销量阶梯信息
    AgreementLadderVo agreementLadderVo =
        agreementLadderVoService.findByLadderCode(quantifyConfiguration.getAgreementLadderCode());
    if (Objects.isNull(agreementLadderVo)) {
      return quantifyTaskList;
    }
    quantifyExpressMap.forEach(
        (key, value) -> {
          // 获取包量任务构造器
          QuantifyTaskBuilder taskBuilder = this.findBuilderByCalculateCycle(key);
          if (Objects.isNull(taskBuilder)) {
            return;
          }
          // 使用构造器构造包量任务
          List<QuantifyTask> quantifyTasks = new ArrayList<>();
          if ("1".equals(quantifyPolicy.getRewardCycleType())) {
            quantifyTasks = taskBuilder.assemblyQuantifyTask(agreementVo, agreementLadderVo);
          }
          if ("2".equals(quantifyPolicy.getRewardCycleType())) {
            quantifyTasks =
                taskBuilder.assemblyQuantifyTaskFiscal(
                    agreementVo, agreementLadderVo, quantifyPolicy.getId());
          }
          if (CollectionUtils.isEmpty(quantifyTasks)) {
            return;
          }
          // 调用编码生成接口，生成业务编码
          List<String> businessCodes =
              generateCodeService.generateCode(
                  QuantifyConstant.QUANTIFY_TASK_CODE, quantifyTasks.size());
          Validate.isTrue(
              !CollectionUtils.isEmpty(businessCodes)
                  && businessCodes.size() == quantifyTasks.size(),
              "生成业务单号异常");
          for (int i = 0; i < quantifyTasks.size(); i++) {
            // 补充业务单号以及其他信息
            QuantifyTask task = quantifyTasks.get(i);
            task.setProfitStatus("1");
            task.setQuantifyPolicyId(quantifyPolicy.getId());
            task.setRewardCycleType(quantifyPolicy.getRewardCycleType());
            task.setBusinessCode(businessCodes.get(i));
            task.setQuantifyPolicyId(quantifyConfiguration.getQuantifyPolicyId());
            task.setTaskType(key);
            task.setAgreementId(agreementVo.getAgreementCode());
            task.setAgreementName(agreementVo.getAgreementName());
            task.setTemplateId(agreementVo.getTemplateCode());
            task.setTemplateName(agreementVo.getBelongTemplate());
            task.setTerminalCode(agreementVo.getTerminalCode());
            task.setTerminalName(agreementVo.getRelationTerminal());
            task.setQuantifyPolicyConfigurationId(quantifyConfiguration.getId());
            task.setTenantCode(TenantUtils.getTenantCode());
            task.setOnlyHighestLevel(quantifyConfiguration.getOnlyHighestLevel());
            task.setQuantifyTaskExpressions(
                (List<QuantifyTaskExpression>)
                    nebulaToolkitService.copyCollectionByWhiteList(
                        value,
                        QuantifyExpression.class,
                        QuantifyTaskExpression.class,
                        HashSet.class,
                        ArrayList.class));
            task.getQuantifyTaskExpressions()
                .forEach(
                    quantifyExpress -> {
                      quantifyExpress.setBusinessCode(task.getBusinessCode());
                      quantifyExpress.setId(null);
                    });
            quantifyTaskList.add(task);
          }
        });
    return quantifyTaskList;
  }

  /**
   * 根据包量政策配置，筛选不同结算周期对应的分利表达式
   *
   * @param quantifyConfiguration 包量政策配置
   * @return 结算周期-分利表达式map
   */
  private Map<String, List<QuantifyExpression>> convertMapByCalculateCycle(
      QuantifyConfiguration quantifyConfiguration) {
    Map<String, List<QuantifyExpression>> quantifyExpressionMap = Maps.newHashMap();
    quantifyConfiguration
        .getQuantifyExpressions()
        .forEach(
            quantifyExpression -> {
              List<QuantifyExpression> expressions =
                  quantifyExpressionMap.get(quantifyExpression.getCalculateCycle());
              if (CollectionUtils.isEmpty(expressions)) {
                expressions = Lists.newArrayList();
              }
              expressions.add(quantifyExpression);
              quantifyExpressionMap.put(quantifyExpression.getCalculateCycle(), expressions);
            });
    return quantifyExpressionMap;
  }

  /**
   * 根据包量政策的结算类别寻找对应的组装实现类
   *
   * @param calculateCycle 结算类别(月/季/半年/年)
   * @return 包量任务组装类
   */
  private QuantifyTaskBuilder findBuilderByCalculateCycle(String calculateCycle) {
    QuantifyTaskBuilder quantifyTaskBuilder = null;
    for (QuantifyTaskBuilder t : quantifyTaskBuilderList) {
      if (t.quantifyTaskCycle().equals(calculateCycle)) {
        quantifyTaskBuilder = t;
        break;
      }
    }
    return quantifyTaskBuilder;
  }

  @Override
  public QuantifyTask findById(String id) {
    return quantifyTaskRepository.getById(id);
  }

  @Override
  public List<QuantifyTask> findByConditions(Pageable pageable, QuantifyTaskDto quantifyTaskDto) {
    return quantifyTaskRepository.findByConditions(pageable, quantifyTaskDto);
  }

  /**
   * 包量任务结算
   *
   * @param businessCode 包量任务业务编码
   */
  @Override
  @Transactional
  public void triggerQuantityTask(String businessCode) {
    /**
     * step1：获取包量任务实例 step2：根据协议的模板编码获取对应包量政策 step3：包量结算，构造包量结算列表 step4：通知费用层进行数据结算
     * step5：根据结算信息更新包量任务状态和达成率
     */
    QuantifyTask quantifyTask = quantifyTaskRepository.findByBusinessCode(businessCode);
    Validate.notNull(quantifyTask, "业务单号：" + businessCode + "未获取到对应包量任务！");
    Validate.isTrue(
        quantifyTask.getTaskStatus().equals(QuantifyTaskStatusEnum.STATUS_DOING.getCode()),
        "只有处于执行中任务可以进行结算");
    QuantifyPolicy policy = quantifyPolicyService.findByTemplateCode(quantifyTask.getTemplateId());
    Validate.notNull(policy, "业务单号：" + businessCode + "未获取到对应包量政策！");
    // 验证协议是否在进行中
    LoginUserAgreementDto loginUserAgreementDto = new LoginUserAgreementDto();
    loginUserAgreementDto.setSignStatus(SignStatusEnum.SIGNED.getCode());
    loginUserAgreementDto.setAgreementStatus(AgreementStatusEnum.EXECUTING.getCode());
    loginUserAgreementDto.setAgreementCode(quantifyTask.getAgreementId());
    List<AgreementVo> agreementVos =
        this.agreementVoService.findByConditions(loginUserAgreementDto);
    Validate.isTrue(
        !CollectionUtils.isEmpty(agreementVos), "业务单号：" + businessCode + "未获取到对应的已生效协议！");
    // 验证模板
    ProfitAgreementTemplatePaginationDto dto = new ProfitAgreementTemplatePaginationDto();
    dto.setTemplateCode(policy.getTemplateCode());
    List<AgreementTemplateVo> agreementTemplateVos =
        agreementTemplateVoService.findByConditions(dto);
    Validate.isTrue(agreementTemplateVos.size() == 1, "业务单号：" + businessCode + "未获取到对应模板信息！");
    // 包量任务结算
    List<QuantifyTaskDetail> quantifyTaskDetails =
        buildTaskDetail(policy, quantifyTask, agreementTemplateVos.stream().findFirst().get());
    // 通知费用层
    this.sendToCost(quantifyTask, quantifyTaskDetails);
    // 更新包量任务信息
    this.updateQuantifyTask(quantifyTask, quantifyTaskDetails);
    // 保存包量结算详情
    this.quantifyTaskDetailRepository.saveBatch(quantifyTaskDetails);
  }

  /**
   * 根据包量政策和任务进行结算，构造任务结算明细列表
   *
   * @param policy 包量政策
   * @param quantifyTask 包量任务
   * @return 任务结算明细列表
   */
  private List<QuantifyTaskDetail> buildTaskDetail(
      QuantifyPolicy policy, QuantifyTask quantifyTask, AgreementTemplateVo agreementTemplateVo) {
    List<QuantifyTaskDetail> quantifyTaskDetails = Lists.newArrayList();
    policy
        .getQuantifyConfigurations()
        .forEach(
            configuration -> {
              if (!quantifyTask.getQuantifyPolicyConfigurationId().equals(configuration.getId())) {
                return;
              }
              BigDecimal scanCount = BigDecimal.ZERO;
              ScanCodeRecordPageDto scanCodeRecordPageDto = new ScanCodeRecordPageDto();
              scanCodeRecordPageDto.setCreateTimeStart(quantifyTask.getTaskStartTime());
              if (quantifyTask.getTaskEndTime().before(agreementTemplateVo.getEffectiveEndTime())) {
                // 如果任务结束时间小于协议生效结束时间，则取任务结束时间来获取销量
                scanCodeRecordPageDto.setCreateTimeEnd(quantifyTask.getTaskEndTime());
              } else {
                scanCodeRecordPageDto.setCreateTimeEnd(agreementTemplateVo.getEffectiveEndTime());
              }
              scanCodeRecordPageDto.setScanParticipatorCode(quantifyTask.getTerminalCode());
              scanCodeRecordPageDto.setScanCodeException(ScanCodeExceptionEnum.NO.getCode());
              if (MaterialDimensionConstant.DIMENSION_ALL.equals(policy.getDimensionFlag())) {
                // 如果商品标签是全部，则直接调用扫码模块获取该时间段内扫码数量
                scanCodeRecordPageDto.setProductCodes(Lists.newArrayList());
                scanCount =
                    scanCodeRecordVoService.countSalesByScanCodeRecordPageDto(
                        scanCodeRecordPageDto);
              } else {
                // 非全部商品，则通过产品维度获取物料编码集合，然后调用扫码模块获取扫码数量
                List<String> dimensionCodes =
                    configuration.getQuantifyRanges().stream()
                        .map(QuantifyRange::getSpecialCode)
                        .collect(Collectors.toList());
                List<String> materialCodes =
                    materialVoService.findMaterialCodeByDimensionCodesAndDimensionType(
                        dimensionCodes, policy.getDimensionFlag());
                if (!CollectionUtils.isEmpty(materialCodes)) {
                  scanCodeRecordPageDto.setProductCodes(materialCodes);
                  scanCount =
                      scanCodeRecordVoService.countSalesByScanCodeRecordPageDto(
                          scanCodeRecordPageDto);
                }
              }
              // 用扫码次数/销量目标计算达标比例,四舍五入并保留两位小数
              BigDecimal reachRate =
                  scanCount
                      .divide(quantifyTask.getSaleTarget(), 4, RoundingMode.HALF_UP)
                      .multiply(new BigDecimal(100));
              // 根据分利标记选择对应的分利计算规则
              if (quantifyTask.getOnlyHighestLevel().equals(AutoSignEnum.YES.getCode())) {
                quantifyTaskDetails.addAll(
                    this.calculateWithOnlyHighestLevel(quantifyTask, scanCount, reachRate));
              } else {
                quantifyTaskDetails.addAll(
                    this.calculateWithAllLevel(quantifyTask, scanCount, reachRate));
              }
            });
    return quantifyTaskDetails;
  }

  /**
   * 向消费层发布结算消息
   *
   * @param quantifyTask 包量任务
   * @param quantifyTaskDetails 包量结算信息列表
   */
  private void sendToCost(QuantifyTask quantifyTask, List<QuantifyTaskDetail> quantifyTaskDetails) {
    if (CollectionUtils.isEmpty(quantifyTaskDetails)) {
      return;
    }
    List<PolicyRewardConditionDto> policyRewardConditionDtos = Lists.newArrayList();
    quantifyTaskDetails.forEach(
        detail -> {
          if (AutoSignEnum.YES.getCode().equals(detail.getProfitStatus())) {
            // 针对需要传递至费用层构造DTO
            PolicyRewardConditionDto policyRewardConditionDto = new PolicyRewardConditionDto();
            policyRewardConditionDto.setPolicyName(this.quantifyPolicyMountRegister.getName());
            // 参与者信息
            policyRewardConditionDto.setActualParticipatorFlag(
                ParticipatorTypeEnum.TERMINAL.getDictCode());
            policyRewardConditionDto.setActualParticipatorCode(quantifyTask.getTerminalCode());
            policyRewardConditionDto.setParticipatorFlag(
                ParticipatorTypeEnum.TERMINAL.getDictCode());
            policyRewardConditionDto.setParticipatorName(ParticipatorTypeEnum.TERMINAL.getValue());
            policyRewardConditionDto.setTemplateCode(quantifyTask.getTemplateId());
            // 奖励方式
            policyRewardConditionDto.setRewardMethodFlag(detail.getRewardMethodFlag());
            policyRewardConditionDto.setRewardMethodName(detail.getRewardMethodName());
            // 奖励条件
            policyRewardConditionDto.setAwardConditionFlag(detail.getAwardConditionFlag());
            policyRewardConditionDto.setAwardConditionName(detail.getAwardConditionName());
            // 奖励金额
            policyRewardConditionDto.setRewardData(detail.getRewardData().toString());
            // 触发动作
            policyRewardConditionDto.setTriggerAction(this.quantifyPolicyMountRegister.getName());
            policyRewardConditionDto.setTriggerObject(quantifyTask.getBusinessCode());
            // 协议编码
            policyRewardConditionDto.setAgreementCode(quantifyTask.getAgreementId());
            policyRewardConditionDto.setGiftCode(detail.getGiftCode());
            policyRewardConditionDto.setRebateType("包量政策分利");
            policyRewardConditionDtos.add(policyRewardConditionDto);
          }
        });
    if (!CollectionUtils.isEmpty(policyRewardConditionDtos)
        && !CollectionUtils.isEmpty(policyRewardServiceObservers)) {
      // 观察者和需要传递至费用层dto不为空时，进行结算传递
      policyRewardServiceObservers.forEach(
          policyRewardServiceObserver ->
              policyRewardServiceObserver.createRewardRecord(policyRewardConditionDtos));
    }
  }

  /**
   * 更新包量任务状态和达成率
   *
   * @param quantifyTask 包量任务
   * @param quantifyTaskDetails 包量任务结算列表
   */
  private void updateQuantifyTask(
      QuantifyTask quantifyTask, List<QuantifyTaskDetail> quantifyTaskDetails) {
    if (CollectionUtils.isEmpty(quantifyTaskDetails)) {
      // 任务结算明细为空，直接设置任务状态为过期
      quantifyTask.setReachRate(BigDecimal.ZERO);
      quantifyTask.setTaskStatus(QuantifyTaskStatusEnum.STATUS_EXPIRED.getCode());
    }
    BigDecimal reachRate = BigDecimal.ZERO;
    String taskStatus = QuantifyTaskStatusEnum.STATUS_EXPIRED.getCode();
    Boolean bool = false;
    for (QuantifyTaskDetail detail : quantifyTaskDetails) {
      if (AutoSignEnum.YES.getCode().equals(detail.getProfitStatus())) {
        bool = true;
        break;
      }
    }
    for (QuantifyTaskDetail detail : quantifyTaskDetails) {
      reachRate = reachRate.add(detail.getReachRate());
      if (detail.getProfitStatus().equals(AutoSignEnum.YES.getCode())) {
        taskStatus = QuantifyTaskStatusEnum.STATUS_COMPLETE.getCode();
      }
    }
    quantifyTask.setReachRate(
        reachRate.divide(new BigDecimal(quantifyTaskDetails.size()), 4, RoundingMode.HALF_UP));
    quantifyTask.setTaskStatus(taskStatus);
    if (bool) {
      quantifyTask.setProfitStatus(ProfitStatusEnum.STATUS_COMPLETE.getCode());
    }
    quantifyTask.setProfitTime(new Date());
    this.quantifyTaskRepository.updateById(quantifyTask);
  }

  /** 定时更新任务时间已经开始的待执行任务状态 */
  @Override
  @DynamicTaskService(cornExpression = "1 0 0 * * ?", taskDesc = "每日更新待执行任务状态")
  public void updateQuantifyTaskStatus() {
    List<QuantifyTask> quantifyTasks =
        this.quantifyTaskRepository.findByAgreementIdAndStatus(
            null, QuantifyTaskStatusEnum.STATUS_WAIT.getCode(), new Date());
    if (!CollectionUtils.isEmpty(quantifyTasks)) {
      quantifyTasks.forEach(
          quantifyTask ->
              quantifyTask.setTaskStatus(QuantifyTaskStatusEnum.STATUS_DOING.getCode()));
      this.quantifyTaskRepository.saveOrUpdateBatch(quantifyTasks);
    }
  }

  @Override
  @Transactional
  public void updateSaleQuantity(String productCode, String barCode) {
    Date now = new Date();
    LoginUserDetailsForCPS loginUser = this.loginUserService.getLoginDetails(LoginUserDetailsForCPS.class);
    Validate.notNull(loginUser, "未获取到操作用户信息");
    String terminalCode = loginUser.getConsumerCode();
    Validate.notBlank(terminalCode, "用户对应的终端编码不能为空");

    BarCodeVo barCodeVo = barCodeVoService.findByBarCode(barCode);
    Validate.notNull(barCodeVo, "未获取到条码对应的配置信息");

    List<QuantifyTask> tasks =
        this.quantifyTaskRepository.findMaterialCurTaskByTerminalCodeAndProductCode(
            terminalCode, productCode);
    if (CollectionUtils.isEmpty(tasks)) {
      return;
    }
    BigDecimal countSales = BigDecimal.ZERO;
    if (saleStandardUnit.equals(BarCodeTypeEnum.BOX.getKey())) {
      countSales = countSales.add(barCodeVo.getStandardUnitBoxValue());
    } else if (saleStandardUnit.equals(BarCodeTypeEnum.BOTTLE.getKey())) {
      countSales = countSales.add(barCodeVo.getStandardUnitBottleValue());
    }
    List<QuantifyStatistics> statistics = Lists.newArrayList();
    for (QuantifyTask task : tasks) {
      task.setSaleActual(
          Optional.ofNullable(task.getSaleActual()).orElse(BigDecimal.ZERO).add(countSales));
      QuantifyStatistics cur = new QuantifyStatistics();
      cur.setBusinessCode(task.getBusinessCode());
      cur.setBarCode(barCode);
      cur.setProductCode(productCode);
      cur.setAmount(countSales);
      cur.setCreateTime(now);
      cur.setTenantCode(TenantUtils.getTenantCode());
      statistics.add(cur);
    }
    log.info("updateSaleQuantity[{}]", JSONUtil.toJsonStr(tasks));
    log.info("statistics[{}]", JSONUtil.toJsonStr(statistics));
    this.quantifyTaskRepository.saveOrUpdateBatch(tasks);
    this.quantifyStatisticsRepository.saveBatch(statistics);
  }

  /**
   * 计算分利达成情况(只有最高级别可以分利)
   *
   * @param quantifyTask 包量任务
   * @param scanCount 扫码数量
   * @param reachRate 任务达成率
   * @return 分利达成明细
   */
  private List<QuantifyTaskDetail> calculateWithOnlyHighestLevel(
      QuantifyTask quantifyTask, BigDecimal scanCount, BigDecimal reachRate) {
    List<QuantifyTaskDetail> quantifyTaskDetails = Lists.newArrayList();
    if (CollectionUtils.isEmpty(quantifyTask.getQuantifyTaskExpressions())) {
      return quantifyTaskDetails;
    }
    // 按照达成率倒序排列分利表达式
    quantifyTask
        .getQuantifyTaskExpressions()
        .sort((o1, o2) -> o2.getReachRate().compareTo(o1.getReachRate()));
    // 是否已经分利标记
    BigDecimal alreadyReachRate = null;
    for (QuantifyTaskExpression expression : quantifyTask.getQuantifyTaskExpressions()) {
      QuantifyTaskDetail quantifyTaskDetail =
          nebulaToolkitService.copyObjectByWhiteList(
              expression, QuantifyTaskDetail.class, Set.class, ArrayList.class);
      quantifyTaskDetail.setId(null);
      quantifyTaskDetail.setBusinessCode(quantifyTask.getBusinessCode());
      quantifyTaskDetail.setSaleTarget(quantifyTask.getSaleTarget());
      quantifyTaskDetail.setActualSales(scanCount);
      quantifyTaskDetail.setReachRate(reachRate);
      if (!Objects.isNull(alreadyReachRate)) {
        // 如果已经有分利表达式达标，则比较当前分利比例是否等于已分利比例，相等则可以进行分利，不相等不能进行分利
        if (alreadyReachRate.compareTo(expression.getReachRate()) == 0) {
          quantifyTaskDetail.setProfitStatus(
              reachRate.compareTo(expression.getReachRate()) > -1
                  ? AutoSignEnum.YES.getCode()
                  : AutoSignEnum.NO.getCode());
        } else {
          quantifyTaskDetail.setProfitStatus(AutoSignEnum.NO.getCode());
        }
      } else {
        // 如果没有分利表达式达标，根据达标比例设定结算明细的分利状态
        quantifyTaskDetail.setProfitStatus(
            reachRate.compareTo(expression.getReachRate()) > -1
                ? AutoSignEnum.YES.getCode()
                : AutoSignEnum.NO.getCode());
        // 更新达标标记
        alreadyReachRate =
            quantifyTaskDetail.getProfitStatus().equals(AutoSignEnum.YES.getCode())
                ? expression.getReachRate()
                : null;
      }
      quantifyTaskDetails.add(quantifyTaskDetail);
    }
    return quantifyTaskDetails;
  }

  /**
   * 计算分利达成情况(所以级别都可以分利)
   *
   * @param quantifyTask 包量任务
   * @param scanCount 扫码数量
   * @param reachRate 任务达成率
   * @return 分利达成明细
   */
  private List<QuantifyTaskDetail> calculateWithAllLevel(
      QuantifyTask quantifyTask, BigDecimal scanCount, BigDecimal reachRate) {
    List<QuantifyTaskDetail> quantifyTaskDetails = Lists.newArrayList();
    if (CollectionUtils.isEmpty(quantifyTask.getQuantifyTaskExpressions())) {
      return quantifyTaskDetails;
    }
    for (QuantifyTaskExpression expression : quantifyTask.getQuantifyTaskExpressions()) {
      QuantifyTaskDetail quantifyTaskDetail =
          nebulaToolkitService.copyObjectByWhiteList(
              expression, QuantifyTaskDetail.class, Set.class, ArrayList.class);
      quantifyTaskDetail.setId(null);
      quantifyTaskDetail.setBusinessCode(quantifyTask.getBusinessCode());
      quantifyTaskDetail.setSaleTarget(quantifyTask.getSaleTarget());
      quantifyTaskDetail.setActualSales(scanCount);
      quantifyTaskDetail.setReachRate(reachRate);
      // 根据达标比例设定结算明细的分利状态
      quantifyTaskDetail.setProfitStatus(
          reachRate.compareTo(expression.getReachRate()) > -1
              ? AutoSignEnum.YES.getCode()
              : AutoSignEnum.NO.getCode());
      quantifyTaskDetails.add(quantifyTaskDetail);
    }
    return quantifyTaskDetails;
  }
}
