package com.biz.crm.cps.business.policy.scan.cycle.local.service.dynamictask;

import com.biz.crm.cps.business.policy.scan.cycle.local.entity.ScanCodeTask;
import com.biz.crm.cps.business.policy.scan.cycle.local.service.ScanCodeTaskService;
import com.biz.crm.cps.business.policy.scan.sdk.common.constant.ScanCodeConstant;
import com.biz.crm.cps.business.policy.scan.sdk.common.constant.SchedulerConstant;
import com.biz.crm.cps.business.policy.scan.sdk.common.enums.RewardTaskStatusEnum;
import com.biz.crm.cps.business.policy.scan.sdk.dto.CycleScanCodeTaskDto;
import com.biz.crm.cps.business.policy.sdk.dto.PolicyRewardConditionDto;
import com.biz.crm.cps.business.policy.sdk.service.observer.PolicyRewardServiceObserver;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.task.annotations.DynamicTaskService;
import org.apache.commons.lang3.time.DateUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.Calendar;
import java.util.Date;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;

/**
 * 协议动态任务
 * @author songjingen
 */
@Component
public class ScanCodeDynamicTask {

  @Autowired
  private ScanCodeTaskService scanCodeTaskService;

  @Autowired
  private List<PolicyRewardServiceObserver> policyRewardServiceObservers;

  @Autowired
  private NebulaToolkitService nebulaToolkitService;

  /**
   * 按照自然月结算任务，月度
   */
  @Transactional
  @DynamicTaskService(cornExpression = SchedulerConstant.CRON_MONTH, taskDesc = "每日更新协议状态")
  public void settleByMonth() {
    Date endTime = new Date();
    Date starterTime = DateUtils.addMonths(DateUtils.ceiling(endTime, Calendar.MONTH), -2);
    CycleScanCodeTaskDto cycleScanCodeTaskDto = new CycleScanCodeTaskDto();
    cycleScanCodeTaskDto.setStarterTime(starterTime);
    cycleScanCodeTaskDto.setEndTime(endTime);
    cycleScanCodeTaskDto.setRewardCycle(ScanCodeConstant.MONTH);
    cycleScanCodeTaskDto.setRewardStatus(RewardTaskStatusEnum.STATUS_WAIT.getCode());
    List<ScanCodeTask> scanCodeTasks = this.scanCodeTaskService.findByConditions(cycleScanCodeTaskDto);
    if (CollectionUtils.isEmpty(scanCodeTasks)) {
      return;
    }
    this.callRewardObservers(scanCodeTasks);
  }

  /**
   * 按照每自然三个月结算任务，季度
   */
  @Transactional
  @DynamicTaskService(cornExpression = SchedulerConstant.CRON_SEASON, taskDesc = "每日更新协议状态")
  public void settleBySeason() {
    Date endTime = new Date();
    Date starterTime = DateUtils.addMonths(DateUtils.ceiling(endTime, Calendar.MONTH), -4);
    CycleScanCodeTaskDto cycleScanCodeTaskDto = new CycleScanCodeTaskDto();
    cycleScanCodeTaskDto.setStarterTime(starterTime);
    cycleScanCodeTaskDto.setEndTime(endTime);
    cycleScanCodeTaskDto.setRewardCycle(ScanCodeConstant.SEASON);
    cycleScanCodeTaskDto.setRewardStatus(RewardTaskStatusEnum.STATUS_WAIT.getCode());
    List<ScanCodeTask> scanCodeTasks = this.scanCodeTaskService.findByConditions(cycleScanCodeTaskDto);
    if (CollectionUtils.isEmpty(scanCodeTasks)) {
      return;
    }
    this.callRewardObservers(scanCodeTasks);
  }

  /**
   * 按照每自然六个月结算任务，半年
   */
  @Transactional
  @DynamicTaskService(cornExpression = SchedulerConstant.CRON_HALF_YEAR, taskDesc = "每日更新协议状态")
  public void settleByHalfYear() {
    Date endTime = new Date();
    Date starterTime = DateUtils.addMonths(DateUtils.ceiling(endTime, Calendar.MONTH), -7);
    CycleScanCodeTaskDto cycleScanCodeTaskDto = new CycleScanCodeTaskDto();
    cycleScanCodeTaskDto.setStarterTime(starterTime);
    cycleScanCodeTaskDto.setEndTime(endTime);
    cycleScanCodeTaskDto.setRewardCycle(ScanCodeConstant.HALF_YEAR);
    cycleScanCodeTaskDto.setRewardStatus(RewardTaskStatusEnum.STATUS_WAIT.getCode());
    List<ScanCodeTask> scanCodeTasks = this.scanCodeTaskService.findByConditions(cycleScanCodeTaskDto);
    if (CollectionUtils.isEmpty(scanCodeTasks)) {
      return;
    }
    this.callRewardObservers(scanCodeTasks);
  }

  /**
   * 按照每自然年结算任务，年度
   */
  @Transactional
  @DynamicTaskService(cornExpression = SchedulerConstant.CRON_YEAR, taskDesc = "每日更新协议状态")
  public void settleByYear() {
    Date endTime = new Date();
    Date starterTime = DateUtils.addYears(DateUtils.ceiling(endTime, Calendar.YEAR), -2);
    CycleScanCodeTaskDto cycleScanCodeTaskDto = new CycleScanCodeTaskDto();
    cycleScanCodeTaskDto.setStarterTime(starterTime);
    cycleScanCodeTaskDto.setEndTime(endTime);
    cycleScanCodeTaskDto.setRewardCycle(ScanCodeConstant.YEAR);
    cycleScanCodeTaskDto.setRewardStatus(RewardTaskStatusEnum.STATUS_WAIT.getCode());
    List<ScanCodeTask> scanCodeTasks = this.scanCodeTaskService.findByConditions(cycleScanCodeTaskDto);
    if (CollectionUtils.isEmpty(scanCodeTasks)) {
      return;
    }
    this.callRewardObservers(scanCodeTasks);
  }

  /**
   * 调用奖励模块组内的PolicyRewardServiceObserver实现类
   * 记录奖励流水
   * @param scanCodeTasks
   */
  private void callRewardObservers(List<ScanCodeTask> scanCodeTasks) {
    if (CollectionUtils.isEmpty(policyRewardServiceObservers)) {
      return;
    }
    //修改状态为已分利
    scanCodeTasks.stream().forEach(cycleScanCodeTask -> cycleScanCodeTask.setRewardStatus(RewardTaskStatusEnum.STATUS_COMPLETE.getCode()));
    this.scanCodeTaskService.updateBatch(scanCodeTasks);
    //通知奖励模块组
    List<PolicyRewardConditionDto> policyRewardConditionDtos = (List<PolicyRewardConditionDto>) this.nebulaToolkitService.copyCollectionByWhiteList(scanCodeTasks, ScanCodeTask.class, PolicyRewardConditionDto.class, HashSet.class, LinkedList.class);
    for (PolicyRewardServiceObserver rewardServiceObserver : policyRewardServiceObservers) {
      rewardServiceObserver.createRewardRecord(policyRewardConditionDtos);
    }
  }
}
