package com.biz.crm.cps.business.policy.scan.cycle.local.service.internal;

import com.biz.crm.cps.business.policy.scan.cycle.local.entity.ScanCodeConfiguration;
import com.biz.crm.cps.business.policy.scan.cycle.local.entity.ScanCodeEvent;
import com.biz.crm.cps.business.policy.scan.cycle.local.repository.ScanCodeEventRepository;
import com.biz.crm.cps.business.policy.scan.cycle.local.service.ScanCodeEventService;
import com.biz.crm.cps.business.policy.scan.cycle.local.service.ScanCodeExpressionService;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.Set;

/**
 * 扫码事件的Service类
 * @author hefan
 */
@Service
public class ScanCodeEventServiceImpl implements ScanCodeEventService {

  @Autowired
  ScanCodeEventRepository scanCodeEventRepository;

  @Autowired
  ScanCodeExpressionService scanCodeExpressionService;

  @Transactional
  @Override
  public void create(ScanCodeConfiguration current) {
    Validate.notNull(current, "扫码配置信息未传");
    Set<ScanCodeEvent> scanCodeEvents = current.getScanCodeEvents();
    Validate.isTrue(!CollectionUtils.isEmpty(scanCodeEvents), "扫码事件信息未配置");
    for (ScanCodeEvent scanCodeEvent : scanCodeEvents) {
      scanCodeEvent.setConfigurationId(current.getId());
      this.createValidation(scanCodeEvent);
    }
    this.scanCodeEventRepository.saveBatch(scanCodeEvents);
    for (ScanCodeEvent scanCodeEvent : scanCodeEvents) {
      this.scanCodeExpressionService.create(scanCodeEvent);
    }
  }

  /**
   * 根据配置行，修改事件
   * @param configuration
   * @param before 当前时间是否在模板签署时间之前
   */
  @Override
  @Transactional
  public void update(ScanCodeConfiguration configuration, Boolean before) {
    Validate.notNull(configuration, "扫码配置信息未传");
    Set<ScanCodeEvent> scanCodeEvents = configuration.getScanCodeEvents();
    Validate.isTrue(!CollectionUtils.isEmpty(scanCodeEvents), "扫码事件信息未配置");
    String configurationId = configuration.getId();
    //如果为tre则可以覆盖全部事件
    this.scanCodeEventRepository.deleteByConfigurationId(configurationId);
    for (ScanCodeEvent scanCodeEvent : scanCodeEvents) {
      scanCodeEvent.setConfigurationId(configurationId);
      this.createValidation(scanCodeEvent);
    }
    this.scanCodeEventRepository.saveBatch(scanCodeEvents);
    for (ScanCodeEvent scanCodeEvent : scanCodeEvents) {
      this.scanCodeExpressionService.create(scanCodeEvent);
    }
  }

  /**
   * 保存校验
   * @param entity
   */
  private void createValidation(ScanCodeEvent entity) {
    Validate.notNull(entity, "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(entity.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    entity.setId(null);
    Validate.notBlank(entity.getConfigurationId(), "添加信息时，配置行ID 不能为空！");
    Validate.notBlank(entity.getParticipatorFlag(), "添加信息时，扫码行为主体(分利参与者 标记） 不能为空！");
    Validate.notBlank(entity.getParticipatorName(), "添加信息时，扫码行为主体(分利参与者 中文名）不能为空！");
    Validate.notBlank(entity.getBarCodeTypeFlag(), "添加信息时，条码类型（箱、瓶、盖 标记） 不能为空！");
    Validate.notBlank(entity.getBarCodeTypeName(), "添加信息时，条码类型（箱、瓶、盖 中文名） 不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况）
    Validate.isTrue(entity.getParticipatorFlag() == null || entity.getParticipatorFlag().length() < 16, "扫码行为主体(分利参与者 标记）,在进行添加时填入值超过了限定长度(32)，请检查!");
    Validate.isTrue(entity.getParticipatorName() == null || entity.getParticipatorName().length() < 16, "扫码行为主体(分利参与者 中文名）,在进行添加时填入值超过了限定长度(32)，请检查!");
    Validate.isTrue(entity.getBarCodeTypeFlag() == null || entity.getBarCodeTypeFlag().length() < 16, "条码类型（箱、瓶、盖 标记）,在进行添加时填入值超过了限定长度(32)，请检查!");
    Validate.isTrue(entity.getBarCodeTypeName() == null || entity.getBarCodeTypeName().length() < 16, "条码类型（箱、瓶、盖 中文名）,在进行添加时填入值超过了限定长度(32)，请检查!");

  }
}
