package com.biz.crm.cps.business.policy.scan.cycle.local.service.internal;

import com.biz.crm.cps.business.policy.scan.cycle.local.entity.ScanCodeEvent;
import com.biz.crm.cps.business.policy.scan.cycle.local.entity.ScanCodeExpression;
import com.biz.crm.cps.business.policy.scan.cycle.local.repository.ScanCodeExpressionRepository;
import com.biz.crm.cps.business.policy.scan.cycle.local.service.ScanCodeExpressionService;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.Set;

/**
 * 扫码表达式的service
 * @author hefan
 */
@Service
public class ScanCodeExpressionServiceImpl implements ScanCodeExpressionService {

  @Autowired
  private ScanCodeExpressionRepository scanCodeExpressionRepository;

  @Transactional
  @Override
  public void create(ScanCodeEvent scanCodeEvent) {
    Validate.notNull(scanCodeEvent, "扫码事件信息未传");
    Set<ScanCodeExpression> scanCodeExpressions = scanCodeEvent.getScanCodeExpressions();
    Validate.isTrue(!CollectionUtils.isEmpty(scanCodeExpressions), "扫码表达式信息未配置");
    for (ScanCodeExpression expression : scanCodeExpressions) {
      expression.setEventId(scanCodeEvent.getId());
      this.createValidation(expression);
    }
    this.scanCodeExpressionRepository.saveBatch(scanCodeExpressions);
    for (ScanCodeExpression scanCodeExpression : scanCodeExpressions) {
      Set<ScanCodeExpression> transferSet = scanCodeExpression.getScanCodeExpressions();
      if (CollectionUtils.isEmpty(transferSet)) {
        continue;
      }
      for (ScanCodeExpression transfer : transferSet) {
        transfer.setEventId(scanCodeEvent.getId());
        transfer.setExpressionId(scanCodeExpression.getId());
        this.createValidation(transfer);
      }
      this.scanCodeExpressionRepository.saveBatch(transferSet);
    }
  }

  /**
   * 保存校验
   * @param entity
   */
  private void createValidation(ScanCodeExpression entity) {
    Validate.notNull(entity, "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(entity.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    entity.setId(null);
    Validate.notBlank(entity.getEventId(), "添加信息时，扫码事件的ID 不能为空！");
    Validate.notBlank(entity.getRewardMethodFlag(), "添加信息时，奖励方式标志 不能为空！");
    Validate.notBlank(entity.getRewardMethodName(), "添加信息时，奖励方式名称 不能为空！");
    Validate.notBlank(entity.getAwardConditionFlag(), "添加信息时，奖励条件标志 不能为空！");
    Validate.notBlank(entity.getAwardConditionName(), "添加信息时，奖励条件名称 不能为空！");
    Validate.notBlank(entity.getRewardData(), "添加信息时，奖励数据 不能为空！");
    Validate.notBlank(entity.getRewardCycle(), "添加信息时，奖励周期 不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况）
    Validate.isTrue(entity.getEventId() == null || entity.getEventId().length() < 255, "扫码事件的ID ,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(entity.getExpressionId() == null || entity.getExpressionId().length() < 255, "上级id ,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(entity.getParticipatorFlag() == null || entity.getParticipatorFlag().length() < 16, "扫码行为主体(分利参与者 标记） ,在进行添加时填入值超过了限定长度(16)，请检查!");
    Validate.isTrue(entity.getParticipatorName() == null || entity.getParticipatorName().length() < 16, "扫码行为主体(分利参与者 中文名） ,在进行添加时填入值超过了限定长度(16)，请检查!");
    Validate.isTrue(entity.getRewardMethodFlag() == null || entity.getRewardMethodFlag().length() < 16, "奖励方式标志 ,在进行添加时填入值超过了限定长度(16)，请检查!");
    Validate.isTrue(entity.getRewardMethodName() == null || entity.getRewardMethodName().length() < 16, "奖励方式名称 ,在进行添加时填入值超过了限定长度(16)，请检查!");
    Validate.isTrue(entity.getAwardConditionFlag() == null || entity.getAwardConditionFlag().length() < 16, "奖励条件标志 ,在进行添加时填入值超过了限定长度(16)，请检查!");
    Validate.isTrue(entity.getAwardConditionName() == null || entity.getAwardConditionName().length() < 16, "奖励条件名称 ,在进行添加时填入值超过了限定长度(16)，请检查!");
    Validate.isTrue(entity.getRewardData() == null || entity.getRewardData().length() < 255, "奖励数据 ,在进行添加时填入值超过了限定长度(255)，请检查!");

  }
}
