package com.biz.crm.cps.business.policy.scan.cycle.local.service.internal;

import com.biz.crm.cps.business.policy.scan.cycle.local.entity.ScanCodeConfiguration;
import com.biz.crm.cps.business.policy.scan.cycle.local.entity.ScanCodeRange;
import com.biz.crm.cps.business.policy.scan.cycle.local.repository.ScanCodeRangeRepository;
import com.biz.crm.cps.business.policy.scan.cycle.local.service.ScanCodeRangeService;
import com.biz.crm.cps.business.product.sdk.common.constant.MaterialDimensionConstant;
import com.biz.crm.cps.business.product.sdk.service.MaterialVoService;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;


/**
 * 扫码范围的service
 * @author hefan
 */
@Service
public class ScanCodeRangeServiceImpl implements ScanCodeRangeService {

  @Autowired
  private ScanCodeRangeRepository scanCodeRangeRepository;

  @Autowired
  private MaterialVoService materialVoService;

  @Transactional
  @Override
  public void create(ScanCodeConfiguration current) {
    Validate.notNull(current, "扫码配置信息未传");
    String dimensionFlag = current.getDimensionFlag();
    List<String> dimensionCodes = Lists.newArrayList();
    // 产品范围为全部就不存数据
    if (MaterialDimensionConstant.DIMENSION_ALL.equals(dimensionFlag)) {
      return;
    }
    Set<ScanCodeRange> scanCodeRanges = current.getScanCodeRanges();
    Validate.isTrue(!CollectionUtils.isEmpty(scanCodeRanges), "没有配置产品范围");
    for (ScanCodeRange range : scanCodeRanges) {
      range.setConfigurationId(current.getId());
      dimensionCodes.add(range.getSpecialCode());
      this.createValidation(range);
    }
    scanCodeRangeRepository.saveBatch(scanCodeRanges);
    //异步调用物料同步方法
    CompletableFuture.runAsync(() -> {
      materialVoService.sync(dimensionCodes,dimensionFlag);
    });
  }

  /***
   * 根据配置行修改产品范围
   * @param configuration
   * @param before 当前时间是否在模板签署时间之前
   */
  @Override
  @Transactional
  public void update(ScanCodeConfiguration configuration, Boolean before) {
    Validate.notNull(configuration, "扫码配置信息未传");
    String dimensionFlag = configuration.getDimensionFlag();
    List<String> dimensionCodes = Lists.newArrayList();
    // 产品范围为全部就不存数据
    if (MaterialDimensionConstant.DIMENSION_ALL.equals(dimensionFlag)) {
      return;
    }
    Set<ScanCodeRange> scanCodeRanges = configuration.getScanCodeRanges();
    Validate.isTrue(!CollectionUtils.isEmpty(scanCodeRanges), "没有配置产品范围");
    //如果在模板签署开始时间之前则进行全部覆盖，反之只能增加不能删除
    String configurationId = configuration.getId();
    if (before) {
      //删除之前的数据
      this.scanCodeRangeRepository.deleteByConfigurationId(configurationId);
    } else {
      // 验证数据是否缺失,并筛选数据
      List<ScanCodeRange> byConfigurationId = this.scanCodeRangeRepository.findByConfigurationId(configurationId);
      if (!CollectionUtils.isEmpty(byConfigurationId)) {
        Set<String> specialCodes = byConfigurationId.stream().map(ScanCodeRange::getSpecialCode).collect(Collectors.toSet());
        Set<String> newSpecialCodes = scanCodeRanges.stream().map(ScanCodeRange::getSpecialCode).collect(Collectors.toSet());
        newSpecialCodes.retainAll(specialCodes);
        Validate.isTrue(newSpecialCodes.size() == specialCodes.size(), "在配置产品范围时，只能增加不能移除！");
        scanCodeRanges = scanCodeRanges.stream().filter(scanCodeRange -> !specialCodes.contains(scanCodeRange.getSpecialCode())).collect(Collectors.toSet());
      }
    }
    for (ScanCodeRange range : scanCodeRanges) {
      range.setConfigurationId(configurationId);
      dimensionCodes.add(range.getSpecialCode());
      this.createValidation(range);
    }
    scanCodeRangeRepository.saveBatch(scanCodeRanges);
    //异步调用物料同步方法
    CompletableFuture.runAsync(() -> {
      materialVoService.sync(dimensionCodes,dimensionFlag);
    });

  }

  /**
   * 保存校验
   * @param entity
   */
  private void createValidation(ScanCodeRange entity) {
    Validate.notNull(entity, "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(entity.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    entity.setId(null);
    Validate.notBlank(entity.getConfigurationId(), "添加信息时，配置行ID 不能为空！");
    Validate.notBlank(entity.getSpecialCode(), "添加信息时，特征编码 不能为空！");
    Validate.notBlank(entity.getSpecialName(), "添加信息时，特征名称 不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况）
    Validate.isTrue(entity.getSpecialCode() == null || entity.getSpecialCode().length() < 32, "特征编码,在进行添加时填入值超过了限定长度(32)，请检查!");
    Validate.isTrue(entity.getSpecialName() == null || entity.getSpecialName().length() < 255, "特征名称,在进行添加时填入值超过了限定长度(255)，请检查!");

  }


}
