package com.biz.crm.cps.business.policy.scan.local.service.internal;

import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.cps.business.agreement.sdk.dto.AgreementPolicyDto;
import com.biz.crm.cps.business.agreement.sdk.dto.ProfitAgreementTemplatePaginationDto;
import com.biz.crm.cps.business.agreement.sdk.service.AgreementTemplateVoService;
import com.biz.crm.cps.business.agreement.sdk.vo.AgreementTemplateVo;
import com.biz.crm.cps.business.agreement.sdk.vo.TemplateOrgRelationshipVo;
import com.biz.crm.cps.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.cps.business.policy.scan.local.entity.ScanCodeConfiguration;
import com.biz.crm.cps.business.policy.scan.local.entity.ScanCodeEvent;
import com.biz.crm.cps.business.policy.scan.local.entity.ScanCodeExpression;
import com.biz.crm.cps.business.policy.scan.local.entity.ScanCodeRange;
import com.biz.crm.cps.business.policy.scan.local.repository.ScanCodeConfigurationRepository;
import com.biz.crm.cps.business.policy.scan.local.service.ScanCodeConfigurationService;
import com.biz.crm.cps.business.policy.scan.local.service.ScanCodeEventService;
import com.biz.crm.cps.business.policy.scan.local.service.ScanCodeRangeService;
import com.biz.crm.cps.business.policy.scan.local.service.observer.ScanCodePolicyMountRegisterImpl;
import com.biz.crm.cps.business.policy.scan.sdk.dto.ScanCodeConfigurationDto;
import com.biz.crm.cps.business.policy.scan.sdk.dto.ScanCodeExpresionDto;
import com.biz.crm.cps.business.policy.scan.sdk.dto.ScanCodeParticipatorDto;
import com.biz.crm.cps.business.product.sdk.common.constant.MaterialDimensionConstant;
import com.biz.crm.cps.business.product.sdk.service.MaterialVoService;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * ScanCodeConfiguration 的 service实现类
 *
 * @author hefan
 */
@Service
public class ScanCodeConfigurationServiceImpl implements ScanCodeConfigurationService {

  @Autowired
  private ScanCodeConfigurationRepository scanCodeConfigurationRepository;

  @Autowired
  ScanCodeRangeService scanCodeRangeService;

  @Autowired
  ScanCodeEventService scanCodeEventService;

  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Autowired(required = false)
  private LoginUserService loginUserService;

  @Autowired
  private ScanCodePolicyMountRegisterImpl scanCodePolicyMountRegister;

  @Autowired
  private AgreementTemplateVoService agreementTemplateVoService;

  @Autowired
  private MaterialVoService materialVoService;

  @Transactional
  @Override
  public ScanCodeConfiguration create(AgreementPolicyDto agreementPolicyDto) {
    /**
     * 返回null，代表这档子事不归我管
     */
    // 从 request中获取特定对象
    ScanCodeConfiguration scanCodeConfiguration = this.mapping(agreementPolicyDto);
    if (Objects.isNull(scanCodeConfiguration)) {
      return null;
    }
    Set<ScanCodeEvent> scanCodeEvents = scanCodeConfiguration.getScanCodeEvents();
    if (CollectionUtils.isEmpty(scanCodeEvents)) {
      return null;
    }
    ScanCodeConfiguration current = this.createForm(scanCodeConfiguration);
    //====================================================
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    this.scanCodeConfigurationRepository.save(current);
    // 保存范围
    this.scanCodeRangeService.create(current);
    // 保存事件
    this.scanCodeEventService.create(current);
    return current;
  }

  @Override
  public List<ScanCodeConfiguration> findDetailsByTemplateCodes(Set<String> templateCodes) {
    if (CollectionUtils.isEmpty(templateCodes)) {
      return null;
    }
    String tenantCode = TenantUtils.getTenantCode();
    return this.scanCodeConfigurationRepository.findDetailsByTemplateCodes(templateCodes, tenantCode);
  }

  @Override
  public List<ScanCodeConfiguration> findDetailsByTemplateCode(String templateCode) {
    if (StringUtils.isBlank(templateCode)) {
      return null;
    }
    String tenantCode = TenantUtils.getTenantCode();
    return this.scanCodeConfigurationRepository.findDetailsByTemplateCode(templateCode, tenantCode);
  }

  /**
   * 创建一个新的 ScanCodeConfiguration模型对象
   */
  private ScanCodeConfiguration createForm(ScanCodeConfiguration scanCodeConfiguration) {
    /*
     * 对静态模型的保存操作过程为：
     * 1、如果当前模型对象不是主模型
     *  1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     *  1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     *  1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     * 2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *   2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *   2.3.2、以及验证每个分组的OneToMany明细信息
     * */
    Date now = new Date();
    String account = this.loginUserService.getLoginAccountName();
    scanCodeConfiguration.setTenantCode(TenantUtils.getTenantCode());
    scanCodeConfiguration.setCreateAccount(account);
    scanCodeConfiguration.setCreateTime(now);
    scanCodeConfiguration.setModifyAccount(account);
    scanCodeConfiguration.setModifyTime(now);
    this.createValidation(scanCodeConfiguration);
    // =========和业务有关的验证填写在这个区域======================
    // ===============================
    // 返回最终处理的结果，里面带有详细的关联信息
    return scanCodeConfiguration;
  }

  /**
   * 保存校验
   *
   * @param entity
   */
  private void createValidation(ScanCodeConfiguration entity) {
    Validate.notNull(entity, "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(entity.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    entity.setId(null);
    Validate.notBlank(entity.getTenantCode(), "添加信息时，租户编号不能为空！");
    Validate.notBlank(entity.getDimensionFlag(), "添加信息时，扫码产品维度标识不能为空！");
    Validate.notBlank(entity.getDimensionName(), "添加信息时，扫码产品维度名称不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况）
    Validate.isTrue(entity.getDimensionFlag() == null || entity.getDimensionFlag().length() < 32, "扫码产品维度标识,在进行添加时填入值超过了限定长度(32)，请检查!");
    Validate.isTrue(entity.getDimensionName() == null || entity.getDimensionName().length() < 32, "扫码产品维度名称,在进行添加时填入值超过了限定长度(32)，请检查!");
  }

  /**
   * 从字符串中获取扫码政策相关的信息
   *
   * @param agreementPolicyDto
   * @return
   */
  private ScanCodeConfiguration mapping(AgreementPolicyDto agreementPolicyDto) {
    if (agreementPolicyDto instanceof ScanCodeConfigurationDto) {
      String separator = "@";
      ScanCodeConfigurationDto dto = (ScanCodeConfigurationDto) agreementPolicyDto;
      ScanCodeConfiguration scanCodeConfiguration = this.nebulaToolkitService.copyObjectByBlankList(dto, ScanCodeConfiguration.class, HashSet.class, LinkedList.class);
      Set<ScanCodeParticipatorDto> scanCodeParticipators = dto.getScanCodeParticipators();
      Set<ScanCodeEvent> scanCodeEvents = Sets.newHashSet();
      for (ScanCodeParticipatorDto scanCodeParticipator : scanCodeParticipators) {
        Set<ScanCodeExpresionDto> scanCodeExpressionsDto = scanCodeParticipator.getScanCodeExpressions();
        Map<String, List<ScanCodeExpresionDto>> barCodeTypeflagMap = scanCodeExpressionsDto.stream().collect(Collectors.groupingBy(
            scanCodeExpresionDto -> StringUtils.join(scanCodeExpresionDto.getBarCodeTypeFlag(), separator, scanCodeExpresionDto.getBarCodeTypeName())
        ));
        // 主体 + 码类型 为一个事件
        for (Map.Entry<String, List<ScanCodeExpresionDto>> entry : barCodeTypeflagMap.entrySet()) {
          String key = entry.getKey();
          String[] split = key.split(separator);
          ScanCodeEvent scanCodeEvent = new ScanCodeEvent();
          scanCodeEvent.setParticipatorFlag(scanCodeParticipator.getParticipatorFlag());
          scanCodeEvent.setParticipatorName(scanCodeParticipator.getParticipatorName());
          scanCodeEvent.setBarCodeTypeFlag(split[0]);
          scanCodeEvent.setBarCodeTypeName(split[1]);
          Set<ScanCodeExpression> scanCodeExpressions = Sets.newHashSet(this.nebulaToolkitService.copyCollectionByBlankList(entry.getValue(), ScanCodeExpresionDto.class, ScanCodeExpression.class, HashSet.class, LinkedList.class));
          scanCodeEvent.setScanCodeExpressions(scanCodeExpressions);
          scanCodeEvents.add(scanCodeEvent);
        }
      }
      scanCodeConfiguration.setScanCodeEvents(scanCodeEvents);
      return scanCodeConfiguration;
    } else {
      return null;
    }

  }

  /**
   * 扫码政策范围重复校验
   * step1:获取当前有效期时间内，同组织的包含扫码政策的模板信息
   * step2:获取创建政策的物料编码集合
   * step3:查询交集模板对应的政策列表
   * step4:循环政策列表，获取物料列表并与创建政策的物料编码取交集，若交集存在则抛出异常
   *
   * @param agreementTemplateVo 扫码政策范围实体类
   */
  @Override
  public void validatePolicyScope(AgreementTemplateVo agreementTemplateVo, AgreementPolicyDto agreementPolicyDto) {

    ProfitAgreementTemplatePaginationDto templatePaginationDto = new ProfitAgreementTemplatePaginationDto();
    if (!CollectionUtils.isEmpty(agreementTemplateVo.getTemplateOrgRelationships())) {
      //如果协议模板设定了组织范围，则取组织范围放入查询参数
      List<String> orgCodes = agreementTemplateVo.getTemplateOrgRelationships().stream().map(TemplateOrgRelationshipVo::getOrgCode).collect(Collectors.toList());
      templatePaginationDto.setOrgCodes(orgCodes);
    }
    templatePaginationDto.setPolicyCode(this.scanCodePolicyMountRegister.getKey());
    templatePaginationDto.setEffectiveScopeStartTime(agreementTemplateVo.getEffectiveStartTime());
    templatePaginationDto.setEffectiveScopeEndTime(agreementTemplateVo.getEffectiveEndTime());
    templatePaginationDto.setStatus(EnableStatusEnum.ENABLE.getCode());
    //根据组织列表，生效时间查询已存在扫码政策的协议模板
    List<AgreementTemplateVo> templateVos = this.agreementTemplateVoService.findByConditions(templatePaginationDto);
    if (CollectionUtils.isEmpty(templateVos)) {
      //如果没有找到交集模板，则校验通过
      return;
    }
    ScanCodeConfiguration scanCodeConfiguration = nebulaToolkitService.copyObjectByWhiteList(agreementPolicyDto, ScanCodeConfiguration.class, HashSet.class, ArrayList.class, "scanCodeRanges");
    //通过交集模板列表查询对应的扫码
    List<String> templateCodes = templateVos.stream().map(AgreementTemplateVo::getTemplateCode).collect(Collectors.toList());
    List<ScanCodeConfiguration> scanCodeConfigurations = this.scanCodeConfigurationRepository.findDetailsByTemplateCodes(new HashSet<String>(templateCodes), TenantUtils.getTenantCode());
    if (CollectionUtils.isEmpty(scanCodeConfigurations)) {
      //如果没有找到扫码政策，则校验通过
      return;
    }
    //获取本次创建政策包含物料编码
    List<String> createPolicyMaterialCodes = this.policyRangeValidation(scanCodeConfiguration);
    //如果本次创建政策包含物料编码为空，直接返回
    if (CollectionUtils.isEmpty(createPolicyMaterialCodes)) {
      return;
    }
    //循环交集模板列表，获取对应包含的物料编码列表，并求交集
    scanCodeConfigurations.forEach(policy -> {
      List<String> materialCodes = this.policyRangeValidation(policy);
      if (!CollectionUtils.isEmpty(materialCodes)) {
        //当政策的物料编码不为空，取政策物料编码和创建政策的物料编码的合集
        materialCodes.retainAll(createPolicyMaterialCodes);
        Validate.isTrue(CollectionUtils.isEmpty(materialCodes), "本次创建扫码产品物料已经创建过本时段扫码政策！");
      }
    });
  }

  /**
   * 获取政策配置的物料编码列表
   * 此方法扫码政策验重使用，前提为本次创建模板已经在数据库中查询到组织-时间有交集的模板信息
   * 故此方法内部会校验产品维度不能为全部，当产品维度为全部时抛出异常
   * 返回政策下所有配置的物料编码集合
   *
   * @param scanCodeConfiguration 扫码政策
   */
  private List<String> policyRangeValidation(ScanCodeConfiguration scanCodeConfiguration) {
    List<String> materialCodes = Lists.newArrayList();
    Validate.isTrue(!MaterialDimensionConstant.DIMENSION_ALL.equals(scanCodeConfiguration.getDimensionFlag()), "同时间段内有模板配置的扫码政策为全部商品！");
    Validate.isTrue(!CollectionUtils.isEmpty(scanCodeConfiguration.getScanCodeRanges()), "扫码政策必须包含产品范围！");
    List<String> materialCodeList = this.materialVoService.findMaterialCodeByDimensionCodesAndDimensionType(scanCodeConfiguration.getScanCodeRanges().stream().map(ScanCodeRange::getSpecialCode).collect(Collectors.toList()), scanCodeConfiguration.getDimensionFlag());
    if (!CollectionUtils.isEmpty(materialCodeList)) {
      materialCodes.addAll(materialCodeList);
    }
    return materialCodes;
  }

}
