package com.biz.crm.cps.business.policy.scan.local.service.observer;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.biz.crm.cps.business.agreement.sdk.dto.AgreementPolicyDto;
import com.biz.crm.cps.business.agreement.sdk.dto.PolicyObserverDto;
import com.biz.crm.cps.business.agreement.sdk.service.observer.AgreementPolicyMountRegister;
import com.biz.crm.cps.business.agreement.sdk.service.observer.AgreementPolicyPojoObserver;
import com.biz.crm.cps.business.agreement.sdk.service.observer.AgreementPolicyServiceObserver;
import com.biz.crm.cps.business.agreement.sdk.vo.AgreementPolicyVo;
import com.biz.crm.cps.business.agreement.sdk.vo.AgreementTemplateVo;
import com.biz.crm.cps.business.agreement.sdk.vo.PolicyRewardDataVo;
import com.biz.crm.cps.business.policy.scan.local.entity.ScanCodeConfiguration;
import com.biz.crm.cps.business.policy.scan.local.entity.ScanCodeEvent;
import com.biz.crm.cps.business.policy.scan.local.entity.ScanCodeExpression;
import com.biz.crm.cps.business.policy.scan.local.entity.ScanCodeRange;
import com.biz.crm.cps.business.policy.scan.local.service.ScanCodeConfigurationService;
import com.biz.crm.cps.business.policy.scan.sdk.common.enums.ScanCodeActionEnum;
import com.biz.crm.cps.business.policy.scan.sdk.vo.ScanCodeConfigurationVo;
import com.biz.crm.cps.business.policy.scan.sdk.vo.ScanCodeExpresionVo;
import com.biz.crm.cps.business.policy.scan.sdk.vo.ScanCodeParticipatorVo;
import com.biz.crm.cps.business.policy.scan.sdk.vo.ScanCodePolicyVo;
import com.biz.crm.cps.business.policy.sdk.dto.PolicyRewardConditionDto;
import com.biz.crm.cps.business.policy.sdk.service.observer.PolicyRewardServiceObserver;
import com.biz.crm.cps.business.product.sdk.common.constant.MaterialDimensionConstant;
import com.biz.crm.cps.business.product.sdk.dto.MaterialDimensionDto;
import com.biz.crm.cps.business.product.sdk.dto.ProductDimensionDto;
import com.biz.crm.cps.business.product.sdk.service.MaterialVoService;
import com.biz.crm.cps.business.product.sdk.service.ProductVoService;
import com.biz.crm.cps.external.barcode.sdk.common.enums.BarCodeTypeEnum;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.JsonUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 扫码政策，提供协议模板要求政策提供的API
 * @author hefan
 */
@Component("ScanCodePolicyServiceObserverImpl")
public class ScanCodePolicyServiceObserverImpl implements AgreementPolicyServiceObserver {

  @Autowired
  private ScanCodeConfigurationService scanCodeConfigurationService;

  @Autowired(required = false)
  private List<PolicyRewardServiceObserver> policyRewardServiceObservers;

  @Autowired
  @Qualifier("ScanCodePolicyPojoObserverImpl")
  private AgreementPolicyPojoObserver agreementPolicyPojoObserver;

  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Autowired(required = false)
  private ProductVoService productVoService;

  @Autowired(required = false)
  private MaterialVoService materialVoService;

  @Autowired
  @Qualifier("ScanCodePolicyMountRegisterImpl")
  private AgreementPolicyMountRegister agreementPolicyMountRegister;

  /**
   * 保存扫码政策信息到数据库
   * @param agreementPolicyDto
   */
  @Override
  public void onCreate(AgreementPolicyDto agreementPolicyDto) {
    this.scanCodeConfigurationService.create(agreementPolicyDto);
  }

  @Override
  public AgreementPolicyVo onFindDetailsByTemplateCode(String templateCode) {
    // 查询了数据库里的信息，但是表达式是没有层级关系的
    List<ScanCodeConfiguration> scanCodeConfigurations = this.scanCodeConfigurationService.findDetailsByTemplateCode(templateCode);
    if (CollectionUtils.isEmpty(scanCodeConfigurations)) {
      return null;
    }
    ScanCodePolicyVo scanCodePolicyVo = new ScanCodePolicyVo();
    List<ScanCodeConfigurationVo> scanCodeConfigurationVos = Lists.newLinkedList();
    for (ScanCodeConfiguration scanCodeConfiguration : scanCodeConfigurations) {
      // 把entity转换成vo
      ScanCodeConfigurationVo scanCodeConfigurationVo = this.entityToVo(scanCodeConfiguration);
      scanCodeConfigurationVos.add(scanCodeConfigurationVo);
    }
    scanCodePolicyVo.setScanCodeConfigurationVos(scanCodeConfigurationVos);
    scanCodePolicyVo.setPolicyKey(this.agreementPolicyMountRegister.getKey());
    scanCodePolicyVo.setTemplateCode(templateCode);
    return scanCodePolicyVo;
  }

  /**
   * 扫码政策配置，把entity转换成vo
   * @param scanCodeConfiguration
   * @return
   */
  private ScanCodeConfigurationVo entityToVo(ScanCodeConfiguration scanCodeConfiguration) {
    ScanCodeConfigurationVo vo = this.nebulaToolkitService.copyObjectByBlankList(scanCodeConfiguration, ScanCodeConfigurationVo.class, HashSet.class, LinkedList.class);
    String separator = "@";
    Set<ScanCodeParticipatorVo> scanCodeParticipatorVos = Sets.newLinkedHashSet();
    Set<ScanCodeEvent> scanCodeEvents = scanCodeConfiguration.getScanCodeEvents();
    // 把扫码事件，按主体分组
    Map<String, List<ScanCodeEvent>> participatorMap = scanCodeEvents.stream().collect(Collectors.groupingBy(
        scanCodeEvent -> StringUtils.join(scanCodeEvent.getParticipatorFlag(), separator, scanCodeEvent.getParticipatorName())
    ));
    // 遍历多少次，就有多少个主体
    for (Map.Entry<String, List<ScanCodeEvent>> entry : participatorMap.entrySet()) {
      String participator = entry.getKey();
      String[] split = participator.split(separator);
      ScanCodeParticipatorVo scanCodeParticipatorVo = new ScanCodeParticipatorVo();
      scanCodeParticipatorVo.setParticipatorFlag(split[0]);
      scanCodeParticipatorVo.setParticipatorName(split[1]);
      // 事件的 箱、瓶标识要放到奖励表达式中
      Set<ScanCodeExpresionVo> scanCodeExpresionVos = this.getScanCodeExpresionVos(entry.getValue());
      scanCodeParticipatorVo.setScanCodeExpressions(scanCodeExpresionVos);
      scanCodeParticipatorVos.add(scanCodeParticipatorVo);
    }
    vo.setScanCodeParticipators(scanCodeParticipatorVos);
    return vo;

  }

  /**
   * 转换外层扫码表达式vo
   * @param value
   * @return
   */
  private Set<ScanCodeExpresionVo> getScanCodeExpresionVos(List<ScanCodeEvent> value) {
    Set<ScanCodeExpresionVo> scanCodeExpresionVos = Sets.newLinkedHashSet();
    for (ScanCodeEvent scanCodeEvent : value) {
      // 这里的表达式是没有层级关系的
      Set<ScanCodeExpression> scanCodeExpressions = scanCodeEvent.getScanCodeExpressions();
      // 把表达式 内外分开
      Map<Boolean, List<ScanCodeExpression>> expressionIdMap = scanCodeExpressions.stream().collect(Collectors.partitioningBy(
          scanCodeExpression -> StringUtils.isNotBlank(scanCodeExpression.getExpressionId())
      ));
      List<ScanCodeExpression> outerLayerExpressions = expressionIdMap.get(false);
      List<ScanCodeExpression> transferExpressions = expressionIdMap.get(true);
      // 把内层的传递表达式按外层表达式ID分组
      Map<String, List<ScanCodeExpression>> expressionIdTransferMap = transferExpressions.stream().collect(Collectors.groupingBy(
          ScanCodeExpression::getExpressionId
      ));
      // 遍历外层表达式
      for (ScanCodeExpression outerLayerExpression : outerLayerExpressions) {
        ScanCodeExpresionVo scanCodeExpresionVo = this.nebulaToolkitService.copyObjectByBlankList(outerLayerExpression, ScanCodeExpresionVo.class, HashSet.class, LinkedList.class);
        scanCodeExpresionVo.setBarCodeTypeFlag(scanCodeEvent.getBarCodeTypeFlag());
        scanCodeExpresionVo.setBarCodeTypeName(scanCodeEvent.getBarCodeTypeName());
        // 获取传递奖励表达式
        List<ScanCodeExpresionVo> transferScanCodeExpresionVos = this.getScanCodeExpresionVos(scanCodeEvent, expressionIdTransferMap, outerLayerExpression);
        if (!CollectionUtils.isEmpty(transferScanCodeExpresionVos)) {
          scanCodeExpresionVo.setScanCodeExpressions(Sets.newLinkedHashSet(transferScanCodeExpresionVos));
        }
        scanCodeExpresionVos.add(scanCodeExpresionVo);
      }
    }
    return scanCodeExpresionVos;
  }

  /**
   * 转换传递奖励表达式VO
   * @param scanCodeEvent
   * @param expressionIdTransferMap
   * @param outerLayerExpression
   * @return
   */
  private List<ScanCodeExpresionVo> getScanCodeExpresionVos(ScanCodeEvent scanCodeEvent, Map<String, List<ScanCodeExpression>> expressionIdTransferMap, ScanCodeExpression outerLayerExpression) {
    List<ScanCodeExpression> expressionIdTransferExpressions = expressionIdTransferMap.get(outerLayerExpression.getId());
    if (CollectionUtils.isEmpty(expressionIdTransferExpressions)) {
      return null;
    }
    List<ScanCodeExpresionVo> transferScanCodeExpresionVos = (List<ScanCodeExpresionVo>) this.nebulaToolkitService.copyCollectionByBlankList(expressionIdTransferExpressions, ScanCodeExpression.class, ScanCodeExpresionVo.class, HashSet.class, LinkedList.class);
    for (ScanCodeExpresionVo transferScanCodeExpresionVo : transferScanCodeExpresionVos) {
      transferScanCodeExpresionVo.setBarCodeTypeFlag(scanCodeEvent.getBarCodeTypeFlag());
      transferScanCodeExpresionVo.setBarCodeTypeName(scanCodeEvent.getBarCodeTypeName());
    }
    return transferScanCodeExpresionVos;
  }

  @Override
  public void onProcessScanCode(PolicyObserverDto policyObserverDto) {
    List<PolicyRewardConditionDto> policyRewardConditionDtos = this.findPolicyConfig(policyObserverDto);
    // 公共信息补充 比如：实际扫码者，触发动作，触发对象
    String participatorCode = policyObserverDto.getParticipatorCode();
    String participatorType = policyObserverDto.getParticipatorType();
    String triggerAction = this.judgeTriggerAction(policyObserverDto);
    String triggerObject = policyObserverDto.getProductCode();
    String recordCode = policyObserverDto.getRecordCode();

    this.setCommonProperty(policyRewardConditionDtos, participatorCode, participatorType, triggerAction, triggerObject, recordCode);
    //将分利规则集合，告知流水模块组的观察者们。
    this.callRewardObservers(policyRewardConditionDtos);
  }

  private String judgeTriggerAction(PolicyObserverDto policyObserverDto) {
    String triggerAction = "系统内无有效动作";
    // 1-箱码，2-盒码，3-盖内码
    String barCodeType = policyObserverDto.getBarCodeType();
    if (BarCodeTypeEnum.BOX.getFlag().equals(barCodeType)) {
      triggerAction = ScanCodeActionEnum.BOX_CODE.getValue();
    } else if (BarCodeTypeEnum.BOTTLE.getFlag().equals(barCodeType)) {
      triggerAction = ScanCodeActionEnum.CASE_CODE.getValue();
    } else if (BarCodeTypeEnum.COVER_CODE.getFlag().equals(barCodeType)) {
      triggerAction = ScanCodeActionEnum.COVER_CODE.getValue();
    }
    return triggerAction;
  }

  /**
   * 将公共属性值set注入
   * @param policyRewardConditionDtos
   * @param participatorCode
   * @param participatorType
   */
  private void setCommonProperty(List<PolicyRewardConditionDto> policyRewardConditionDtos, String participatorCode, String participatorType, String triggerAction, String triggerObject, String recordCode) {
    for (PolicyRewardConditionDto rewardConditionDto : policyRewardConditionDtos) {
      rewardConditionDto.setPolicyName(this.agreementPolicyMountRegister.getName());
      rewardConditionDto.setActualParticipatorCode(participatorCode);
      rewardConditionDto.setActualParticipatorFlag(participatorType);
      rewardConditionDto.setTriggerAction(triggerAction);
      rewardConditionDto.setTriggerObject(triggerObject);
      rewardConditionDto.setRecordCode(recordCode);
      List<PolicyRewardConditionDto> conditionDtos = rewardConditionDto.getPolicyRewardConditionDtos();
      if (!CollectionUtils.isEmpty(conditionDtos)) {
        this.setCommonProperty(conditionDtos, participatorCode, participatorType, triggerAction, triggerObject,recordCode);
      }
    }
  }

  @Override
  public void onCreate(JSONObject jsonObject) {
    JSONArray jsonArray = jsonObject.getJSONArray(this.agreementPolicyMountRegister.getKey());
    if (CollectionUtils.isEmpty(jsonArray)) {
      return;
    }
    for (Object o : jsonArray) {
      JSONObject json = JsonUtils.toJSONObject(o.toString());
      AgreementPolicyDto deserialize = this.agreementPolicyPojoObserver.deserialize(json);
      deserialize.setTemplateCode(jsonObject.getString("templateCode"));
      //政策验重
      AgreementTemplateVo agreementTemplateVo = JsonUtils.json2Obj(jsonObject.getString("profitAgreementTemplate"), AgreementTemplateVo.class);
      this.scanCodeConfigurationService.validatePolicyScope(agreementTemplateVo, deserialize);
      this.onCreate(deserialize);
    }
  }

  @Override
  public List<PolicyRewardDataVo> onFindPolicyRewardData(Set<String> templateCodeSet, String usertype) {
    /**
     * 查询这些协议模板的奖励条件
     */
    List<PolicyRewardDataVo> policyRewardDataVos = Lists.newLinkedList();
    List<ScanCodeConfiguration> scanCodeConfigurations = this.scanCodeConfigurationService.findDetailsByTemplateCodes(templateCodeSet);
    for (ScanCodeConfiguration scanCodeConfiguration : scanCodeConfigurations) {
      Set<ScanCodeRange> scanCodeRanges = scanCodeConfiguration.getScanCodeRanges();
      Set<ScanCodeEvent> scanCodeEvents = scanCodeConfiguration.getScanCodeEvents();
      for (ScanCodeEvent scanCodeEvent : scanCodeEvents) {
        // 判断事件是不是登录者类型
        String participatorFlag = scanCodeEvent.getParticipatorFlag();
        if (!participatorFlag.equals(usertype)) {
          continue;
        }
        Set<ScanCodeExpression> scanCodeExpressions = scanCodeEvent.getScanCodeExpressions();
        // expressionId 为空的是外层条件，有值的是传递条件
        Map<Boolean, List<ScanCodeExpression>> booleanMap = scanCodeExpressions.stream().collect(
            Collectors.partitioningBy((ScanCodeExpression exp) -> StringUtils.isNotBlank(exp.getExpressionId()))
        );
        List<ScanCodeExpression> expressionList = booleanMap.get(false);
        if (CollectionUtils.isEmpty(expressionList)){
          continue;
        }
        for (ScanCodeExpression expression : expressionList) {
          // 每箱固定奖励积分
          String barCodeTypeName = scanCodeEvent.getBarCodeTypeName();
          String awardConditionName = expression.getAwardConditionName();
          String rewardMethodName = expression.getRewardMethodName();
          String condition = StringUtils.join("每", barCodeTypeName, awardConditionName, rewardMethodName);
          // TODO 将json转成一句话
          String rewardData = expression.getRewardData();
          if (CollectionUtils.isEmpty(scanCodeRanges)) {
            PolicyRewardDataVo policyRewardDataVo = new PolicyRewardDataVo();
            policyRewardDataVo.setSpecialName(scanCodeConfiguration.getDimensionName());
            policyRewardDataVo.setCondition(condition);
            policyRewardDataVo.setRewardData(rewardData);
            policyRewardDataVos.add(policyRewardDataVo);
          } else {
            for (ScanCodeRange scanCodeRange : scanCodeRanges) {
              PolicyRewardDataVo policyRewardDataVo = new PolicyRewardDataVo();
              policyRewardDataVo.setSpecialName(scanCodeRange.getSpecialName());
              policyRewardDataVo.setCondition(condition);
              policyRewardDataVo.setRewardData(rewardData);
              policyRewardDataVos.add(policyRewardDataVo);
            }
          }
        }
      }
    }
    return policyRewardDataVos;
  }

  @Override
  public void onChange(JSONObject jsonObject) {

  }

  /**
   * 查询扫码政策的配置信息。
   * 返回奖励模块组需要的dto
   * @param policyObserverDto
   * @return
   */
  private List<PolicyRewardConditionDto> findPolicyConfig(PolicyObserverDto policyObserverDto) {
    /**
     * 根据配置模板集合查询【扫码政策配置】 --入参
     * 根据维度筛选【配置行】
     * 根据主体，码类型 筛选奖励条件 -- 入参
     * 返回奖励条件集合
     */
    //根据配置模板集合查询【扫码政策配置】
    Set<String> templateCodes = policyObserverDto.getTemplateCodes();
    List<ScanCodeConfiguration> scanCodeConfigurations = this.scanCodeConfigurationService.findDetailsByTemplateCodes(templateCodes);

    // 根据维度筛选【配置行】--------逐行分析
    List<PolicyRewardConditionDto> policyRewardConditionDtos = new LinkedList<>();
    for (ScanCodeConfiguration scanCodeConfiguration : scanCodeConfigurations) {
      // 产品维度
      String dimensionFlag = scanCodeConfiguration.getDimensionFlag();
      // 配置行里设定的产品范围
      Set<ScanCodeRange> scanCodeRanges = scanCodeConfiguration.getScanCodeRanges();
      // 判断商品在不在这个范围里
      //ProductDimensionDto productDimensionDto = policyObserverDto.getProductDimensionDto();
      // 判断物料在不在这个范围里
      MaterialDimensionDto materialDimensionDto=policyObserverDto.getMaterialDimensionDto();
      boolean matchRange = this.matchRange(scanCodeRanges, materialDimensionDto, dimensionFlag);
      //根据主体，码类型 筛选奖励条件
      if (matchRange) {
        List<PolicyRewardConditionDto> conditionDtos = this.filterRewardConditions(policyObserverDto, scanCodeConfiguration);
        policyRewardConditionDtos.addAll(conditionDtos);
      }
    }
    return policyRewardConditionDtos;
  }

  /**
   * 根据主体，码类型 筛选奖励条件
   * @param policyObserverDto
   * @param scanCodeConfiguration
   */
  private List<PolicyRewardConditionDto> filterRewardConditions(PolicyObserverDto policyObserverDto, ScanCodeConfiguration scanCodeConfiguration) {
    Set<ScanCodeEvent> scanCodeEvents = scanCodeConfiguration.getScanCodeEvents();
    String templateCode = scanCodeConfiguration.getTemplateCode();
    if (CollectionUtils.isEmpty(scanCodeEvents)) {
      return Lists.newLinkedList();
    }
    List<PolicyRewardConditionDto> policyRewardConditionDtos = new LinkedList<>();
    for (ScanCodeEvent scanCodeEvent : scanCodeEvents) {
      String barCodeTypeFlag = scanCodeEvent.getBarCodeTypeFlag();
      String participatorFlag = scanCodeEvent.getParticipatorFlag();
      String participatorType = policyObserverDto.getParticipatorType();
      String barCodeType = policyObserverDto.getBarCodeType();
      // 参与者匹配 并且 码类型匹配 fixme 扫了箱，但是只配置了盒。是不是要按扫了n次盒来算
      if (barCodeTypeFlag.equals(barCodeType) && participatorFlag.equals(participatorType)) {
        // 【大步骤】将找到的表达式，转换成 dto
        Set<ScanCodeExpression> scanCodeExpressions = scanCodeEvent.getScanCodeExpressions();
        // expressionId 为空的是外层条件，有值的是传递条件
        Map<Boolean, List<ScanCodeExpression>> booleanMap = scanCodeExpressions.stream().collect(
            Collectors.partitioningBy((ScanCodeExpression exp) -> StringUtils.isNotBlank(exp.getExpressionId()))
        );
        List<ScanCodeExpression> expressionList = booleanMap.get(false);
        if (CollectionUtils.isEmpty(expressionList)) {
          continue;
        }
        List<ScanCodeExpression> transferList = booleanMap.get(true);
        // 将传递表达式按expressionId分组
        Map<String, List<ScanCodeExpression>> expressionIdMap = transferList.stream().collect(Collectors.groupingBy(ScanCodeExpression::getExpressionId));
        // 遍历外层表达式
        for (ScanCodeExpression scanCodeExpression : expressionList) {
          // 将ScanCodeExpression 转换成 dto
          PolicyRewardConditionDto outerLayerDto = this.transform(scanCodeExpression);
          List<ScanCodeExpression> transferScanCodeExpression = expressionIdMap.get(scanCodeExpression.getId());
          // 将传递表达式转换成dto
          if (!CollectionUtils.isEmpty(transferScanCodeExpression)) {
            List<PolicyRewardConditionDto> transferDtos = this.transformList(transferScanCodeExpression);
            transferDtos.forEach(transferDto -> transferDto.setTemplateCode(templateCode));
            outerLayerDto.setPolicyRewardConditionDtos(transferDtos);
          }
          outerLayerDto.setAgreementCode(null);
          outerLayerDto.setTemplateCode(templateCode);
          //外部dto.set
          policyRewardConditionDtos.add(outerLayerDto);
        }
      }
    }
    return policyRewardConditionDtos;
  }

  /**
   *
   * @param transferScanCodeExpression
   * @return
   */
  private List<PolicyRewardConditionDto> transformList(List<ScanCodeExpression> transferScanCodeExpression) {
    List<PolicyRewardConditionDto> list = (List<PolicyRewardConditionDto>) this.nebulaToolkitService.copyCollectionByWhiteList(transferScanCodeExpression, ScanCodeExpression.class, PolicyRewardConditionDto.class, HashSet.class, LinkedList.class);
    return list;
  }

  /**
   *
   * @param scanCodeExpression
   * @return
   */
  private PolicyRewardConditionDto transform(ScanCodeExpression scanCodeExpression) {
    return this.nebulaToolkitService.copyObjectByWhiteList(scanCodeExpression, PolicyRewardConditionDto.class, HashSet.class, LinkedList.class);
  }

  /**
   * 判断商品在不在这个范围里
   * @param scanCodeRanges
   * @param productDimensionDto
   * @return
   */
  private boolean matchRange(Set<ScanCodeRange> scanCodeRanges, ProductDimensionDto productDimensionDto, String dimensionFlag) {
    // 全选
    if (CollectionUtils.isEmpty(scanCodeRanges)) {
      return true;
    } else {
      // 根据维度选择一个商品对应维度code
      String specialCode = this.findSpecialCodeByDimension(dimensionFlag, productDimensionDto);
      for (ScanCodeRange scanCodeRange : scanCodeRanges) {
        if (StringUtils.isNotBlank(specialCode) && specialCode.equals(scanCodeRange.getSpecialCode())) {
          return true;
        }
      }
    }
    return false;
  }

  /**
   * 判断商品在不在这个范围里
   * @param scanCodeRanges
   * @param materialDimensionDto
   * @return
   */
  private boolean matchRange(Set<ScanCodeRange> scanCodeRanges, MaterialDimensionDto materialDimensionDto, String dimensionFlag) {
    // 全选
    if (CollectionUtils.isEmpty(scanCodeRanges)) {
      return true;
    } else {
      // 根据维度选择一个商品对应维度code
      List<String> specialCodes = this.findSpecialCodeByDimension(dimensionFlag, materialDimensionDto);
      for (ScanCodeRange scanCodeRange : scanCodeRanges) {
        if (!CollectionUtils.isEmpty(specialCodes) && (specialCodes.contains(MaterialDimensionConstant.DIMENSION_ALL) || specialCodes.contains(scanCodeRange.getSpecialCode()))) {
          return true;
        }
      }
    }
    return false;
  }
  /**
   * 根据维度选择一个商品对应维度code
   * @param dimensionFlag
   * @param productDimensionDto
   * @return
   */
  private String findSpecialCodeByDimension(String dimensionFlag, ProductDimensionDto productDimensionDto) {
    return productVoService.getDimensionCode(productDimensionDto, dimensionFlag);
  }

  /**
   * 根据维度选择一个物料对应维度code
   * @param dimensionFlag
   * @param materialDimensionDto
   * @return
   */
  private List<String> findSpecialCodeByDimension(String dimensionFlag, MaterialDimensionDto materialDimensionDto) {
    return materialVoService.findDimensionCodes(materialDimensionDto, dimensionFlag);
  }

  /**
   * 调用奖励模块组内的PolicyRewardServiceObserver实现类
   * 记录奖励流水
   * @param policyRewardConditionDtos
   */
  private void callRewardObservers(List<PolicyRewardConditionDto> policyRewardConditionDtos) {
    if (CollectionUtils.isEmpty(policyRewardServiceObservers)) {
      return;
    }
    for (PolicyRewardServiceObserver rewardServiceObserver : policyRewardServiceObservers) {
      rewardServiceObserver.createRewardRecord(policyRewardConditionDtos);
    }
  }

}
