package com.biz.crm.cps.business.price.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.cps.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.cps.business.price.local.entity.ControlProductPriceEntity;
import com.biz.crm.cps.business.price.local.repository.ControlProductPriceRepository;
import com.biz.crm.cps.business.price.local.service.ControlProductPriceService;
import com.biz.crm.cps.business.price.sdk.dto.ControlProductPriceDto;
import com.biz.crm.cps.business.price.sdk.event.ControlProductPriceEventListener;
import com.biz.crm.cps.business.price.sdk.vo.ControlProductPriceOrgMappingVo;
import com.biz.crm.cps.business.price.sdk.vo.ControlProductPriceVo;
import com.biz.crm.cps.business.price.sdk.vo.ProductPriceVo;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

/**
 * 控价商品定价接口实现
 * @Author: zengxingwang
 * @Date: 2021/7/20 10:59
 */
@Service
@ConditionalOnMissingBean(name = "ControlProductPriceExpandImpl")
public class ControlProductPriceServiceImpl implements ControlProductPriceService {

  @Autowired
  private ControlProductPriceRepository controlProductPriceRepository;

  @Autowired(required = false)
  private List<ControlProductPriceEventListener> controlProductPriceEventListeners;

  /**
   * 配置控价商品
   * @param dto
   * @return
   */
  @Override
  @Transactional
  public void create(ControlProductPriceDto dto) {
    this.validate(dto);
    Set<ControlProductPriceOrgMappingVo> productPriceOrgMappings = dto.getProductPriceOrgMappings();
    String SaleOrgName = "";
    String SaleOrgCode = "";
    for(ControlProductPriceOrgMappingVo productPriceOrgMapping : productPriceOrgMappings){
      SaleOrgName = StringUtils.join(SaleOrgName, ",", productPriceOrgMapping.getSaleOrgName());
      SaleOrgCode = StringUtils.join(SaleOrgCode, ",", productPriceOrgMapping.getSaleOrgCode());
    }
    Set<ProductPriceVo> products = dto.getProducts();
    List<String> materialCodes = new ArrayList<>();
    for(ProductPriceVo vo : products){
      materialCodes.add(vo.getMaterialCode());
    }
    long count = controlProductPriceRepository.findByMaterialCodeIn(materialCodes);
    Validate.isTrue(count == 0L, "配置已存在,请检查");

    List<ControlProductPriceEntity> productList = new ArrayList<>();
    for(ProductPriceVo productPriceVo : products){
      ControlProductPriceEntity entity = new ControlProductPriceEntity();
      BeanUtils.copyProperties(productPriceVo, entity);
      //去掉字符串的第一个字符逗号
      entity.setSaleOrgName(SaleOrgName.substring(1));
      entity.setSaleOrgCode(SaleOrgCode.substring(1));
      entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
      productList.add(entity);
    }
    this.controlProductPriceRepository.saveBatch(productList);
    if (controlProductPriceEventListeners != null) {
      for (ControlProductPriceEventListener eventListener : controlProductPriceEventListeners) {
        eventListener.onCreate(materialCodes);
      }
    }
  }

  /**
   * 根据id启用
   * @param ids
   */
  @Override
  @Transactional
  public void enable(List<String> ids) {
    Validate.notEmpty(ids, "启用时，主键不能为空");
    this.controlProductPriceRepository.updateEnableStatusByIdIn(EnableStatusEnum.ENABLE, ids);
  }

  /**
   * 根据id禁用
   * @param ids
   */
  @Override
  @Transactional
  public void disable(List<String> ids) {
    Validate.notEmpty(ids, "禁用时，id不能为空");
    List<ControlProductPriceEntity> controlProductPriceEntities = controlProductPriceRepository.findByIds(ids);
    if(CollectionUtils.isNotEmpty(controlProductPriceEntities)){
      List<String> materialCodes = new ArrayList<>();
      for(ControlProductPriceEntity entity : controlProductPriceEntities){
        materialCodes.add(entity.getMaterialCode());
      }
      //todo 事件验证是否可禁用
      if (controlProductPriceEventListeners != null) {
        for (ControlProductPriceEventListener eventListener : controlProductPriceEventListeners) {
          Boolean bool = eventListener.onDisable(materialCodes);
          Validate.isTrue(bool, "不可禁用");
        }
      }
    }
    this.controlProductPriceRepository.updateEnableStatusByIdIn(EnableStatusEnum.DISABLE, ids);
  }

  /**
   * 条件查询控价商品列表
   * @param dto
   * @return
   */
  @Override
  public Page<ControlProductPriceVo> findByConditions(ControlProductPriceDto dto, Pageable pageable) {
    return controlProductPriceRepository.findByConditions(dto, pageable);
  }

  /**
   * 校验数据
   * @param dto
   */
  private void validate(ControlProductPriceDto dto) {
    Validate.notNull(dto, "配置控价产品时，数据不能为空");

    Set<ControlProductPriceOrgMappingVo> productPriceOrgMappings = dto.getProductPriceOrgMappings();
    Validate.notEmpty(productPriceOrgMappings, "销售区域不能为空");
    for(ControlProductPriceOrgMappingVo productPriceOrgMapping : productPriceOrgMappings) {
      Validate.notBlank(productPriceOrgMapping.getSaleOrgCode(), "区域编码不能为空");
      Validate.notBlank(productPriceOrgMapping.getSaleOrgName(), "区域名称不能为空");
      //todo 验证组织是否有效
    }

    Set<ProductPriceVo> products = dto.getProducts();
    Validate.notEmpty(products, "物料数据不能为空");
    for(ProductPriceVo productPriceVo : products){
      Validate.notBlank(productPriceVo.getMaterialCode(), "产品编码不能为空");
      //todo 验证产品是否有效
      Validate.notNull(productPriceVo.getSuggestedPrice(), "建议销售价不能为空");
      Validate.notNull(productPriceVo.getBasePrice(), "低价不能为空");
    }
  }
}
