package com.biz.crm.cps.business.product.local.controller;

import com.biz.crm.cps.business.common.local.model.Result;
import com.biz.crm.cps.business.product.local.entity.Material;
import com.biz.crm.cps.business.product.local.service.MaterialService;
import com.biz.crm.cps.external.mdm.sdk.dto.MaterialMdmPaginationDto;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import java.util.List;
import javax.annotation.Resource;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

/**
 * 物料信息Controller相关http
 *
 * @author hecheng
 */
@Api(tags = "物料信息")
@Slf4j
@RestController
@RequestMapping("/v1/product/material")
public class MaterialController {

  @Resource
  private MaterialService materialService;

  /**
   * 相关的创建过程，http接口。请注意该创建过程除了可以创建material中的基本信息以外，还可以对material中属于OneToMany关联的明细信息一同进行创建注意：基于（material）模型的创建操作传入的materialJSON对象，其主键信息不能有值，服务端将会自动为其赋予相关值。另外，创建操作成功后，系统将返回该对象的基本信息（不包括任何关联信息）
   */
  @ApiOperation(value = "相关的创建过程，http接口。请注意该创建过程除了可以创建material中的基本信息以外，还可以对material中属于OneToMany关联的明细信息一同进行创建注意：基于（material）模型的创建操作传入的materialJSON对象，其主键信息不能有值，服务端将会自动为其赋予相关值。另外，创建操作成功后，系统将返回该对象的基本信息（不包括任何关联信息）")
  @PostMapping(value = "")
  public Result<Material> create(@RequestBody @ApiParam(name = "material", value = "相关的创建过程，http接口。请注意该创建过程除了可以创建material中的基本信息以外，还可以对material中属于OneToMany关联的明细信息一同进行创建注意：基于（material）模型的创建操作传入的materialJSON对象，其主键信息不能有值，服务端将会自动为其赋予相关值。另外，创建操作成功后，系统将返回该对象的基本信息（不包括任何关联信息）") Material material) {
    try {
      Material current = this.materialService.create(material);
      return Result.ok(current);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 相关的更新过程，http接口。请注意该更新过程只会更新在模型层被标记为了updateable的属性，包括一般属性、ManyToOne和OneToOne性质的关联属性，而ManyToMany、OneToMany的关联属性，虽然也会传入，但需要开发人员自行在Service层完善其更新过程注意：基于模型（material）的修改操作传入的materialJSON对象，其主键信息必须有值，服务端将验证这个主键值是否已经存在。另外，创建操作成功后，系统将返回该对象的基本信息（不包括任何关联信息）
   */
  @ApiOperation(value = "相关的更新过程，http接口。请注意该更新过程只会更新在模型层被标记为了updateable的属性，包括一般属性、ManyToOne和OneToOne性质的关联属性，而ManyToMany、OneToMany的关联属性，虽然也会传入，但需要开发人员自行在Service层完善其更新过程注意：基于模型（material）的修改操作传入的materialJSON对象，其主键信息必须有值，服务端将验证这个主键值是否已经存在。另外，创建操作成功后，系统将返回该对象的基本信息（不包括任何关联信息）")
  @PatchMapping(value = "")
  public Result<Material> update(@RequestBody @ApiParam(name = "material", value = "相关的更新过程，http接口。请注意该更新过程只会更新在模型层被标记为了updateable的属性，包括一般属性、ManyToOne和OneToOne性质的关联属性，而ManyToMany、OneToMany的关联属性，虽然也会传入，但需要开发人员自行在Service层完善其更新过程注意：基于模型（material）的修改操作传入的materialJSON对象，其主键信息必须有值，服务端将验证这个主键值是否已经存在。另外，创建操作成功后，系统将返回该对象的基本信息（不包括任何关联信息）") Material material) {
    try {
      Material current = this.materialService.update(material);
      return Result.ok(current);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 相关的查询过程，http接口。通过主键进行数据的查询
   */
  @ApiOperation(value = "相关的查询过程，http接口。通过主键进行数据的查询")
  @GetMapping(value = "/{id}")
  public Result<Material> findById(@PathVariable("id") @ApiParam(name = "id", value = "主键") String id) {
    try {
      Material current = this.materialService.findById(id);
      return Result.ok(current);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 按照material实体中的（id）主键进行查询明细查询，查询的明细包括当前业务表单所有的关联属性。
   *
   * @param id 主键
   */
  @ApiOperation(value = "按照material实体中的（id）主键进行查询明细查询，查询的明细包括当前业务表单所有的关联属性。")
  @RequestMapping(value = "/findDetailsById", method = {RequestMethod.GET})
  public Result<Material> findDetailsById(@RequestParam("id") @ApiParam("主键") String id) {
    try {
      Material result = this.materialService.findDetailsById(id);
      return Result.ok(result);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 启用
   */
  @ApiOperation(value = "启用")
  @PatchMapping("/enable")
  public Result<?> enable(@RequestBody List<String> ids) {
    try {
      this.materialService.enableBatch(ids);
      return Result.ok("启用成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 禁用
   */
  @ApiOperation(value = "禁用")
  @PatchMapping("/disable")
  public Result<?> disable(@RequestBody List<String> ids) {
    try {
      this.materialService.disableBatch(ids);
      return Result.ok("禁用成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 批量录入物料信息
   */
  @ApiOperation(value = "批量录入物料信息")
  @PostMapping(value = "/createBatch")
  public Result<List<Material>> createBatch(
          @RequestBody @ApiParam(name = "material", value = "物料集合") List<Material> material) {
    try {
      List<Material> current = this.materialService.createBatch(material);
      return Result.ok(current);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 根据条件从mdm侧拉取数据，并存入cps侧系统从指定页 开始 拉去到最后一页
   */
  @ApiOperation(value = "根据条件从mdm侧拉取数据，并存入cps侧系统从指定页 开始 拉去到最后一页")
  @PostMapping(value = "/sync")
  public Result<Boolean> sync(@PageableDefault(50) Pageable pageable,
          @RequestBody @ApiParam(name = "materialMdmPaginationDto", value = "物料查询条件") MaterialMdmPaginationDto materialMdmPaginationDto) {
    try {
      this.materialService.sync(pageable, materialMdmPaginationDto);
      return Result.ok(true);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 标箱转换
   */
  @ApiOperation(value = "标箱转换")
  @GetMapping(value = "/findStandardUnitByBarCodeAndBarCodeType")
  public Result<Material> findStandardUnitByBarCodeAndBarCodeType(@RequestParam("barCode") @ApiParam("barCode") String barCode, @RequestParam("barCodeType") @ApiParam("barCodeType") String barCodeType) {
    try {
      String standardUnit = this.materialService.findStandardUnitByBarCodeAndBarCodeType(barCode, barCodeType);
      Material material = new Material();
      material.setStandardUnit(standardUnit);
      return Result.ok(material);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }
}
