package com.biz.crm.cps.business.product.local.repository;

import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.cps.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.cps.business.product.local.entity.ProductLevel;
import com.biz.crm.cps.business.product.local.mapper.ProductLevelMapper;
import com.biz.crm.cps.business.product.sdk.common.constant.ProductLevelCodeConstant;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

/**
 * 商品层级 的 数据库访问类 {@link ProductLevel}
 *
 * @author hecheng
 */
@Component
public class ProductLevelRepository extends ServiceImpl<ProductLevelMapper, ProductLevel> {

  /**
   * 根据主键查询详细信息（包括关联表）
   *
   * @param id
   * @return
   */
  public ProductLevel findDetailsById(String id) {
    return this.baseMapper.findDetailsById(id);
  }

  /**
   * 根据主键集合，修改 enable_status
   *
   * @param enable
   * @param ids
   */
  public void updateEnableStatusByIdIn(EnableStatusEnum enable, List<String> ids) {
    if (CollectionUtils.isEmpty(ids)) {
      return;
    }
    UpdateWrapper<ProductLevel> updateWrapper = new UpdateWrapper<>();
    updateWrapper.set("enable_status", enable.getCode());
    updateWrapper.in("id", ids);
    this.update(updateWrapper);
  }

  /**
   * 根据productLevelCode层级编码 查询（包括关联表）
   *
   * @param productLevelCode
   * @return
   */
  public ProductLevel findByProductLevelCode(String productLevelCode) {
    return this.baseMapper.findByProductLevelCode(productLevelCode);
  }

  /**
   * 根据parentCode父级编码 查询
   *
   * @param parentCode
   * @return
   */
  public ProductLevel findByParentCode(String parentCode) {
    return this.baseMapper.findByParentCode(parentCode);
  }

  /**
   * 按照ProductLevel的productLevelCode编码，查询所有的父节点信息
   *
   * @param productLevelCode
   * @return
   */
  public List<ProductLevel> findAllParentsByProductLevelCode(String productLevelCode) {
    List<ProductLevel> list = new ArrayList<>();
    ProductLevel one = this.lambdaQuery()
        .eq(StringUtils.isNotEmpty(productLevelCode), ProductLevel::getProductLevelCode,
            productLevelCode)
        .one();
    if (one != null) {
      Set<String> ruleCodes = this.splitParentRuleCodes(one.getRuleCode(),
          ProductLevelCodeConstant.PRODUCT_LEVEL_RULE_CODE_SEPARATOR);
      List<ProductLevel> entityList = this.lambdaQuery()
          .in(ProductLevel::getRuleCode, ruleCodes)
          .list();
      list.addAll(entityList);
    }
    return list;
  }

  /**
   * 按照ProductLevel的productLevelCode编码集合，查询信息
   *
   * @param productLevelCodes
   * @return
   */
  public List<ProductLevel> findByProductLevelCodes(List<String> productLevelCodes) {
    return this.lambdaQuery()
        .in(ProductLevel::getProductLevelCode, productLevelCodes)
        .list();
  }

  /**
   * 根据降维编码分离出上级降维编码集合
   *
   * @param ruleCode
   * @param separator
   * @return
   */
  private Set<String> splitParentRuleCodes(String ruleCode, String separator) {
    Set<String> set = new LinkedHashSet<>();
    if (StringUtils.isNotEmpty(ruleCode)) {
      set.add(ruleCode);
      String[] codes = StringUtils.split(ruleCode, separator);
      for (int i = 1; i < codes.length; i++) {
        String subRuleCode =
            Arrays.stream(codes).limit(i).collect(Collectors.joining(separator)) + separator;
        set.add(subRuleCode);
      }
    }
    return set;
  }


}
