package com.biz.crm.cps.business.product.local.repository;

import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.cps.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.cps.business.product.local.entity.Product;
import com.biz.crm.cps.business.product.local.mapper.ProductMapper;
import com.biz.crm.cps.business.product.sdk.dto.ProductPaginationDto;

import java.util.List;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

/**
 * 商品 的 数据库访问类 {@link Product}
 *
 * @author hecheng
 */
@Component
public class ProductRepository extends ServiceImpl<ProductMapper, Product> {

  /**
   * 根据主键查询详细信息（包括关联表）
   *
   * @param id
   * @return
   */
  public Product findDetailsById(String id) {
    return this.baseMapper.findDetailsById(id);
  }

  /**
   * 根据主键集合，修改 enable_status
   *
   * @param enable
   * @param ids
   */
  public void updateEnableStatusByIdIn(EnableStatusEnum enable, List<String> ids) {
    if (CollectionUtils.isEmpty(ids)) {
      return;
    }
    UpdateWrapper<Product> updateWrapper = new UpdateWrapper<>();
    updateWrapper.set("enable_status", enable.getCode());
    updateWrapper.in("id", ids);
    this.update(updateWrapper);
  }

  /**
   * 根据productCode 产品编码 查询详细信息（包括关联表）
   *
   * @param productCode
   * @return
   */
  public Product findByProductCode(String productCode) {
    return this.baseMapper.findByProductCode(productCode);
  }

  /**
   * 根据productLevelCode 产品层级编码 查询详细信息（包括关联表）
   *
   * @param productLevelCode
   * @return
   */
  public List<Product> findByProductLevelCode(String productLevelCode) {
    return this.baseMapper.findByProductLevelCode(productLevelCode);
  }

  /**
   * 按ids 查询商品信息
   *
   * @param ids
   * @return
   */
  public List<Product> findByIds(List<String> ids) {
    return this.lambdaQuery().in(!CollectionUtils.isEmpty(ids), Product::getId, ids).list();
  }

  /**
   * 按productCodes 查询商品信息
   *
   * @param productCodes
   * @return
   */
  public List<Product> findByProductCodes(List<String> productCodes) {
    return this.lambdaQuery().in(Product::getProductCode, productCodes).list();
  }

  /**
   * @param pageable
   * @param productPaginationDto
   * @return
   */
  public Page<Product> findByConditions(Pageable pageable,
          ProductPaginationDto productPaginationDto) {
    Page<Product> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    return this.baseMapper.findByConditions(page, productPaginationDto);
  }
}
