package com.biz.crm.cps.business.product.local.service;

import com.biz.crm.cps.business.product.local.entity.Material;
import com.biz.crm.cps.external.mdm.sdk.dto.MaterialMdmPaginationDto;
import com.biz.crm.cps.external.mdm.sdk.vo.MaterialMdmVo;
import java.util.List;
import org.springframework.data.domain.Pageable;

/**
 * 物料service
 *
 * @author hecheng
 */
public interface MaterialService {

  /**
   * 创建一个新的Material模型对象（包括了可能的第三方系统调用、复杂逻辑处理等）
   */
  Material create(Material material);

  /**
   * 创建一个新的Material模型对象 该代码由satrun骨架生成，默认不包括任何可能第三方系统调用、任何复杂逻辑处理等，主要应用场景为前端表单数据的暂存功能</br> 该方法与本接口中的updateFrom方法呼应
   */
  Material createForm(Material material);

  /**
   * 更新一个已有的Material模型对象，其主键属性必须有值。 这个方法实际上一共分为三个步骤（默认）：</br> 1、调用updateValidation方法完成表单数据更新前的验证</br> 2、调用updateForm方法完成表单数据的更新</br> 3、完成开发人员自行在本update方法中书写的，进行第三方系统调用（或特殊处理过程）的执行。</br>
   * 这样做的目的，实际上是为了保证updateForm方法中纯粹是处理表单数据的，在数据恢复表单引擎默认调用updateForm方法时，不会影响任何第三方业务数据 （当然，如果系统有特别要求，可由开发人员自行完成代码逻辑调整）
   */
  Material update(Material material);

  /**
   * 该方法只用于处理业务表单信息，包括了主业务模型、其下的关联模型、分组信息和明细细信息等 该方法非常重要，因为如果进行静态表单的数据恢复，那么表单引擎将默认调用主业务模型（服务层）的这个方法。</br> 这样一来保证了数据恢复时，不会涉及任何第三方系统的调用（当然，如果开发人员需要涉及的，可以自行进行修改）
   */
  Material updateForm(Material material);

  /**
   * 按照Material的主键编号，查询指定的数据信息（不包括任何关联信息）
   *
   * @param id 主键
   */
  Material findById(String id);

  /**
   * 按照主键进行详情查询（包括关联信息）
   *
   * @param id 主键
   */
  Material findDetailsById(String id);

  /**
   * 批量启用
   *
   * @param ids
   */
  void enableBatch(List<String> ids);

  /**
   * 批量禁用
   *
   * @param ids
   */
  void disableBatch(List<String> ids);

  /**
   * 批量录入信息
   *
   * @param materials
   * @return
   */
  List<Material> createBatch(List<Material> materials);

  /**
   * 创建或者更新
   *
   * @param target
   */
  void saveBatch(List<Material> target);

  /**
   * 按照Material的materialCode物料编码，查询指定的数据信息（不包括任何关联信息）
   *
   * @param materialCode 物料编码
   */
  Material findByMaterialCode(String materialCode);

  /**
   * 按照Material的productLevelCode层级编码，查询指定的数据信息（不包括任何关联信息）
   *
   * @param productLevelCode 层级编码
   */
  List<Material> findByProductLevelCode(String productLevelCode);

  /**
   * 根据条件从mdm侧拉取数据，并存入cps侧系统 从指定页 开始 拉去到最后一页
   *
   * @param materialMdmPaginationDto
   */
  void sync(Pageable pageable, MaterialMdmPaginationDto materialMdmPaginationDto);

  /**
   * 处理同步数据 物料、物料图片、物料层级
   *
   * @param materialMdmVos
   */
  void syncDataHandle(List<MaterialMdmVo> materialMdmVos);

  /**
   * 按物料编码集合查询物料
   *
   * @param materialCodes
   * @return
   */
  List<Material> findByMaterialCodes(List<String> materialCodes);

  /**
   * 按物料层级集合查询物料
   *
   * @param productLevelCodes
   * @return
   */
  List<Material> findByProductLevelCodes(List<String> productLevelCodes);
  /**
   * 按物料层级集合查询物料 包含孩子
   *
   * @param productLevelCodes
   * @return
   */
  List<Material> findAllChildrenByProductLevelCodes(List<String> productLevelCodes);
  /**
   * 通过 码 和码类型 查询物料标准单位
   *
   * @param barCode
   * @param barCodeType
   * @return
   */
  String findStandardUnitByBarCodeAndBarCodeType(String barCode, String barCodeType);

  /**
   * 查询所有物料信息
   *
   * @return
   */
  List<Material> findAll();
}
