package com.biz.crm.cps.business.product.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.cps.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.cps.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.cps.business.product.local.entity.MaterialGroup;
import com.biz.crm.cps.business.product.local.entity.MaterialGroupRel;
import com.biz.crm.cps.business.product.local.repository.MaterialGroupRepository;
import com.biz.crm.cps.business.product.local.service.MaterialGroupRelService;
import com.biz.crm.cps.business.product.local.service.MaterialGroupService;
import com.biz.crm.cps.business.product.local.service.MaterialService;
import com.biz.crm.cps.external.mdm.sdk.dto.MaterialGroupMdmPaginationDto;
import com.biz.crm.cps.external.mdm.sdk.service.MaterialMdmService;
import com.biz.crm.cps.external.mdm.sdk.vo.MaterialGroupMdmVo;
import com.biz.crm.cps.external.mdm.sdk.vo.MaterialMdmVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.security.core.context.SecurityContext;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.TreeSet;
import java.util.stream.Collectors;

/**
 * @author hecheng
 * @description: 物料组service实现
 * @date 2021/8/23 下午2:43
 */
@Slf4j
@Service
public class MaterialGroupServiceImpl implements MaterialGroupService {

  @Autowired
  private MaterialGroupRepository materialGroupRepository;
  @Autowired(required = false)
  private MaterialMdmService materialMdmService;
  @Autowired
  private MaterialService materialService;
  @Autowired
  private MaterialGroupRelService materialGroupRelService;
  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Transactional
  @Override
  public MaterialGroup create(MaterialGroup materialGroup) {
    MaterialGroup current = this.createForm(materialGroup);
    this.materialGroupRepository.save(current);
    return current;
  }

  /**
   * 构造对象
   *
   * @param materialGroup
   * @return
   */
  private MaterialGroup createForm(MaterialGroup materialGroup) {
    Date now = new Date();
    String account = this.getLoginAccountName();
    materialGroup.setTenantCode(TenantUtils.getTenantCode());
    materialGroup.setCreateAccount(account);
    materialGroup.setCreateTime(now);
    materialGroup.setModifyAccount(account);
    materialGroup.setModifyTime(now);
    this.createValidation(materialGroup);
    if (StringUtils.isBlank(materialGroup.getDelFlag())) {
      materialGroup.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    }
    if (StringUtils.isBlank(materialGroup.getEnableStatus())) {
      materialGroup.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    }
    return materialGroup;
  }

  @Transactional
  @Override
  public MaterialGroup update(MaterialGroup materialGroup) {
    MaterialGroup current = this.updateForm(materialGroup);
    return current;
  }

  /**
   * 构造数据
   *
   * @param materialGroup
   * @return
   */
  private MaterialGroup updateForm(MaterialGroup materialGroup) {
    /*
     * 对静态模型的修改操作的过程为：
     * 1、如果当前模型对象不是主模型
     *  1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     *  1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     *  1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理
     *
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     *  2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *    2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *    2.3.2、以及验证每个分组的OneToMany明细信息
     */
    this.updateValidation(materialGroup);
    //这里可根据id或者code更新
    String currentId = materialGroup.getId();
    String materialGroupCode = materialGroup.getMaterialGroupCode();
    MaterialGroup current = null;
    if (StringUtils.isNotBlank(currentId)) {
      current = this.materialGroupRepository.getById(currentId);
    } else if (StringUtils.isNotBlank(materialGroupCode)) {
      current = this.findByMaterialGroupCode(materialGroupCode);
    }
    current = Validate.notNull(current, "未发现指定的原始模型对象信");
    BeanUtils.copyProperties(materialGroup, current, "id", "modifyTime", "createAccount", "createTime", "tenantCode");
    // 开始赋值——更新时间与更新人
    Date now = new Date();
    String account = this.getLoginAccountName();
    current.setModifyAccount(account);
    current.setModifyTime(now);
    this.materialGroupRepository.saveOrUpdate(current);
    return current;
  }

  @Override
  public MaterialGroup findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.materialGroupRepository.getById(id);
  }

  @Override
  public MaterialGroup findDetailsById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.materialGroupRepository.findDetailsById(id);
  }

  @Override
  public MaterialGroup findByMaterialGroupCode(String materialGroupCode) {
    if (StringUtils.isBlank(materialGroupCode)) {
      return null;
    }
    return this.materialGroupRepository.findByMaterialGroupCode(materialGroupCode);
  }

  @Transactional
  @Override
  public void sync(Pageable pageable, MaterialGroupMdmPaginationDto materialGroupMdmPaginationDto) {
    boolean isQuery = true;
    int pageNumber = pageable.getPageNumber() - 1;
    while (isQuery) {
      int pageSize = pageable.getPageSize();
      pageNumber = pageNumber + 1;
      PageRequest pageRequest = PageRequest.of(pageNumber, pageSize);
      Page<MaterialGroupMdmVo> page = this.materialMdmService.findByConditions(pageRequest, materialGroupMdmPaginationDto);
      //处理同步数据
      this.syncDataHandle(page.getRecords());
      if (page.getRecords().size() < pageSize) {
        isQuery = false;
      }
    }
  }

  /**
   * 处理同步
   *
   * @param materialGroupMdmVos
   */
  private void syncDataHandle(List<MaterialGroupMdmVo> materialGroupMdmVos) {
    if (CollectionUtils.isEmpty(materialGroupMdmVos)) {
      log.info("同步物料组时：未查询到数据！");
      return;
    }

    //物料
    List<MaterialMdmVo> materialMdmVos = materialGroupMdmVos.stream().flatMap(item -> item.getMaterialMdmVos().stream())
            .collect(Collectors.collectingAndThen(Collectors.toCollection(() -> new TreeSet<>(Comparator.comparing(MaterialMdmVo::getId))), ArrayList::new));
    //同步 物料、物料图片、物料层级
    this.materialService.syncDataHandle(materialMdmVos);
    for (MaterialGroupMdmVo materialGroupMdmVo : materialGroupMdmVos) {
      List<MaterialMdmVo> materialMdmVoList = materialGroupMdmVo.getMaterialMdmVos();
      MaterialGroup materialGroup = this.nebulaToolkitService.copyObjectByWhiteList(materialGroupMdmVo, MaterialGroup.class, HashSet.class, ArrayList.class);
      materialGroup.setExternalIdentifier(materialGroupMdmVo.getId());
      materialGroup.setId(null);
      MaterialGroup old = this.findByMaterialGroupCode(materialGroupMdmVo.getMaterialGroupCode());
      MaterialGroup materialGroupResult = null;
      if (old == null) {
        materialGroupResult = this.create(materialGroup);
      } else {
        BeanUtils.copyProperties(materialGroup, old, "id", "modifyTime", "createAccount", "createTime", "tenantCode");
        materialGroupResult = this.update(old);
      }
      if (materialGroupResult != null) {
        MaterialGroup finalMaterialGroupResult = materialGroupResult;
        List<MaterialGroupRel> materialGroupRels = materialMdmVoList.stream().map(item -> {
          MaterialGroupRel rel = new MaterialGroupRel();
          rel.setMaterialCode(item.getMaterialCode());
          rel.setMaterialGroupCode(finalMaterialGroupResult.getMaterialGroupCode());
          return rel;
        }).collect(Collectors.toList());
        if(!CollectionUtils.isEmpty(materialGroupRels)){
          this.materialGroupRelService.createBatch(materialGroupRels);
        }
      }
    }

  }

  /**
   * 在更新一个的materialGroup模型对象之前，检查对象各属性的正确性
   *
   * @param materialGroup
   */
  private void updateValidation(MaterialGroup materialGroup) {
    Validate.isTrue(!(StringUtils.isBlank(materialGroup.getId()) && StringUtils.isBlank(materialGroup.getMaterialGroupCode())),
            "修改信息时，当期信息的数据编号（主键/编码）必须有值！");
    Validate.notNull(materialGroup.getMaterialGroupName(), "修改信息时，物料组名称不能为空！");
    Validate.notNull(materialGroup.getMaterialNum(), "修改信息时，物料数量不能为空！");
    Validate.isTrue(materialGroup.getMaterialNum().compareTo(BigDecimal.ZERO)>=0 , "修改信息时，物料数量必须大于等于0！");
    Validate.isTrue(materialGroup.getDescription() == null || materialGroup.getDescription().length() < 255, "物料组描述，在进行添加时填入值超过了限定长度(255)，请检查!");
  }

  /**
   * 数据校验
   *
   * @param materialGroup
   */
  private void createValidation(MaterialGroup materialGroup) {
    Validate.notNull(materialGroup, "进行当前操作时，信息对象必须传入!!");
    materialGroup.setId(null);
    Validate.notNull(materialGroup.getMaterialGroupName(), "添加信息时，物料组名称不能为空！");
    Validate.notNull(materialGroup.getMaterialNum(), "添加信息时，物料数量不能为空！");
    Validate.isTrue(materialGroup.getMaterialNum().compareTo(BigDecimal.ZERO)>=0 , "添加信息时，物料数量必须大于等于0！");
    Validate.isTrue(materialGroup.getDescription() == null || materialGroup.getDescription().length() < 255, "物料组描述，在进行添加时填入值超过了限定长度(255)，请检查!");

  }

  /**
   * 获取当前登录人名称
   *
   * @return
   */
  private String getLoginAccountName() {
    SecurityContext context = SecurityContextHolder.getContext();
    String account = "admin";
    if (context != null && context.getAuthentication() != null) {
      account = context.getAuthentication().getName();
    }
    return account;
  }
}
