package com.biz.crm.cps.business.product.local.service.internal;

import com.biz.crm.cps.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.cps.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.cps.business.product.local.entity.MaterialMedia;
import com.biz.crm.cps.business.product.local.repository.MaterialMediaRepository;
import com.biz.crm.cps.business.product.local.service.MaterialMediaService;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.context.SecurityContext;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

/**
 * @author hecheng
 * @description: 物料图片service实现
 * @date 2021/8/23 上午11:16
 */
@Service
public class MaterialMediaServiceImpl implements MaterialMediaService {

  @Autowired
  private MaterialMediaRepository materialMediaRepository;

  @Transactional
  @Override
  public void createBatch(List<MaterialMedia> materialMedias) {
    Validate.notEmpty(materialMedias, "物料图片关系不能为空");
    Map<String, List<MaterialMedia>> materialMediaMap = materialMedias.stream().collect(Collectors.groupingBy(MaterialMedia::getMaterialCode));
    for (Map.Entry<String, List<MaterialMedia>> entry : materialMediaMap.entrySet()) {
      String materialCode = entry.getKey();
      List<MaterialMedia> tempMaterialMedias = entry.getValue();
      List<MaterialMedia> old = this.materialMediaRepository.findByMaterialCode(materialCode);
      if (!CollectionUtils.isEmpty(old)) {
        this.materialMediaRepository.deleteByMaterialCode(materialCode);
      }
      for (MaterialMedia materialMedia : tempMaterialMedias) {
        this.createForm(materialMedia);
      }
      this.materialMediaRepository.saveBatch(tempMaterialMedias);
    }
  }

  /**
   * 组装数据
   *
   * @param materialMedia
   * @return
   */
  private MaterialMedia createForm(MaterialMedia materialMedia) {
    Date now = new Date();
    String account = this.getLoginAccountName();
    materialMedia.setCreateAccount(account);
    materialMedia.setCreateTime(now);
    materialMedia.setModifyAccount(account);
    materialMedia.setModifyTime(now);
    this.createValidation(materialMedia);
    if (StringUtils.isBlank(materialMedia.getDelFlag())) {
      materialMedia.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    }
    if (StringUtils.isBlank(materialMedia.getEnableStatus())) {
      materialMedia.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    }
    return materialMedia;
  }


  @Override
  public MaterialMedia findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.materialMediaRepository.getById(id);
  }

  @Override
  public MaterialMedia findDetailsById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.materialMediaRepository.findDetailsById(id);
  }

  @Override
  public List<MaterialMedia> findByMaterialCode(String materialCode) {
    if (StringUtils.isBlank(materialCode)) {
      return null;
    }
    return this.materialMediaRepository.findByMaterialCode(materialCode);
  }

  /**
   * 数据校验
   *
   * @param materialMedia
   */
  private void createValidation(MaterialMedia materialMedia) {
    Validate.notNull(materialMedia, "进行当前操作时，信息对象必须传入!!");
    materialMedia.setId(null);
    Validate.notBlank(materialMedia.getMaterialCode(), "添加信息时，物料编码不能为空！");
    Validate.notBlank(materialMedia.getMaterialId(), "添加信息时，物料id不能为空！");
    Validate.isTrue(materialMedia.getUrlAddress() == null || materialMedia.getUrlAddress().length() < 255, "图片地址，在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(materialMedia.getFileName() == null ||materialMedia.getFileName().length() < 128, "文件名称，在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(materialMedia.getOriginalFileName() == null || materialMedia.getOriginalFileName().length() < 128, "源文件名称，在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(materialMedia.getRelativeLocal() == null || materialMedia.getRelativeLocal().length() < 255, "地址，在进行添加时填入值超过了限定长度(255)，请检查!");

  }

  /**
   * 获取当前登录人名称
   *
   * @return
   */
  private String getLoginAccountName() {
    SecurityContext context = SecurityContextHolder.getContext();
    String account = "admin";
    if (context != null && context.getAuthentication() != null) {
      account = context.getAuthentication().getName();
    }
    return account;
  }
}
