package com.biz.crm.cps.business.product.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.cps.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.cps.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.cps.business.product.local.entity.Material;
import com.biz.crm.cps.business.product.local.entity.MaterialMedia;
import com.biz.crm.cps.business.product.local.entity.ProductLevel;
import com.biz.crm.cps.business.product.local.repository.MaterialRepository;
import com.biz.crm.cps.business.product.local.service.MaterialMediaService;
import com.biz.crm.cps.business.product.local.service.MaterialService;
import com.biz.crm.cps.business.product.local.service.ProductLevelService;
import com.biz.crm.cps.business.product.sdk.common.constant.MaterialCodeConstant;
import com.biz.crm.cps.business.product.sdk.common.enums.MaterialStandardUnitTypeEnum;
import com.biz.crm.cps.business.product.sdk.event.MaterialEventListener;
import com.biz.crm.cps.business.product.sdk.vo.MaterialVo;
import com.biz.crm.cps.external.barcode.sdk.common.enums.BarCodeTypeEnum;
import com.biz.crm.cps.external.barcode.sdk.service.BarCodeVoService;
import com.biz.crm.cps.external.barcode.sdk.vo.BarCodeVo;
import com.biz.crm.cps.external.mdm.sdk.dto.MaterialMdmPaginationDto;
import com.biz.crm.cps.external.mdm.sdk.service.MaterialMdmService;
import com.biz.crm.cps.external.mdm.sdk.vo.MaterialMdmVo;
import com.biz.crm.cps.external.mdm.sdk.vo.MaterialMediaMdmVo;
import com.biz.crm.cps.external.mdm.sdk.vo.ProductLevelMdmVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.security.core.context.SecurityContext;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.TreeSet;
import java.util.stream.Collectors;

/**
 * @author hecheng
 * @description: 物料service实现
 * @date 2021/8/9 下午2:37
 */
@Slf4j
@Service
public class MaterialServiceImpl implements MaterialService {

  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;
  @Autowired(required = false)
  private GenerateCodeService generateCodeService;
  @Autowired
  private MaterialRepository materialRepository;
  @Autowired
  private MaterialMediaService materialMediaService;
  @Autowired
  private ProductLevelService productLevelService;
  @Autowired(required = false)
  private List<MaterialEventListener> materialEventListenerList;
  @Autowired(required = false)
  private MaterialMdmService materialMdmService;
  @Autowired(required = false)
  private BarCodeVoService barCodeVoService;

  @Transactional
  @Override
  public Material create(Material material) {
    Material current = this.createForm(material);
    this.materialRepository.save(current);
    return current;
  }

  @Override
  public Material createForm(Material material) {
    /*
     * 对静态模型的保存操作过程为：
     * 1、如果当前模型对象不是主模型
     *  1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     *  1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     *  1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     * 2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *   2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *   2.3.2、以及验证每个分组的OneToMany明细信息
     * */
    Date now = new Date();
    String account = this.getLoginAccountName();
    material.setTenantCode(TenantUtils.getTenantCode());
    material.setCreateAccount(account);
    material.setCreateTime(now);
    material.setModifyAccount(account);
    material.setModifyTime(now);
    this.createValidation(material);
    if (StringUtils.isBlank(material.getDelFlag())) {
      material.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    }
    if (StringUtils.isBlank(material.getEnableStatus())) {
      material.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    }
    String materialCode = material.getMaterialCode();
    if (StringUtils.isBlank(materialCode)) {
      materialCode = this.generateCodeService.generateCode(MaterialCodeConstant.MATERIAL_CODE, 1).get(0);
      material.setMaterialCode(materialCode);
    }
    return material;
  }

  @Override
  public Material update(Material material) {
    Material current = this.updateForm(material);

    return current;
  }

  @Transactional
  @Override
  public Material updateForm(Material material) {
    /*
     * 对静态模型的修改操作的过程为：
     * 1、如果当前模型对象不是主模型
     *  1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     *  1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     *  1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理
     *
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     *  2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *    2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *    2.3.2、以及验证每个分组的OneToMany明细信息
     */
    this.updateValidation(material);
    //这里可根据id或者code更新
    String currentId = material.getId();
    String materialCode = material.getMaterialCode();
    Material current = null;
    if (StringUtils.isNotBlank(currentId)) {
      current = this.materialRepository.getById(currentId);
    } else if (StringUtils.isNotBlank(materialCode)) {
      current = this.findByMaterialCode(materialCode);
    }
    current = Validate.notNull(current, "未发现指定的原始模型对象信");
    // 复制旧对象
    MaterialVo oldVo = null;
    if (!CollectionUtils.isEmpty(materialEventListenerList)) {
      oldVo = this.nebulaToolkitService.copyObjectByWhiteList(current, MaterialVo.class, HashSet.class, ArrayList.class);
    }
    BeanUtils.copyProperties(material, current, "id", "modifyTime", "createAccount", "createTime",
            "tenantCode");

    // 开始赋值——更新时间与更新人
    Date now = new Date();
    String account = this.getLoginAccountName();
    current.setModifyAccount(account);
    current.setModifyTime(now);
    this.materialRepository.saveOrUpdate(current);
    // 发送修改通知
    if (!CollectionUtils.isEmpty(materialEventListenerList)) {
      MaterialVo newVo = this.nebulaToolkitService.copyObjectByWhiteList(current, MaterialVo.class, HashSet.class, ArrayList.class);
      for (MaterialEventListener event : materialEventListenerList) {
        event.onChange(oldVo, newVo);
      }
    }
    return current;
  }

  @Override
  public Material findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return materialRepository.getById(id);
  }

  @Override
  public Material findDetailsById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return materialRepository.findDetailsById(id);
  }

  @Transactional
  @Override
  public void enableBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "请选中要操作的数据");
    this.materialRepository.updateEnableStatusByIdIn(EnableStatusEnum.ENABLE, ids);

  }

  @Transactional
  @Override
  public void disableBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "请传入要操作的数据");
    this.materialRepository.updateEnableStatusByIdIn(EnableStatusEnum.DISABLE, ids);
    if (!CollectionUtils.isEmpty(materialEventListenerList)) {
      List<Material> terminals = this.materialRepository.findByIds(ids);
      List<MaterialVo> voList = (List<MaterialVo>) this.nebulaToolkitService.copyCollectionByWhiteList(terminals, Material.class, MaterialVo.class, HashSet.class, ArrayList.class);
      for (MaterialEventListener event : materialEventListenerList) {
        event.onDisable(voList);
      }
    }
  }

  @Transactional
  @Override
  public List<Material> createBatch(List<Material> materials) {
    if (!CollectionUtils.isEmpty(materials)) {
      for (Material material : materials) {
        this.createForm(material);
      }
      this.materialRepository.saveBatch(materials);
      return materials;
    }
    return Lists.newArrayList();
  }

  /**
   * 批量更新
   *
   * @param materials
   * @return
   */
  private List<Material> updateBatch(List<Material> materials) {
    if (!CollectionUtils.isEmpty(materials)) {
      for (Material material : materials) {
        this.updateForm(material);
      }
      return materials;
    }
    return Lists.newArrayList();
  }

  @Transactional
  @Override
  public void saveBatch(List<Material> target) {
    //1  先区分出 新增的 和更新的
    if (!CollectionUtils.isEmpty(target)) {
      List<String> materialCodes = target.stream().map(Material::getMaterialCode)
              .distinct().collect(Collectors.toList());
      List<Material> oldList = Lists.newArrayList();
      if (!CollectionUtils.isEmpty(materialCodes)) {
        oldList = this.materialRepository.findByMaterialCodes(materialCodes);
      }
      List<Material> addList = Lists.newArrayList();
      List<Material> needUpdateList = Lists.newArrayList();
      List<Material> delList = Lists.newArrayList();
      this.nebulaToolkitService.collectionDiscrepancy(target, oldList, Material::getMaterialCode, delList, needUpdateList, addList);
      this.createBatch(addList);
      if (!CollectionUtils.isEmpty(needUpdateList)) {
        List<String> needUpdateCodeList = needUpdateList.stream().map(Material::getMaterialCode).collect(Collectors.toList());
        List<Material> updateList = target.stream().filter(item -> needUpdateCodeList.contains(item.getMaterialCode())).collect(Collectors.toList());
        this.updateBatch(updateList);
      }
    }
  }

  @Override
  public Material findByMaterialCode(String materialCode) {
    if (StringUtils.isBlank(materialCode)) {
      return null;
    }
    return this.materialRepository.findByMaterialCode(materialCode);
  }

  @Override
  public List<Material> findByProductLevelCode(String productLevelCode) {
    if (StringUtils.isBlank(productLevelCode)) {
      return null;
    }
    return this.materialRepository.findByProductLevelCode(productLevelCode);
  }

  @Override
  @Transactional
  public void sync(Pageable pageable, MaterialMdmPaginationDto materialMdmPaginationDto) {
    boolean isQuery = true;
    int pageNumber = pageable.getPageNumber() - 1;
    while (isQuery) {
      int pageSize = pageable.getPageSize();
      pageNumber = pageNumber + 1;
      PageRequest pageRequest = PageRequest.of(pageNumber, pageSize);
      Page<MaterialMdmVo> page = this.materialMdmService.findByConditions(pageRequest, materialMdmPaginationDto);
      //处理同步数据
      this.syncDataHandle(page.getRecords());
      if (page.getRecords().size() < pageSize) {
        isQuery = false;
      }
    }
  }

  /**
   * 处理同步
   *
   * @param materialMdmVos
   */
  @Override
  public void syncDataHandle(List<MaterialMdmVo> materialMdmVos) {
    if (CollectionUtils.isEmpty(materialMdmVos)) {
      log.info("同步物料时：未查询到数据！");
      return;
    }
    //1物料层级
    List<ProductLevelMdmVo> productLevelMdmVos = materialMdmVos.stream().flatMap(item -> item.getProductLevels().stream())
            .collect(Collectors.collectingAndThen(Collectors.toCollection(() -> new TreeSet<>(Comparator.comparing(ProductLevelMdmVo::getId))), ArrayList::new));
    //2物料图片
    List<MaterialMediaMdmVo> mediaMdmVos = materialMdmVos.stream().flatMap(item -> item.getMaterialMedias().stream())
            .collect(Collectors.collectingAndThen(Collectors.toCollection(() -> new TreeSet<>(Comparator.comparing(MaterialMediaMdmVo::getId))), ArrayList::new));

    this.bulidProductLevel(productLevelMdmVos);

    this.bulidMaterial(materialMdmVos);

    this.bulidMaterialMedia(mediaMdmVos);
  }

  @Override
  public List<Material> findByMaterialCodes(List<String> materialCodes) {
    if (CollectionUtils.isEmpty(materialCodes)) {
      return null;
    }
    return this.materialRepository.findByMaterialCodes(materialCodes);
  }

  @Override
  public List<Material> findByProductLevelCodes(List<String> productLevelCodes) {
    if (CollectionUtils.isEmpty(productLevelCodes)) {
      return null;
    }
    return this.materialRepository.findByProductLevelCodes(productLevelCodes);
  }

  @Override
  public List<Material> findAllChildrenByProductLevelCodes(List<String> productLevelCodes) {
    if (CollectionUtils.isEmpty(productLevelCodes)) {
      return null;
    }
    List<ProductLevel> levels = this.productLevelService.findByProductLevelCodes(productLevelCodes);
    if (CollectionUtils.isEmpty(levels)) {
      return null;
    }
    List<String> ruleCodes = levels.stream().map(ProductLevel::getRuleCode).distinct().collect(Collectors.toList());
    return this.materialRepository.findAllChildrenByRuleCodes(ruleCodes);
  }

  @Override
  public String findStandardUnitByBarCodeAndBarCodeType(String barCode, String barCodeType) {
    if (StringUtils.isBlank(barCode) || StringUtils.isBlank(barCodeType)) {
      return null;
    }
    BarCodeVo barCodeVo = this.barCodeVoService.findByBarCode(barCode);
    if (Objects.isNull(barCodeVo)) {
      return null;
    }
    Material material = this.materialRepository.findByMaterialCode(barCodeVo.getProductCode());
    if (Objects.isNull(material)) {
      return null;
    }
    String unit = "~";
    MaterialStandardUnitTypeEnum standardUnitEnum = MaterialStandardUnitTypeEnum.getByKey(material.getStandardUnit());
    String standardUnit = Objects.isNull(standardUnitEnum) ? "" : standardUnitEnum.getValue();
    BigDecimal boxUnitConversion = material.getBoxUnitConversion();
    BigDecimal caseUnitConversion = material.getCaseUnitConversion();
    if (Objects.isNull(boxUnitConversion) || Objects.isNull(caseUnitConversion)) {
      return unit;
    }
    if (Objects.equals(barCodeType, BarCodeTypeEnum.BOTTLE.getFlag())) {
      unit = boxUnitConversion.toString() + standardUnit;
    } else if (Objects.equals(barCodeType, BarCodeTypeEnum.BOX.getFlag())) {
      unit = caseUnitConversion.toString() + standardUnit;
    }
    return unit;
  }

  @Override
  public List<Material> findAll() {
    return this.materialRepository.list();
  }

  /**
   * 根据商品层级类型 筛选符合条件的层级Vo构建商品层级
   *
   * @param productLevelMdmVos
   * @return
   */
  private void bulidProductLevel(List<ProductLevelMdmVo> productLevelMdmVos) {
    if (!CollectionUtils.isEmpty(productLevelMdmVos)) {
      List<ProductLevel> target = productLevelMdmVos.stream().map(item -> {
        ProductLevel productLevel = this.nebulaToolkitService.copyObjectByWhiteList(item, ProductLevel.class, HashSet.class, ArrayList.class);
        productLevel.setExternalIdentifier(item.getId());
        productLevel.setId(null);
        return productLevel;
      }).collect(Collectors.toList());
      this.productLevelService.saveBatch(target);
    }
  }

  /**
   * 根据物料vo 构建物料
   *
   * @param materialMdmVos
   * @return
   */
  private void bulidMaterial(List<MaterialMdmVo> materialMdmVos) {
    if (!CollectionUtils.isEmpty(materialMdmVos)) {
      List<Material> target = materialMdmVos.stream().map(item -> {
        Material material = new Material();
        BeanUtils.copyProperties(item, material, "costPrice");
        if (StringUtils.isNotBlank(item.getCostPrice())) {
          material.setCostPrice(new BigDecimal(item.getCostPrice()));
        }
        material.setExternalIdentifier(item.getId());
        material.setId(null);
        return material;
      }).collect(Collectors.toList());
      this.saveBatch(target);
    }
  }

  /**
   * 根据物料vo 构建物料图片
   *
   * @param mediaMdmVos
   * @return
   */
  private void bulidMaterialMedia(List<MaterialMediaMdmVo> mediaMdmVos) {
    if (!CollectionUtils.isEmpty(mediaMdmVos)) {
      List<MaterialMedia> target = mediaMdmVos.stream().map(item -> {
        MaterialMedia media = new MaterialMedia();
        BeanUtils.copyProperties(item, media);
        media.setExternalIdentifier(item.getId());
        media.setId(null);
        return media;
      }).collect(Collectors.toList());
      if (!CollectionUtils.isEmpty(target)) {
        this.materialMediaService.createBatch(target);
      }
    }
  }

  /**
   * 在创建一个新的material模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(Material material) {
    Validate.notNull(material, "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    material.setId(null);
    Validate.notBlank(material.getTenantCode(), "添加信息时，租户编号不能为空！");
    Validate.notBlank(material.getProductLevelCode(), "添加信息时，商品层级编码不能为空！");
    Validate.notBlank(material.getMaterialName(), "添加信息时，产品名称不能为空！");
//    Validate.notNull(ProductLevelTypeEnum.getByCode(productLevel.getProductLevelType()),
//        "添加信息时，商品层级类型不合法！");
    Validate.isTrue(material.getMaterialName().length() < 128, "产品名称，在进行添加时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(material.getAiCode() == null || material.getAiCode().length() < 128, "ai编码，在进行添加时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(material.getBarCode() == null || material.getBarCode().length() < 128, "条形码，在进行添加时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(material.getBaseUnit() == null || material.getBaseUnit().length() < 64, "基本单位，在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(material.getSaleCompany() == null || material.getSaleCompany().length() < 64, "销售公司，在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(material.getSaleUnit() == null || material.getSaleUnit().length() < 64, "销售单位，在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(material.getSpecification() == null || material.getSpecification().length() < 64, "规格，在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(material.getUnitConversion() == null || material.getUnitConversion().length() < 64, "单位换算系数，在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(material.getGrossWeight() == null || material.getGrossWeight().length() < 64, "毛重，在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(material.getNetWeight() == null || material.getNetWeight().length() < 64, "净重，在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(material.getCapacity() == null || material.getCapacity().length() < 64, "容量，在进行添加时填入值超过了限定长度(64)，请检查!");
    if (StringUtils.isNotBlank(material.getProductLevelCode())) {
      ProductLevel productLevel = this.productLevelService.findByProductLevelCode(material.getProductLevelCode());
      Validate.notNull(productLevel, "商品层级编码[" + material.getProductLevelCode() + "]不存在");
    }
    if (StringUtils.isNotBlank(material.getMaterialCode())) {
      List<Material> list = this.materialRepository.lambdaQuery()
              .eq(Material::getMaterialCode, material.getMaterialCode())
              .select(Material::getId)
              .list();
      Validate.isTrue(CollectionUtils.isEmpty(list), "编码[" + material.getMaterialCode() + "]已存在");
    }
  }

  /**
   * 在更新一个的material模型对象之前，检查对象各属性的正确性
   *
   * @param material
   */
  private void updateValidation(Material material) {
    Validate.isTrue(!(StringUtils.isBlank(material.getId()) && StringUtils.isBlank(material.getMaterialCode())), "修改信息时，当期信息的数据编号（主键/编码）必须有值！");
    Validate.isTrue(material.getMaterialName().length() < 128, "产品名称，在进行修改时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(material.getAiCode() == null || material.getAiCode().length() < 64, "ai编码，在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(material.getBarCode() == null || material.getBarCode().length() < 64, "条形码，在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(material.getBaseUnit() == null || material.getBaseUnit().length() < 64, "基本单位，在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(material.getSaleCompany() == null || material.getSaleCompany().length() < 64, "销售公司，在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(material.getSaleUnit() == null || material.getSaleUnit().length() < 64, "销售单位，在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(material.getSpecification() == null || material.getSpecification().length() < 64, "规格，在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(material.getUnitConversion() == null || material.getUnitConversion().length() < 64, "单位换算系数，在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(material.getGrossWeight() == null || material.getGrossWeight().length() < 64, "毛重，在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(material.getNetWeight() == null || material.getNetWeight().length() < 64, "净重，在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(material.getCapacity() == null || material.getCapacity().length() < 64, "容量，在进行修改时填入值超过了限定长度(64)，请检查!");
    if (StringUtils.isNotBlank(material.getProductLevelCode())) {
      ProductLevel productLevel = this.productLevelService.findByProductLevelCode(material.getProductLevelCode());
      Validate.notNull(productLevel, "商品层级编码[" + material.getProductLevelCode() + "]不存在");
    }
  }

  /**
   * 获取当前登录人名称
   *
   * @return
   */
  private String getLoginAccountName() {
    SecurityContext context = SecurityContextHolder.getContext();
    String account = "admin";
    if (context != null && context.getAuthentication() != null) {
      account = context.getAuthentication().getName();
    }
    return account;
  }
}
