package com.biz.crm.cps.business.product.local.service.internal;

import com.biz.crm.business.common.local.config.DefaultLoginAuthenticationConfig;
import com.biz.crm.cps.business.product.local.entity.Material;
import com.biz.crm.cps.business.product.local.entity.MaterialGroupRel;
import com.biz.crm.cps.business.product.local.entity.ProductLevel;
import com.biz.crm.cps.business.product.local.entity.ProductMaterial;
import com.biz.crm.cps.business.product.local.service.MaterialGroupRelService;
import com.biz.crm.cps.business.product.local.service.MaterialGroupService;
import com.biz.crm.cps.business.product.local.service.MaterialService;
import com.biz.crm.cps.business.product.local.service.ProductLevelService;
import com.biz.crm.cps.business.product.local.service.ProductMaterialService;
import com.biz.crm.cps.business.product.sdk.common.constant.MaterialDimensionConstant;
import com.biz.crm.cps.business.product.sdk.common.enums.MaterialDimensionEnum;
import com.biz.crm.cps.business.product.sdk.dto.MaterialDimensionDto;
import com.biz.crm.cps.business.product.sdk.dto.MaterialGroupRelDto;
import com.biz.crm.cps.business.product.sdk.dto.ProductLevelDto;
import com.biz.crm.cps.business.product.sdk.service.MaterialVoService;
import com.biz.crm.cps.business.product.sdk.vo.MaterialDimensionVo;
import com.biz.crm.cps.business.product.sdk.vo.MaterialGroupRelVo;
import com.biz.crm.cps.business.product.sdk.vo.MaterialVo;
import com.biz.crm.cps.business.product.sdk.vo.ProductLevelVo;
import com.biz.crm.cps.business.product.sdk.vo.ProductMaterialVo;
import com.biz.crm.cps.external.mdm.sdk.dto.MaterialGroupMdmPaginationDto;
import com.biz.crm.cps.external.mdm.sdk.dto.MaterialMdmPaginationDto;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * @author hecheng
 * @description:物料voservice实现
 * @date 2021/8/7 下午3:59
 */
@Service
@Slf4j
public class MaterialVoServiceImpl implements MaterialVoService {

  @Autowired
  private MaterialService materialService;
  @Autowired
  private ProductLevelService productLevelService;
  @Autowired
  private ProductMaterialService productMaterialService;
  @Autowired
  private MaterialGroupRelService materialGroupRelService;
  @Autowired
  private MaterialGroupService materialGroupService;
  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Autowired private DefaultLoginAuthenticationConfig defaultLoginAuthenticationConfig;

  @Override
  public MaterialDimensionVo findByMaterialCode(String materialCode) {
    //填充物料维度Vo
    //1.填充物料基础信息
    //2.填充商品物料信息
    //3.填充层级信息
    if (StringUtils.isBlank(materialCode)) {
      return null;
    }
    Material material = materialService.findByMaterialCode(materialCode);
    if (Objects.isNull(material)) {
      this.sync(materialCode);
      material = materialService.findByMaterialCode(materialCode);
      if (Objects.isNull(material)) {
        return null;
      }
    }
    MaterialDimensionVo vo = this.nebulaToolkitService.copyObjectByWhiteList(material, MaterialDimensionVo.class, Set.class, ArrayList.class);

    List<ProductMaterial> productMaterials = productMaterialService.findByMaterialCode(materialCode);
    vo.setProductMaterials(Sets.newHashSet());
    if (!CollectionUtils.isEmpty(productMaterials)) {
      List<ProductMaterialVo> productMaterialVos = (List<ProductMaterialVo>) this.nebulaToolkitService.copyCollectionByWhiteList(productMaterials, ProductMaterial.class, ProductMaterialVo.class, HashSet.class, ArrayList.class);
      vo.setProductMaterials(Sets.newHashSet(productMaterialVos));
    } else {
      vo.setProductMaterials(Sets.newHashSet());
    }

    if (StringUtils.isNotBlank(material.getProductLevelCode())) {
      List<ProductLevel> productLevelList = this.productLevelService.findAllParentsByProductLevelCode(material.getProductLevelCode());
      if (!CollectionUtils.isEmpty(productLevelList)) {
        List<ProductLevelVo> productLevelVos = (List<ProductLevelVo>) this.nebulaToolkitService.copyCollectionByWhiteList(productLevelList, ProductLevel.class, ProductLevelVo.class, HashSet.class, ArrayList.class);
        vo.setProductLevels(Sets.newHashSet(productLevelVos));
      } else {
        vo.setProductLevels(Sets.newHashSet());
      }
    }
    List<MaterialGroupRel> rels = this.materialGroupRelService.findByMaterialCode(materialCode);
    if (!CollectionUtils.isEmpty(rels)) {
      List<MaterialGroupRelVo> relVos = (List<MaterialGroupRelVo>) this.nebulaToolkitService.copyCollectionByWhiteList(rels, MaterialGroupRel.class, MaterialGroupRelVo.class, HashSet.class, ArrayList.class);
      vo.setMaterialGroupRels(Sets.newHashSet(relVos));
    } else {
      vo.setMaterialGroupRels(Sets.newHashSet());
    }
    return vo;
  }

  /**
   * 通过维度类型 获取维度code 1。是否属于单品 2。是否属于物料组 3。是否是全部 4。是否是层级
   *
   * @param materialDimensionDto
   * @param dimensionType
   * @return
   */
  @Override
  public List<String> findDimensionCodes(MaterialDimensionDto materialDimensionDto, String dimensionType) {
    if (!Objects.isNull(materialDimensionDto)) {
      //1。是否属于单品
      if (Objects.equals(dimensionType, MaterialDimensionEnum.MATERIAL.getDictCode())) {
        return Lists.newArrayList(materialDimensionDto.getMaterialCode());
      } else if (Objects.equals(dimensionType, MaterialDimensionEnum.MATERIAL_GROUP.getDictCode())) {
        //2。是否属于物料组
        Set<MaterialGroupRelDto> rels = materialDimensionDto.getMaterialGroupRels();
        if (!CollectionUtils.isEmpty(rels)) {
          return rels.stream().map(MaterialGroupRelDto::getMaterialGroupCode).collect(Collectors.toList());
        }
      } else if (Objects.equals(dimensionType, MaterialDimensionEnum.ALL.getDictCode())) {
        //3。是否是全部
        return Lists.newArrayList(MaterialDimensionConstant.DIMENSION_ALL);
      } else {
        //4。是否是层级
        Set<ProductLevelDto> levelDtos = materialDimensionDto.getProductLevels();
        if (!CollectionUtils.isEmpty(levelDtos)) {
          ProductLevelDto levelDto = levelDtos.stream().filter(item -> Objects.equals(item.getProductLevelType(), dimensionType)).findFirst().orElse(null);
          if (levelDto != null) {
            return Lists.newArrayList(levelDto.getProductLevelCode());
          }
        }
      }
    }
    return null;
  }

  @Override
  public List<String> findMaterialCodeByDimensionCodesAndDimensionType(List<String> dimensionCodes, String dimensionType) {
    if (StringUtils.isNotBlank(dimensionType) && !CollectionUtils.isEmpty(dimensionCodes)) {
      //1。是否属于单品
      if (Objects.equals(dimensionType, MaterialDimensionEnum.MATERIAL.getDictCode())) {
        List<Material> materials = this.materialService.findByMaterialCodes(dimensionCodes);
        if (!CollectionUtils.isEmpty(materials)) {
          return materials.stream().map(Material::getMaterialCode).collect(Collectors.toList());
        }
      } else if (Objects.equals(dimensionType, MaterialDimensionEnum.MATERIAL_GROUP.getDictCode())) {
        //2。是否属于物料组
        List<MaterialGroupRel> rels = this.materialGroupRelService.findByMaterialGroupCodes(dimensionCodes);
        if (!CollectionUtils.isEmpty(rels)) {
          return rels.stream().map(MaterialGroupRel::getMaterialCode).collect(Collectors.toList());
        }
      } else if (Objects.equals(dimensionType, MaterialDimensionEnum.ALL.getDictCode())) {
        //3。是否是全部
        return Lists.newArrayList(MaterialDimensionConstant.DIMENSION_ALL);
      } else {
        //4。是否是层级
        List<Material> materials = this.materialService.findAllChildrenByProductLevelCodes(dimensionCodes);
        if (!CollectionUtils.isEmpty(materials)) {
          return materials.stream().map(Material::getMaterialCode).collect(Collectors.toList());
        }
      }
    }
    return null;
  }

  @Override
  public List<MaterialVo> findByMaterialCode(List<String> materialCodes) {
    if (CollectionUtils.isEmpty(materialCodes)) {
      return null;
    }
    List<Material> materials = this.materialService.findByMaterialCodes(materialCodes);
    if (!CollectionUtils.isEmpty(materials)) {
      return (List<MaterialVo>) this.nebulaToolkitService.copyCollectionByWhiteList(materials, Material.class, MaterialVo.class, HashSet.class, ArrayList.class);
    }
    return null;
  }

  @Override
  public void sync(List<String> dimensionCodes, String dimensionType) {
    this.defaultLoginAuthenticationConfig.defaultAdminAuthentication();
    if (StringUtils.isNotBlank(dimensionType) && !CollectionUtils.isEmpty(dimensionCodes)) {
      try {
        //1。是否属于单品
        if (Objects.equals(dimensionType, MaterialDimensionEnum.MATERIAL.getDictCode())) {
          MaterialMdmPaginationDto dto = new MaterialMdmPaginationDto();
          dto.setMaterialCodes(dimensionCodes);
          Pageable pageable = PageRequest.of(1, MaterialDimensionConstant.SYNC_SIZE);
          this.materialService.sync(pageable, dto);
        } else if (Objects.equals(dimensionType, MaterialDimensionEnum.MATERIAL_GROUP.getDictCode())) {
          //2。是否属于物料组
          Pageable pageable = PageRequest.of(1, MaterialDimensionConstant.SYNC_SIZE);
          MaterialGroupMdmPaginationDto dto = new MaterialGroupMdmPaginationDto();
          dto.setMaterialGroupCodes(dimensionCodes);
          this.materialGroupService.sync(pageable, dto);
        } else if (Objects.equals(dimensionType, MaterialDimensionEnum.ALL.getDictCode())) {
          //3。是否是全部
          MaterialMdmPaginationDto dto = new MaterialMdmPaginationDto();
          Pageable pageable = PageRequest.of(1, MaterialDimensionConstant.SYNC_SIZE);
          this.materialService.sync(pageable, dto);
        } else {
          //4。是否是层级
          MaterialMdmPaginationDto dto = new MaterialMdmPaginationDto();
          dto.setProductLevelCodes(dimensionCodes);
          Pageable pageable = PageRequest.of(1, MaterialDimensionConstant.SYNC_SIZE);
          this.materialService.sync(pageable, dto);
        }
      } catch (RuntimeException e) {
        log.error("同步物料失败！" + e.getMessage(), e);
      }
    }
  }

  /**
   * 同步物料 ，和物料相关的物料组 这里为了不阻挡其他业务进行所以只记录日志 不抛出异常
   *
   * @param materialCode
   */
  private void sync(String materialCode) {
    try {
      MaterialMdmPaginationDto dto = new MaterialMdmPaginationDto();
      dto.setMaterialCode(materialCode);
      Pageable pageable = PageRequest.of(1, MaterialDimensionConstant.SYNC_SIZE);
      this.materialService.sync(pageable, dto);
      Pageable pageableGroup = PageRequest.of(1, MaterialDimensionConstant.SYNC_SIZE);
      MaterialGroupMdmPaginationDto groupDto = new MaterialGroupMdmPaginationDto();
      dto.setMaterialCode(materialCode);
      this.materialGroupService.sync(pageableGroup, groupDto);
    } catch (RuntimeException e) {
      log.error("同步物料【" + materialCode + "】失败！" + e.getMessage(), e);
    }
  }
}
