package com.biz.crm.cps.business.product.local.service.internal;

import com.biz.crm.cps.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.cps.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.cps.business.product.local.entity.ProductLevel;
import com.biz.crm.cps.business.product.local.repository.ProductLevelRepository;
import com.biz.crm.cps.business.product.local.service.ProductLevelService;
import com.biz.crm.cps.business.product.sdk.common.constant.ProductLevelCodeConstant;
import com.biz.crm.cps.business.product.sdk.common.enums.ProductLevelTypeEnum;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.security.core.context.SecurityContext;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @author hecheng
 * @description:商品层级service实现
 * @date 2021/8/6 下午3:59
 */
@Service
public class ProductLevelServiceImpl implements ProductLevelService {

  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;
  @Autowired(required = false)
  private GenerateCodeService generateCodeService;
  @Autowired
  private ProductLevelRepository productLevelRepository;

  @Transactional
  @Override
  public ProductLevel create(ProductLevel productLevel) {
    ProductLevel current = this.createForm(productLevel);
    this.productLevelRepository.save(current);
    return current;
  }

  @Override
  public ProductLevel createForm(ProductLevel productLevel) {
    /*
     * 对静态模型的保存操作过程为：
     * 1、如果当前模型对象不是主模型
     *  1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     *  1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     *  1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     * 2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *   2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *   2.3.2、以及验证每个分组的OneToMany明细信息
     * */
    Date now = new Date();
    String account = this.getLoginAccountName();
    productLevel.setTenantCode(TenantUtils.getTenantCode());
    productLevel.setCreateAccount(account);
    productLevel.setCreateTime(now);
    productLevel.setModifyAccount(account);
    productLevel.setModifyTime(now);
    this.createValidation(productLevel);
    if (StringUtils.isBlank(productLevel.getDelFlag())) {
      productLevel.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    }
    if (StringUtils.isBlank(productLevel.getEnableStatus())) {
      productLevel.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    }
    String productLevelCode = productLevel.getProductLevelCode();
    if (StringUtils.isBlank(productLevelCode)) {
      productLevelCode =
              this.generateCodeService.generateCode(ProductLevelCodeConstant.PRODUCT_LEVEL_CODE, 1)
                      .get(0);
      productLevel.setProductLevelCode(productLevelCode);
    }
    String parentCode = productLevel.getParentCode();
    String ruleCode = "";
    if (StringUtils.isNotBlank(parentCode)) {
      ProductLevel parent = this.findByProductLevelCode(parentCode);
      Validate.notNull(parent, "上级商品层级不存在！");
      ruleCode = parent.getRuleCode() + productLevelCode
              + ProductLevelCodeConstant.PRODUCT_LEVEL_RULE_CODE_SEPARATOR;
    } else {
      ruleCode = productLevelCode
              + ProductLevelCodeConstant.PRODUCT_LEVEL_RULE_CODE_SEPARATOR;
    }
    productLevel.setRuleCode(ruleCode);
    return productLevel;
  }

  @Transactional
  @Override
  public ProductLevel update(ProductLevel productLevel) {
    ProductLevel current = this.updateForm(productLevel);

    return current;
  }


  @Override
  public ProductLevel updateForm(ProductLevel productLevel) {
    /*
     * 对静态模型的修改操作的过程为：
     * 1、如果当前模型对象不是主模型
     *  1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     *  1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     *  1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理
     *
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     *  2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *    2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *    2.3.2、以及验证每个分组的OneToMany明细信息
     */
    this.updateValidation(productLevel);
    //这里可根据id或者code更新
    String currentId = productLevel.getId();
    String productLevelCode = productLevel.getProductLevelCode();
    ProductLevel current = null;
    if (StringUtils.isNotBlank(currentId)) {
      current = this.productLevelRepository.getById(currentId);
    } else if (StringUtils.isNotBlank(productLevelCode)) {
      current = this.findByProductLevelCode(productLevelCode);
    }
    current = Validate.notNull(current, "未发现指定的原始模型对象信");

    BeanUtils
            .copyProperties(productLevel, current, "id", "modifyTime", "createAccount",
                    "createTime",
                    "tenantCode", "ruleCode");

    // 开始赋值——更新时间与更新人
    Date now = new Date();
    String account = this.getLoginAccountName();
    current.setModifyAccount(account);
    current.setModifyTime(now);
    this.productLevelRepository.saveOrUpdate(current);

    return current;
  }


  @Override
  public ProductLevel findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return productLevelRepository.getById(id);
  }

  @Override
  public ProductLevel findDetailsById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return productLevelRepository.findDetailsById(id);
  }

  @Transactional
  @Override
  public void enableBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "请选中要操作的数据");
    this.productLevelRepository.updateEnableStatusByIdIn(EnableStatusEnum.ENABLE, ids);
  }

  @Transactional
  @Override
  public void disableBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "请传入要操作的数据");
    this.productLevelRepository.updateEnableStatusByIdIn(EnableStatusEnum.DISABLE, ids);
  }

  @Transactional
  @Override
  public List<ProductLevel> createBatch(List<ProductLevel> productLevels) {
    if (!CollectionUtils.isEmpty(productLevels)) {
      for (ProductLevel productLevel : productLevels) {
        this.createForm(productLevel);
      }
      this.productLevelRepository.saveBatch(productLevels);
      return productLevels;
    }
    return Lists.newArrayList();
  }

  /**
   * 批量更新
   *
   * @param productLevels
   * @return
   */
  private List<ProductLevel> updateBatch(List<ProductLevel> productLevels) {
    if (!CollectionUtils.isEmpty(productLevels)) {
      for (ProductLevel productLevel : productLevels) {
        this.updateForm(productLevel);
      }
      return productLevels;
    }
    return Lists.newArrayList();
  }

  @Transactional
  @Override
  public void saveBatch(List<ProductLevel> target) {
    //TODO 这里要从字典中获取 类型 按类型从最父级开始创建,现在先写死从枚举中获取

    //1  先区分出 新增的 和更新的
    if (!CollectionUtils.isEmpty(target)) {
      Arrays.stream(ProductLevelTypeEnum.values()).forEach(item -> {
        List<ProductLevel> tempTarget = target.stream().filter(targetItem -> {
          return Objects.equals(targetItem.getProductLevelType(), item.getKey());
        }).collect(Collectors.toList());
        List<String> productLevelCodes = tempTarget.stream().map(ProductLevel::getProductLevelCode)
                .distinct().collect(Collectors.toList());
        List<ProductLevel> oldList=Lists.newArrayList();
        if(!CollectionUtils.isEmpty(productLevelCodes)){
          oldList = this.productLevelRepository.findByProductLevelCodes(productLevelCodes);
        }
        List<ProductLevel> addList = Lists.newArrayList();
        List<ProductLevel> needUpdateList = Lists.newArrayList();
        List<ProductLevel> delList = Lists.newArrayList();
        this.nebulaToolkitService.collectionDiscrepancy(tempTarget, oldList, ProductLevel::getProductLevelCode,delList,needUpdateList, addList);
        this.createBatch(addList);
        if(!CollectionUtils.isEmpty(needUpdateList)){
          List<String> needUpdateCodeList = needUpdateList.stream().map(ProductLevel::getProductLevelCode).collect(Collectors.toList());
          List<ProductLevel> updateList = target.stream().filter(pitem->needUpdateCodeList.contains(pitem.getProductLevelCode())).collect(Collectors.toList());
          this.updateBatch(updateList);
        }
      });
    }
  }

  @Override
  public ProductLevel findByProductLevelCode(String productLevelCode) {
    if (StringUtils.isBlank(productLevelCode)) {
      return null;
    }
    return this.productLevelRepository.findByProductLevelCode(productLevelCode);
  }

  @Override
  public ProductLevel findByParentCode(String parentCode) {
    if (StringUtils.isBlank(parentCode)) {
      return null;
    }
    return this.productLevelRepository.findByParentCode(parentCode);
  }

  @Override
  public List<ProductLevel> findAllParentsByProductLevelCode(String productLevelCode) {
    if (StringUtils.isBlank(productLevelCode)) {
      return Lists.newArrayList();
    }
    return this.productLevelRepository.findAllParentsByProductLevelCode(productLevelCode);
  }

  @Override
  public List<ProductLevel> findByProductLevelCodes(List<String> productLevelCodes) {
    if (CollectionUtils.isEmpty(productLevelCodes)) {
      return null;
    }
    return this.productLevelRepository.findByProductLevelCodes(productLevelCodes);
  }


  /**
   * 在创建一个新的productLevel模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(ProductLevel productLevel) {
    Validate.notNull(productLevel, "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    productLevel.setId(null);
    Validate.notBlank(productLevel.getTenantCode(), "添加信息时，租户编号不能为空！");
    Validate.notBlank(productLevel.getProductLevelName(), "添加信息时，商品层级名称不能为空！");
    Validate.notBlank(productLevel.getProductLevelType(), "添加信息时，商品层级类型不能为空！");
    Validate.notBlank(productLevel.getRuleCode(), "添加信息时，商品层级ruleCode不能为空！");
    Validate.notNull(ProductLevelTypeEnum.getByKey(productLevel.getProductLevelType()),
            "添加信息时，商品层级类型不合法！");
    Validate.isTrue(productLevel.getProductLevelName().length() < 128,
            "商品层级名称，在进行添加时填入值超过了限定长度(128)，请检查!");

    if (StringUtils.isNotBlank(productLevel.getProductLevelCode())) {
      List<ProductLevel> list = this.productLevelRepository.lambdaQuery()
              .eq(ProductLevel::getProductLevelCode, productLevel.getProductLevelCode())
              .select(ProductLevel::getId)
              .list();
      Validate.isTrue(
              CollectionUtils.isEmpty(list), "编码[" + productLevel.getProductLevelCode() + "]已存在");
    }
  }

  /**
   * 在更新一个的productLevel模型对象之前，检查对象各属性的正确性
   *
   * @param productLevel
   */
  private void updateValidation(ProductLevel productLevel) {
    Validate.isTrue(!(StringUtils.isBlank(productLevel.getId()) && StringUtils
            .isBlank(productLevel.getProductLevelCode())), "修改信息时，当期信息的数据编号（主键/编码）必须有值！");
    Validate.notBlank(productLevel.getProductLevelName(), "修改信息时，商品层级名称不能为空！");
    Validate.notBlank(productLevel.getProductLevelType(), "修改信息时，商品层级类型不能为空！");
    Validate.notBlank(productLevel.getRuleCode(), "修改信息时，商品层级ruleCode不能为空！");
    Validate.notNull(ProductLevelTypeEnum.getByKey(productLevel.getProductLevelType()),
            "修改信息时，商品层级类型不合法！");
    Validate.isTrue(productLevel.getProductLevelName().length() < 128,
            "商品层级名称，在进行添加时填入值超过了限定长度(128)，请检查!");
  }

  /**
   * 获取当前登录人名称
   *
   * @return
   */
  private String getLoginAccountName() {
    SecurityContext context = SecurityContextHolder.getContext();
    String account = "admin";
    if (context != null && context.getAuthentication() != null) {
      account = context.getAuthentication().getName();
    }
    return account;
  }
}
