package com.biz.crm.cps.business.product.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.cps.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.cps.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.cps.business.product.local.entity.Material;
import com.biz.crm.cps.business.product.local.entity.Product;
import com.biz.crm.cps.business.product.local.entity.ProductLevel;
import com.biz.crm.cps.business.product.local.entity.ProductMaterial;
import com.biz.crm.cps.business.product.local.repository.ProductRepository;
import com.biz.crm.cps.business.product.local.service.MaterialService;
import com.biz.crm.cps.business.product.local.service.ProductLevelService;
import com.biz.crm.cps.business.product.local.service.ProductMaterialService;
import com.biz.crm.cps.business.product.local.service.ProductService;
import com.biz.crm.cps.business.product.sdk.common.constant.ProductCodeConstant;
import com.biz.crm.cps.business.product.sdk.common.enums.ProductShelfStatusEnum;
import com.biz.crm.cps.business.product.sdk.common.enums.ProductTypeEnum;
import com.biz.crm.cps.business.product.sdk.dto.ProductPaginationDto;
import com.biz.crm.cps.business.product.sdk.event.ProductEventListener;
import com.biz.crm.cps.business.product.sdk.vo.ProductVo;
import com.biz.crm.cps.external.mdm.sdk.dto.ProductMdmPaginationDto;
//import com.biz.crm.cps.external.mdm.sdk.service.ProductMdmService;
import com.biz.crm.cps.external.mdm.sdk.vo.MaterialMdmVo;
import com.biz.crm.cps.external.mdm.sdk.vo.ProductLevelMdmVo;
import com.biz.crm.cps.external.mdm.sdk.vo.ProductMaterialMdmVo;
import com.biz.crm.cps.external.mdm.sdk.vo.ProductMdmVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.TreeSet;
import java.util.stream.Collectors;

import org.apache.commons.lang3.ObjectUtils;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.security.core.context.SecurityContext;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;


/**
 * @author hecheng
 * @description:商品service实现
 * @date 2021/8/7 下午3:59
 */
@Service
public class ProductServiceImpl implements ProductService {

  @Autowired
  private ProductRepository productRepository;
  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;
  @Autowired(required = false)
  private List<ProductEventListener> productEventListenerList;
  @Autowired(required = false)
  private GenerateCodeService generateCodeService;
  @Autowired
  private ProductLevelService productLevelService;
  @Autowired
  private MaterialService materialService;
  @Autowired
  private ProductMaterialService productMaterialService;
//  @Autowired(required = false)
//  private ProductMdmService productMdmService;

  @Transactional
  @Override
  public Product create(Product product) {
    Product current = this.createForm(product);
    this.productRepository.save(current);
    return current;
  }

  @Override
  public Product createForm(Product product) {
    /*
     * 对静态模型的保存操作过程为：
     * 1、如果当前模型对象不是主模型
     *  1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     *  1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     *  1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     * 2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *   2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *   2.3.2、以及验证每个分组的OneToMany明细信息
     * */
    Date now = new Date();
    String account = this.getLoginAccountName();
    product.setTenantCode(TenantUtils.getTenantCode());
    product.setCreateAccount(account);
    product.setCreateTime(now);
    product.setModifyAccount(account);
    product.setModifyTime(now);

    this.createValidation(product);
    if (StringUtils.isBlank(product.getDelFlag())) {
      product.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    }
    if (StringUtils.isBlank(product.getEnableStatus())) {
      product.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    }
    if (StringUtils.isBlank(product.getProductCode())) {
      product.setProductCode(
              this.generateCodeService.generateCode(ProductCodeConstant.PRODUCT_CODE, 1)
                      .get(0));
    }
    return product;
  }

  @Transactional
  @Override
  public Product update(Product product) {
    Product current = this.updateForm(product);

    return current;
  }

  @Override
  public Product updateForm(Product product) {
    /*
     * 对静态模型的修改操作的过程为：
     * 1、如果当前模型对象不是主模型
     *  1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     *  1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     *  1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理
     *
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     *  2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *    2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *    2.3.2、以及验证每个分组的OneToMany明细信息
     */
    this.updateValidation(product);
    //这里可根据id或者code更新
    String currentId = product.getId();
    String productCode = product.getProductCode();
    Product current = null;
    if (StringUtils.isNotBlank(currentId)) {
      current = this.productRepository.getById(currentId);
    } else if (StringUtils.isNotBlank(productCode)) {
      current = this.findByProductCode(productCode);
    }
    current = Validate.notNull(current, "未发现指定的原始模型对象信");
    // 复制旧对象
    ProductVo oldVo = null;
    if (!CollectionUtils.isEmpty(productEventListenerList)) {
      oldVo = this.nebulaToolkitService
              .copyObjectByWhiteList(current, ProductVo.class, HashSet.class, ArrayList.class);
    }
    BeanUtils
            .copyProperties(product, current, "id", "modifyTime", "createAccount", "createTime",
                    "tenantCode");

    // 开始赋值——更新时间与更新人
    Date now = new Date();
    String account = this.getLoginAccountName();
    current.setModifyAccount(account);
    current.setModifyTime(now);
    this.productRepository.saveOrUpdate(current);
    // 发送修改通知
    if (!CollectionUtils.isEmpty(productEventListenerList)) {
      ProductVo newVo = this.nebulaToolkitService
              .copyObjectByWhiteList(current, ProductVo.class, HashSet.class, ArrayList.class);
      for (ProductEventListener event : productEventListenerList) {
        event.onChange(oldVo, newVo);
      }
    }
    return current;
  }

  @Override
  public Product findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return productRepository.getById(id);
  }

  @Override
  public Product findDetailsById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return productRepository.findDetailsById(id);
  }

  @Transactional
  @Override
  public void enableBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "请选中要操作的数据");
    this.productRepository.updateEnableStatusByIdIn(EnableStatusEnum.ENABLE, ids);
  }

  @Transactional
  @Override
  public void disableBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "请传入要操作的数据");
    this.productRepository.updateEnableStatusByIdIn(EnableStatusEnum.DISABLE, ids);
    if (!CollectionUtils.isEmpty(productEventListenerList)) {
      List<Product> terminals = this.productRepository.findByIds(ids);
      List<ProductVo> voList = (List<ProductVo>) this.nebulaToolkitService
              .copyCollectionByWhiteList(terminals, Product.class, ProductVo.class, HashSet.class,
                      ArrayList.class);
      for (ProductEventListener event : productEventListenerList) {
        event.onDisable(voList);
      }
    }
  }

  @Transactional
  @Override
  public List<Product> createBatch(List<Product> products) {
    if (!CollectionUtils.isEmpty(products)) {
      for (Product product : products) {
        this.createForm(product);
      }
      this.productRepository.saveBatch(products);
      return products;
    }
    return Lists.newArrayList();
  }

  /**
   * 批量更新
   *
   * @param products
   * @return
   */
  private List<Product> updateBatch(List<Product> products) {
    if (!CollectionUtils.isEmpty(products)) {
      for (Product product : products) {
        this.updateForm(product);
      }
      return products;
    }
    return Lists.newArrayList();
  }

  @Override
  public Product findByProductCode(String productCode) {
    if (StringUtils.isBlank(productCode)) {
      return null;
    }
    return productRepository.findByProductCode(productCode);
  }

  @Override
  public List<Product> findByProductLevelCode(String productLevelCode) {
    if (StringUtils.isBlank(productLevelCode)) {
      return null;
    }
    return productRepository.findByProductLevelCode(productLevelCode);
  }

  @Transactional(rollbackFor = Exception.class)
  @Override
  public void sync(ProductMdmPaginationDto productMdmPaginationDto) {
    /**
     * 先获取到所有符号条件的mdm侧的商品信息，
     * 再批量新增到CPS系统 （这里同步时均以本系统生成的id为准，外部系统id 保留在扩展字段中）
     * 1.新增 商品层级信息
     * 2.新增 物料信息
     * 3.新增 商品信息
     * 4.新增 商品物料关系
     */
    //TODO crm 本次没有实现product的同步
    List<ProductMdmVo> productMdmVos = null;

    Validate.notEmpty(productMdmVos, "没有发现需要同步的商品信息");
    //1
    List<ProductLevelMdmVo> productLevelMdmVos = productMdmVos.stream()
            .flatMap(item -> item.getProductLevels().stream())
            .collect(
                    Collectors.collectingAndThen(
                            Collectors.toCollection(
                                    () -> new TreeSet<>(
                                            Comparator.comparing(ProductLevelMdmVo::getId))),
                            ArrayList::new)
            );
    this.bulidProductLevel(productLevelMdmVos);
    //2
    List<ProductMaterialMdmVo> productMaterialMdmVos = productMdmVos.stream()
            .flatMap(item -> item.getProductMaterials().stream())
            .collect(
                    Collectors.collectingAndThen(
                            Collectors.toCollection(
                                    () -> new TreeSet<>(
                                            Comparator.comparing(ProductMaterialMdmVo::getId))),
                            ArrayList::new)
            );
    List<MaterialMdmVo> materialMdmVos = productMaterialMdmVos.stream()
            .map(ProductMaterialMdmVo::getMaterial).collect(
                    Collectors.collectingAndThen(
                            Collectors.toCollection(
                                    () -> new TreeSet<>(
                                            Comparator.comparing(MaterialMdmVo::getId))),
                            ArrayList::new)
            );
    this.bulidMaterial(materialMdmVos);
    //3.
    this.bulidProduct(productMdmVos);
    //4.
    this.bulidProductMaterial(productMaterialMdmVos);
  }

  @Override
  public Page<Product> findByConditions(Pageable pageable,
          ProductPaginationDto productPaginationDto) {
    ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    if(productPaginationDto == null) {
      productPaginationDto = new ProductPaginationDto();
    }
    productPaginationDto.setTenantCode(TenantUtils.getTenantCode());
    if (StringUtils.isNotBlank(productPaginationDto.getProductLevelCode())) {
      ProductLevel productLevel = productLevelService
              .findByProductLevelCode(productPaginationDto.getProductLevelCode());
      if (!Objects.isNull(productLevel)) {
        productPaginationDto.setProductLevelCode(null);
        productPaginationDto.setRuleCode(productLevel.getRuleCode());
      }
    }
    return productRepository.findByConditions(pageable, productPaginationDto);
  }


  @Transactional
  @Override
  public void saveBatch(List<Product> target) {
    //1  先区分出 新增的 和更新的
    if (!CollectionUtils.isEmpty(target)) {
      List<String> productCodes = target.stream().map(Product::getProductCode)
              .distinct().collect(Collectors.toList());
      List<Product> oldList = this.productRepository
              .findByProductCodes(productCodes);
      List<Product> addList = Lists.newArrayList();
      List<Product> needUpdateList = Lists.newArrayList();
      List<Product> delList = Lists.newArrayList();
      this.nebulaToolkitService.collectionDiscrepancy(target, oldList, Product::getProductCode, delList,needUpdateList, addList);
      this.createBatch(addList);
      if(!CollectionUtils.isEmpty(needUpdateList)){
        List<String> needUpdateCodeList = needUpdateList.stream().map(Product::getProductCode).collect(Collectors.toList());
        List<Product> updateList = target.stream().filter(pitem->needUpdateCodeList.contains(pitem.getProductCode())).collect(Collectors.toList());
        this.updateBatch(updateList);
      }
    }
  }

  /**
   * 根据商品vo 构建商品
   *
   * @param productMdmVos
   * @return
   */
  private List<Product> bulidProduct(List<ProductMdmVo> productMdmVos) {
    List<Product> target = productMdmVos.stream().map(item -> {
      Product product = this.nebulaToolkitService
              .copyObjectByWhiteList(item, Product.class, HashSet.class,
                      ArrayList.class);
      product.setExternalIdentifier(item.getId());
      product.setId(null);
      return product;
    }).collect(Collectors.toList());
    this.saveBatch(target);
    return target;
  }

  /**
   * 根据商品物料关系vo 构建商品物料关系
   *
   * @param productMaterialMdmVos
   * @return
   */
  private List<ProductMaterial> bulidProductMaterial(
          List<ProductMaterialMdmVo> productMaterialMdmVos) {
    List<ProductMaterial> target = productMaterialMdmVos.stream().map(item -> {
      ProductMaterial productMaterial = this.nebulaToolkitService
              .copyObjectByWhiteList(item, ProductMaterial.class, HashSet.class,
                      ArrayList.class);
      productMaterial.setExternalIdentifier(item.getId());
      productMaterial.setId(null);
      return productMaterial;
    }).collect(Collectors.toList());
    this.productMaterialService.saveBatch(target);
    return target;
  }

  /**
   * 根据物料vo 构建物料
   *
   * @param materialMdmVos
   * @return
   */
  private List<Material> bulidMaterial(List<MaterialMdmVo> materialMdmVos) {
    List<Material> target = materialMdmVos.stream().map(item -> {
      Material material = new Material();
      BeanUtils
              .copyProperties(item, material, "costPrice");
      material.setCostPrice(new BigDecimal(item.getCostPrice()));
      material.setExternalIdentifier(item.getId());
      material.setId(null);
      return material;
    }).collect(Collectors.toList());
    this.materialService.saveBatch(target);
    return target;
  }

  /**
   * 根据商品层级类型 筛选符合条件的层级Vo构建商品层级
   *
   * @param productLevelMdmVos
   * @return
   */
  private List<ProductLevel> bulidProductLevel(
          List<ProductLevelMdmVo> productLevelMdmVos) {
    List<ProductLevel> target = productLevelMdmVos.stream().map(item -> {
      ProductLevel productLevel = this.nebulaToolkitService
              .copyObjectByWhiteList(item, ProductLevel.class, HashSet.class,
                      ArrayList.class);
      productLevel.setExternalIdentifier(item.getId());
      productLevel.setId(null);
      return productLevel;
    }).collect(Collectors.toList());
    this.productLevelService.saveBatch(target);
    return target;
  }

  /**
   * 在创建一个新的productLevel模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(Product product) {
    Validate.notNull(product, "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(product.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    product.setId(null);
    Validate.notBlank(product.getTenantCode(), "添加信息时，租户编号不能为空！");
    Validate.notBlank(product.getProductName(), "添加信息时，商品名称不能为空！");
    Validate.notBlank(product.getProductType(), "添加信息时，商品类型不能为空！");
    Validate.isTrue(product.getProductName().length() < 128, "商品名称，在进行添加时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(product.getBarCode() == null || product.getBarCode().length() < 64,
            "条形码，在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(product.getSaleCompany() == null || product.getSaleCompany().length() < 128,
            "销售公司，在进行添加时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(
            product.getPrimaryPictureUrl() == null || product.getPrimaryPictureUrl().length() < 255,
            "主图片url，在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(
            product.getSaleUnit() == null || product.getSaleUnit().length() < 64,
            "销售单位，在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(
            product.getSpec() == null || product.getSpec().length() < 64,
            "规格，在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(
            product.getBaseUnit() == null || product.getBaseUnit().length() < 64,
            "基本单位，在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.notNull(ProductTypeEnum.getByKey(product.getProductType()), "添加信息时，商品类型不合法！");
    Validate.notNull(ProductShelfStatusEnum.getByKey(product.getIsShelf()), "添加信息时，商品上下架状态不合法！");

    if (StringUtils.isNotBlank(product.getProductCode())) {
      List<Product> list = this.productRepository.lambdaQuery()
              .eq(Product::getProductCode, product.getProductCode())
              .select(Product::getId)
              .list();
      Validate.isTrue(CollectionUtils.isEmpty(list), "编码[" + product.getProductCode() + "]已存在");
    }
    if (StringUtils.isNotBlank(product.getProductLevelCode())) {
      ProductLevel productLevel = this.productLevelService
              .findByProductLevelCode(product.getProductLevelCode());
      Validate.notNull(productLevel, "商品层级编码[" + product.getProductLevelCode() + "]不存在");
    }
  }

  private void updateValidation(Product product) {
    Validate.isTrue(!(StringUtils.isBlank(product.getId()) && StringUtils
            .isBlank(product.getProductCode())), "修改信息时，当期信息的数据编号（主键/编码）必须有值！");
    Validate.notBlank(product.getProductName(), "修改信息时，商品层级名称不能为空！");
    Validate.notBlank(product.getProductType(), "修改信息时，商品层级类型不能为空！");
    Validate.isTrue(product.getProductName().length() < 128, "商品名称，在进行修改时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(product.getBarCode() == null || product.getBarCode().length() < 64,
            "条形码，在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(product.getSaleCompany() == null || product.getSaleCompany().length() < 128,
            "销售公司，在进行修改时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(
            product.getPrimaryPictureUrl() == null || product.getPrimaryPictureUrl().length() < 255,
            "主图片url，在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(
            product.getSaleUnit() == null || product.getSaleUnit().length() < 64,
            "销售单位，在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(
            product.getSpec() == null || product.getSpec().length() < 64,
            "规格，在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(
            product.getBaseUnit() == null || product.getBaseUnit().length() < 64,
            "基本单位，在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.notNull(ProductTypeEnum.getByKey(product.getProductType()), "修改信息时，商品类型不合法！");
    Validate.notNull(ProductShelfStatusEnum.getByKey(product.getIsShelf()), "修改信息时，商品上下架状态不合法！");

    if (StringUtils.isNotBlank(product.getProductLevelCode())) {
      ProductLevel productLevel = this.productLevelService
              .findByProductLevelCode(product.getProductLevelCode());
      Validate.notNull(productLevel, "商品层级编码[" + product.getProductLevelCode() + "]不存在");
    }
  }

  /**
   * 获取当前登录人名称
   *
   * @return
   */
  private String getLoginAccountName() {
    SecurityContext context = SecurityContextHolder.getContext();
    String account = "admin";
    if (context != null && context.getAuthentication() != null) {
      account = context.getAuthentication().getName();
    }
    return account;
  }
}
