package com.biz.crm.cps.business.policy.quantify.local.service.internal;

import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.cps.bisiness.policy.quantify.sdk.dto.QuantifyPolicyDto;
import com.biz.crm.cps.business.agreement.sdk.dto.AgreementPolicyDto;
import com.biz.crm.cps.business.agreement.sdk.dto.ProfitAgreementTemplatePaginationDto;
import com.biz.crm.cps.business.agreement.sdk.service.AgreementTemplateVoService;
import com.biz.crm.cps.business.agreement.sdk.vo.AgreementTemplateVo;
import com.biz.crm.cps.business.agreement.sdk.vo.TemplateOrgRelationshipVo;
import com.biz.crm.cps.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.cps.business.policy.quantify.local.entity.QuantifyExpression;
import com.biz.crm.cps.business.policy.quantify.local.entity.QuantifyPolicy;
import com.biz.crm.cps.business.policy.quantify.local.entity.QuantifyRange;
import com.biz.crm.cps.business.policy.quantify.local.repository.QuantifyConfigurationRepository;
import com.biz.crm.cps.business.policy.quantify.local.repository.QuantifyExpressionRepository;
import com.biz.crm.cps.business.policy.quantify.local.repository.QuantifyPolicyRepository;
import com.biz.crm.cps.business.policy.quantify.local.repository.QuantifyRangeRepository;
import com.biz.crm.cps.business.policy.quantify.local.service.QuantifyPolicyService;
import com.biz.crm.cps.business.policy.quantify.local.service.observer.QuantifyPolicyMountRegisterImpl;
import com.biz.crm.cps.business.policy.quantify.local.service.observer.QuantifyPolicyPojoObserverRegister;
import com.biz.crm.cps.business.product.sdk.common.constant.MaterialDimensionConstant;
import com.biz.crm.cps.business.product.sdk.service.MaterialVoService;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @author jerry7
 * @date 2021/8/9 19:50
 * 包量政策接口实现类
 */
@Service
public class QuantifyPolicyServiceImpl implements QuantifyPolicyService {

  @Autowired
  private QuantifyPolicyPojoObserverRegister policyPojoObserver;

  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Autowired
  private QuantifyPolicyRepository quantifyPolicyRepository;

  @Autowired
  private QuantifyConfigurationRepository quantifyConfigurationRepository;

  @Autowired
  private QuantifyRangeRepository quantifyRangeRepository;

  @Autowired
  private QuantifyExpressionRepository quantifyExpressionRepository;

  @Autowired
  private LoginUserService loginUserService;

  @Autowired
  private AgreementTemplateVoService agreementTemplateVoService;

  @Autowired
  @Qualifier("QuantifyPolicyMountRegisterImpl")
  private QuantifyPolicyMountRegisterImpl quantifyPolicyMountRegister;


  @Autowired
  private MaterialVoService materialVoService;

  @Override
  @Transactional
  public QuantifyPolicy create(AgreementPolicyDto agreementPolicyDto) {
    if (agreementPolicyDto == null) {
      return null;
    }
    QuantifyPolicy quantifyPolicy = nebulaToolkitService.copyObjectByWhiteList(agreementPolicyDto, QuantifyPolicy.class, HashSet.class, ArrayList.class, "quantifyConfigurations", "quantifyConfigurations.quantifyRanges", "quantifyConfigurations.quantifyExpressions");
    return this.createFrom(quantifyPolicy);
  }

  @Override
  @Transactional
  public QuantifyPolicy createFrom(QuantifyPolicy quantifyPolicy) {
    //校验政策数据
    createValidation(quantifyPolicy);
    Date now = new Date();
    quantifyPolicy.setCreateTime(now);
    quantifyPolicy.setModifyTime(now);
    quantifyPolicy.setCreateAccount(loginUserService.getLoginAccountName());
    quantifyPolicy.setModifyAccount(loginUserService.getLoginAccountName());
    //保存包量政策
    this.quantifyPolicyRepository.save(quantifyPolicy);
    //保存包量配置行
    quantifyPolicy.getQuantifyConfigurations().forEach(quantifyConfiguration -> {
      quantifyConfiguration.setQuantifyPolicyId(quantifyPolicy.getId());
      this.quantifyConfigurationRepository.save(quantifyConfiguration);
      //保存配置行下关联的分利表达式
      quantifyConfiguration.getQuantifyExpressions().forEach(quantifyExpression -> quantifyExpression.setConfigurationId(quantifyConfiguration.getId()));
      this.quantifyExpressionRepository.saveBatch(quantifyConfiguration.getQuantifyExpressions());
      //当产品维度设置不为全部时保存配置行下关联的产品维度
      if (!MaterialDimensionConstant.DIMENSION_ALL.equals(quantifyConfiguration.getDimensionFlag())) {
        quantifyConfiguration.getQuantifyRanges().forEach(quantifyRange -> quantifyRange.setConfigurationId(quantifyConfiguration.getId()));
        this.quantifyRangeRepository.saveBatch(quantifyConfiguration.getQuantifyRanges());
      }
    });
    return quantifyPolicy;
  }

  @Override
  public QuantifyPolicy findDetailByTemplateCode(String templateCode) {
    if (StringUtils.isBlank(templateCode)) {
      return null;
    }
    return this.quantifyPolicyRepository.findDetailByTemplateCode(templateCode);
  }

  @Override
  public QuantifyPolicy findById(String id) {
    return quantifyPolicyRepository.findDetailById(id);
  }

  @Override
  public QuantifyPolicy findByTemplateCode(String templateCode) {
    return quantifyPolicyRepository.findDetailByTemplateCode(templateCode);
  }

  /**
   * 创建包量政策参数校验
   *
   * @param entity 包量政策创建实体
   */
  private void createValidation(QuantifyPolicy entity) {
    Validate.notNull(entity, "进行当前操作时，信息对象必须传入!!");
    Validate.isTrue(StringUtils.isBlank(entity.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    entity.setId(null);
    if (StringUtils.isBlank(entity.getTenantCode())) {
      entity.setTenantCode(TenantUtils.getTenantCode());
    }
    Validate.isTrue(!CollectionUtils.isEmpty(entity.getQuantifyConfigurations()), "政策配置行不能为空！");
    //校验政策行对应的产品边界和分利表达式
    entity.getQuantifyConfigurations().forEach(quantifyConfiguration -> {
      Validate.notBlank(quantifyConfiguration.getDimensionFlag(), "产品维度标记不能为空");
      Validate.notBlank(quantifyConfiguration.getDimensionName(), "产品维度名称不能为空");
      if (!MaterialDimensionConstant.DIMENSION_ALL.equals(quantifyConfiguration.getDimensionFlag())) {
        //如果配置的产品维度不是所有产品，则需要校验产品维度数据不能为空
        Validate.isTrue(!CollectionUtils.isEmpty(quantifyConfiguration.getQuantifyRanges()), "政策关联产品范围不能为空！");
      }
      Validate.isTrue(!CollectionUtils.isEmpty(quantifyConfiguration.getQuantifyExpressions()), "政策关联分利表达式不能为空！");
      quantifyConfiguration.getQuantifyRanges().forEach(this::quantifyRangeValidation);
      quantifyConfiguration.getQuantifyExpressions().forEach(this::quantifyExpressionValidation);
    });
  }

  /**
   * 包量分利表达式校验
   *
   * @param entity 包量分利表达式校验
   */
  private void quantifyExpressionValidation(QuantifyExpression entity) {
    Validate.notNull(entity, "进行当前操作时，信息对象必须传入!!");
    Validate.isTrue(StringUtils.isBlank(entity.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    entity.setId(null);
    Validate.notBlank(entity.getRewardMethodFlag(), "添加信息时，奖励方式标志不能为空！");
    Validate.notBlank(entity.getRewardMethodName(), "添加信息时，奖励方式名称不能为空！");
    Validate.notNull(entity.getReachRate(), "添加信息时，阶梯达成比例不能为空！");
    Validate.isTrue(entity.getReachRate().compareTo(BigDecimal.ZERO) > 0, "阶梯达成比例必须大于0！");
    Validate.notNull(entity.getRewardData(), "添加信息时，奖励金额不能为空！");
    Validate.isTrue(entity.getRewardData().compareTo(BigDecimal.ZERO) > 0, "奖励金额必须大于0！");
  }

  /**
   * 包量产品范围校验
   *
   * @param entity 包量产品范围实体
   */
  private void quantifyRangeValidation(QuantifyRange entity) {
    Validate.notNull(entity, "进行当前操作时，信息对象必须传入!!");
    Validate.isTrue(StringUtils.isBlank(entity.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    entity.setId(null);
    Validate.notBlank(entity.getSpecialCode(), "添加信息时，产品维度编码不能为空！");
    Validate.notBlank(entity.getSpecialName(), "添加信息时，产品维度名称不能为空！");
  }

  /**
   * 包量政策范围重复校验
   * step1:获取当前有效期时间内，同组织的包含包量任务的模板信息
   * step2:获取创建政策的物料编码集合
   * step3:查询交集模板对应的政策列表
   * step4:循环政策列表，获取物料列表并与创建政策的物料编码取交集，若交集存在则抛出异常
   *
   * @param agreementTemplateVo 包量产品范围实体
   */
  @Override
  public void validatePolicyScope(AgreementTemplateVo agreementTemplateVo, AgreementPolicyDto agreementPolicyDto) {
    ProfitAgreementTemplatePaginationDto templatePaginationDto = new ProfitAgreementTemplatePaginationDto();
    if (!CollectionUtils.isEmpty(agreementTemplateVo.getTemplateOrgRelationships())) {
      //如果协议模板设定了组织范围，则取组织范围放入查询参数
      List<String> orgCodes = agreementTemplateVo.getTemplateOrgRelationships().stream().map(TemplateOrgRelationshipVo::getOrgCode).collect(Collectors.toList());
      templatePaginationDto.setOrgCodes(orgCodes);
    }
    templatePaginationDto.setPolicyCode(this.quantifyPolicyMountRegister.getKey());
    templatePaginationDto.setEffectiveScopeStartTime(agreementTemplateVo.getEffectiveStartTime());
    templatePaginationDto.setEffectiveScopeEndTime(agreementTemplateVo.getEffectiveEndTime());
    templatePaginationDto.setStatus(EnableStatusEnum.ENABLE.getCode());
    //根据组织列表，生效时间查询已存在包量政策的协议模板
    List<AgreementTemplateVo> templateVos = this.agreementTemplateVoService.findByConditions(templatePaginationDto);
    if (CollectionUtils.isEmpty(templateVos)) {
      //如果没有找到交集模板，则校验通过
      return;
    }
    QuantifyPolicy quantifyPolicy = nebulaToolkitService.copyObjectByWhiteList(agreementPolicyDto, QuantifyPolicy.class, HashSet.class, ArrayList.class, "quantifyConfigurations", "quantifyConfigurations.quantifyRanges", "quantifyConfigurations.quantifyExpressions");
    //通过交集模板列表查询对应的包量政策
    List<String> templateCodes = templateVos.stream().map(AgreementTemplateVo::getTemplateCode).collect(Collectors.toList());
    QuantifyPolicyDto quantifyPolicyDto = new QuantifyPolicyDto();
    quantifyPolicyDto.setTemplateCodes(templateCodes);
    quantifyPolicyDto.setCalculateCycle(quantifyPolicy.getCalculateCycle());
    List<QuantifyPolicy> quantifyPolicies = this.quantifyPolicyRepository.findByConditions(quantifyPolicyDto);
    if (CollectionUtils.isEmpty(quantifyPolicies)) {
      //如果没有找到包量政策，则校验通过
      return;
    }
    //获取本次创建政策包含物料编码
    List<String> createPolicyMaterialCodes = this.policyRangeValidation(quantifyPolicy);
    //如果本次创建政策包含物料编码为空，直接返回
    if (CollectionUtils.isEmpty(createPolicyMaterialCodes)) {
      return;
    }
    //循环交集模板列表，获取对应包含的物料编码列表，并求交集
    quantifyPolicies.forEach(policy -> {
      List<String> materialCodes = this.policyRangeValidation(policy);
      if (!CollectionUtils.isEmpty(materialCodes)) {
        //当政策的物料编码不为空，取政策物料编码和创建政策的物料编码的合集
        materialCodes.retainAll(createPolicyMaterialCodes);
        Validate.isTrue(CollectionUtils.isEmpty(materialCodes), "本次创建包量产品物料已经创建过本时段包量政策！");
      }
    });
  }

  /**
   * 获取政策配置的物料编码列表
   * 此方法包量政策验重使用，前提为本次创建模板已经在数据库中查询到组织-时间有交集的模板信息
   * 故此方法内部会校验产品维度不能为全部，当产品维度为全部时抛出异常
   * 返回政策下所有配置的物料编码集合
   *
   * @param quantifyPolicy 包量政策
   */
  private List<String> policyRangeValidation(QuantifyPolicy quantifyPolicy) {
    List<String> materialCodes = Lists.newArrayList();
    Validate.isTrue(!CollectionUtils.isEmpty(quantifyPolicy.getQuantifyConfigurations()), "包量政策必须包含配置！");
    quantifyPolicy.getQuantifyConfigurations().forEach(configuration -> {
      Validate.isTrue(!MaterialDimensionConstant.DIMENSION_ALL.equals(configuration.getDimensionFlag()), "同时间段内有相同结算周期模板配置为全部商品！");
      Validate.isTrue(!CollectionUtils.isEmpty(configuration.getQuantifyRanges()), "包量政策配置必须配置产品维度范围！");
      List<String> materialCodeList = this.materialVoService.findMaterialCodeByDimensionCodesAndDimensionType(configuration.getQuantifyRanges().stream().map(QuantifyRange::getSpecialCode).collect(Collectors.toList()), configuration.getDimensionFlag());
      if (!CollectionUtils.isEmpty(materialCodeList)) {
        materialCodes.addAll(materialCodeList);
      }
    });
    return materialCodes;
  }

  @Override
  public List<QuantifyPolicy> findByConditions(QuantifyPolicyDto quantifyPolicyDto) {
    if (Objects.isNull(quantifyPolicyDto)) {
      return null;
    }
    return this.quantifyPolicyRepository.findByConditions(quantifyPolicyDto);
  }
}
