package com.biz.crm.cps.business.policy.quantify.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.cps.bisiness.policy.quantify.sdk.constant.QuantifyConstant;
import com.biz.crm.cps.bisiness.policy.quantify.sdk.constant.QuantifyTaskStatusEnum;
import com.biz.crm.cps.bisiness.policy.quantify.sdk.dto.QuantifyTaskDto;
import com.biz.crm.cps.business.agreement.sdk.common.enums.AgreementStatusEnum;
import com.biz.crm.cps.business.agreement.sdk.common.enums.AutoSignEnum;
import com.biz.crm.cps.business.agreement.sdk.common.enums.SignStatusEnum;
import com.biz.crm.cps.business.agreement.sdk.dto.AgreementDto;
import com.biz.crm.cps.business.agreement.sdk.dto.LoginUserAgreementDto;
import com.biz.crm.cps.business.agreement.sdk.service.AgreementVoService;
import com.biz.crm.cps.business.agreement.sdk.vo.AgreementVo;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.cps.business.participator.sdk.common.enums.ParticipatorTypeEnum;
import com.biz.crm.cps.business.policy.quantify.local.entity.QuantifyExpression;
import com.biz.crm.cps.business.policy.quantify.local.entity.QuantifyPolicy;
import com.biz.crm.cps.business.policy.quantify.local.entity.QuantifyRange;
import com.biz.crm.cps.business.policy.quantify.local.entity.QuantifyTask;
import com.biz.crm.cps.business.policy.quantify.local.entity.QuantifyTaskDetail;
import com.biz.crm.cps.business.policy.quantify.local.repository.QuantifyTaskDetailRepository;
import com.biz.crm.cps.business.policy.quantify.local.repository.QuantifyTaskRepository;
import com.biz.crm.cps.business.policy.quantify.local.service.QuantifyPolicyService;
import com.biz.crm.cps.business.policy.quantify.local.service.QuantifyTaskService;
import com.biz.crm.cps.business.policy.quantify.local.service.builder.QuantifyTaskBuilder;
import com.biz.crm.cps.business.policy.quantify.local.service.builder.QuantifyTaskHalfYearBuilder;
import com.biz.crm.cps.business.policy.quantify.local.service.builder.QuantifyTaskMonthBuilder;
import com.biz.crm.cps.business.policy.quantify.local.service.builder.QuantifyTaskSeasonBuilder;
import com.biz.crm.cps.business.policy.quantify.local.service.builder.QuantifyTaskYearBuilder;
import com.biz.crm.cps.business.policy.quantify.local.service.observer.QuantifyPolicyMountRegisterImpl;
import com.biz.crm.cps.business.policy.quantify.local.utils.SchedulerUtils;
import com.biz.crm.cps.business.policy.sdk.dto.PolicyRewardConditionDto;
import com.biz.crm.cps.business.policy.sdk.service.observer.PolicyRewardServiceObserver;
import com.biz.crm.cps.business.product.sdk.common.constant.MaterialDimensionConstant;
import com.biz.crm.cps.business.product.sdk.service.MaterialVoService;
import com.biz.crm.cps.external.barcode.sdk.common.enums.ScanCodeExceptionEnum;
import com.biz.crm.cps.external.barcode.sdk.dto.ScanCodeRecordPageDto;
import com.biz.crm.cps.external.barcode.sdk.service.ScanCodeRecordVoService;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.task.annotations.DynamicTaskService;
import com.bizunited.nebula.task.service.DynamicTaskSchedulerVoService;
import com.bizunited.nebula.task.vo.DynamicTaskSchedulerVo;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * @author jerry7
 * @date 2021/8/12 15:34
 * 包量任务接口实现
 */
@Service("quantifyTaskServiceImpl")
public class QuantifyTaskServiceImpl implements QuantifyTaskService {

  @Autowired
  private AgreementVoService agreementVoService;

  @Autowired
  private QuantifyPolicyService quantifyPolicyService;

  @Autowired
  private QuantifyTaskRepository quantifyTaskRepository;

  @Autowired
  private GenerateCodeService generateCodeService;

  @Autowired
  private MaterialVoService materialVoService;

  @Autowired
  private ScanCodeRecordVoService scanCodeRecordVoService;

  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Autowired
  private List<PolicyRewardServiceObserver> policyRewardServiceObservers;

  @Autowired
  private DynamicTaskSchedulerVoService dynamicTaskSchedulerVoService;

  @Autowired
  private QuantifyTaskDetailRepository quantifyTaskDetailRepository;

  @Autowired
  private QuantifyPolicyMountRegisterImpl quantifyPolicyMountRegister;

  /**
   * step1:根据协议id查询协议信息(包含模板信息)
   * step2:根据协议模板查询对应的包量政策
   * step3:根据时间段和包量结算周期创建对应的包量任务
   * step4:保存包量任务实体
   *
   * @param agreementId 协议主键id
   * @return 创建结果
   */
  @Override
  @Transactional
  public List<QuantifyTask> create(String agreementId) {
    LoginUserAgreementDto loginUserAgreementDto = new LoginUserAgreementDto();
    loginUserAgreementDto.setAgreementCode(agreementId);
    AgreementDto agreementDto = new AgreementDto();
    agreementDto.setAgreementCode(agreementId);
    Page<AgreementVo> agreementVo = agreementVoService.findByConditions(PageRequest.of(0, 50), agreementDto);
    if (agreementVo != null && !CollectionUtils.isEmpty(agreementVo.getRecords())) {
      return this.create(agreementVo.getRecords().get(0));
    } else {
      return Lists.newArrayList();
    }
  }

  @Override
  @Transactional
  public List<QuantifyTask> create(AgreementVo agreementVo) {
    //根据协议模板查询对应的包量政策
    QuantifyPolicy quantifyPolicy = quantifyPolicyService.findByTemplateCode(agreementVo.getTemplateCode());
    Validate.notNull(quantifyPolicy, "协议模板编码" + agreementVo.getAgreementCode() + "未找到对应的包量政策！");
    QuantifyTaskBuilder quantifyTaskBuilder = this.findBuilderByCalculateCycle(quantifyPolicy.getCalculateCycle());
    //创建包量任务列表
    List<QuantifyTask> taskList = quantifyTaskBuilder.assemblyQuantifyTask(agreementVo, quantifyPolicy);
    if (!CollectionUtils.isEmpty(taskList)) {
      List<String> businessCodes = generateCodeService.generateCode(QuantifyConstant.QUANTIFY_TASK_CODE, taskList.size());
      Validate.isTrue(!CollectionUtils.isEmpty(businessCodes) && businessCodes.size() == taskList.size(), "生成业务单号异常");
      for (int i = 0; i < taskList.size(); i++) {
        //补充业务单号以及其他信息
        QuantifyTask task = taskList.get(i);
        task.setBusinessCode(businessCodes.get(i));
        task.setQuantifyPolicyId(quantifyPolicy.getId());
        task.setTaskType(quantifyPolicy.getCalculateCycle());
        task.setAgreementId(agreementVo.getAgreementCode());
        task.setAgreementName(agreementVo.getAgreementName());
        task.setTemplateId(agreementVo.getTemplateCode());
        task.setTemplateName(agreementVo.getBelongTemplate());
        task.setTerminalCode(agreementVo.getTerminalCode());
        task.setTerminalName(agreementVo.getRelationTerminal());
        task.setTenantCode(TenantUtils.getTenantCode());
      }
    }
    this.quantifyTaskRepository.saveBatch(taskList);
    //生成定时任务
    taskList.forEach(task -> {
              DynamicTaskSchedulerVo taskSchedulerVo = dynamicTaskSchedulerVoService.createIgnorePrefix(task.getBusinessCode(), "quantifyTaskServiceImpl", "triggerQuantityTask"
                      , SchedulerUtils.getSchedulerCron(task.getTaskType()), agreementVo.getEffectiveEndTime(), task.getTaskName(), task.getBusinessCode());
              dynamicTaskSchedulerVoService.start(new String[]{taskSchedulerVo.getTaskCode()});
            }
    );
    return taskList;
  }

  /**
   * 根据包量政策的结算类别寻找对应的组装实现类
   *
   * @param calculateCycle 结算类别(月/季/半年/年)
   * @return 包量任务组装类
   */
  private QuantifyTaskBuilder findBuilderByCalculateCycle(String calculateCycle) {
    switch (calculateCycle) {
      case QuantifyConstant.MONTH:
        return new QuantifyTaskMonthBuilder();
      case QuantifyConstant.SEASON:
        return new QuantifyTaskSeasonBuilder();
      case QuantifyConstant.HALF_YEAR:
        return new QuantifyTaskHalfYearBuilder();
      case QuantifyConstant.YEAR:
      default:
        return new QuantifyTaskYearBuilder();
    }
  }

  @Override
  public QuantifyTask findById(String id) {
    return quantifyTaskRepository.getById(id);
  }

  @Override
  public List<QuantifyTask> findByConditions(Pageable pageable, QuantifyTaskDto quantifyTaskDto) {
    return quantifyTaskRepository.findByConditions(pageable, quantifyTaskDto);
  }

  /**
   * 包量任务结算
   *
   * @param businessCode 包量任务业务编码
   */
  @Override
  @Transactional
  public void triggerQuantityTask(String businessCode) {
    /**
     * step1：获取包量任务实例
     * step2：根据协议的模板编码获取对应包量政策
     * step3：包量结算，构造包量结算列表
     * step4：通知费用层进行数据结算
     * step5：根据结算信息更新包量任务状态和达成率
     */
    QuantifyTask quantifyTask = quantifyTaskRepository.findByBusinessCode(businessCode);
    Validate.notNull(quantifyTask, "业务单号：" + businessCode + "未获取到对应包量任务！");
    Validate.isTrue(quantifyTask.getTaskStatus().equals(QuantifyTaskStatusEnum.STATUS_DOING.getCode()), "只有处于执行中任务可以进行结算");
    QuantifyPolicy policy = quantifyPolicyService.findByTemplateCode(quantifyTask.getTemplateId());
    Validate.notNull(policy, "业务单号：" + businessCode + "未获取到对应包量政策！");
    //验证协议是否在进行中
    LoginUserAgreementDto loginUserAgreementDto = new LoginUserAgreementDto();
    loginUserAgreementDto.setSignStatus(SignStatusEnum.SIGNED.getCode());
    loginUserAgreementDto.setAgreementStatus(AgreementStatusEnum.EXECUTING.getCode());
    loginUserAgreementDto.setAgreementCode(quantifyTask.getAgreementId());
    List<AgreementVo> agreementVos = this.agreementVoService.findByConditions(loginUserAgreementDto);
    Validate.isTrue(!CollectionUtils.isEmpty(agreementVos), "业务单号：" + businessCode + "未获取到对应的已生效协议！");
    //包量任务结算
    List<QuantifyTaskDetail> quantifyTaskDetails = buildTaskDetail(policy, quantifyTask);
    //通知费用层
    this.sendToCost(quantifyTask, quantifyTaskDetails);
    //更新包量任务信息
    this.updateQuantifyTask(quantifyTask, quantifyTaskDetails);
    //保存包量结算详情
    this.quantifyTaskDetailRepository.saveBatch(quantifyTaskDetails);
  }


  /**
   * 根据包量政策和任务进行结算，构造任务结算明细列表
   *
   * @param policy       包量政策
   * @param quantifyTask 包量任务
   * @return 任务结算明细列表
   */
  private List<QuantifyTaskDetail> buildTaskDetail(QuantifyPolicy policy, QuantifyTask quantifyTask) {
    List<QuantifyTaskDetail> quantifyTaskDetails = Lists.newArrayList();
    policy.getQuantifyConfigurations().forEach(configuration -> {
      BigDecimal scanCount = BigDecimal.ZERO;
      ScanCodeRecordPageDto scanCodeRecordPageDto = new ScanCodeRecordPageDto();
      scanCodeRecordPageDto.setCreateTimeStart(quantifyTask.getTaskStartTime());
      scanCodeRecordPageDto.setCreateTimeEnd(quantifyTask.getTaskEndTime());
      scanCodeRecordPageDto.setScanParticipatorCode(quantifyTask.getTerminalCode());
      scanCodeRecordPageDto.setScanCodeException(ScanCodeExceptionEnum.NO.getCode());
      if (MaterialDimensionConstant.DIMENSION_ALL.equals(configuration.getDimensionFlag())) {
        //如果商品标签是全部，则直接调用扫码模块获取该时间段内扫码数量
        scanCodeRecordPageDto.setProductCodes(Lists.newArrayList());
        scanCount = scanCodeRecordVoService.countSalesByScanCodeRecordPageDto(scanCodeRecordPageDto);
      } else {
        //非全部商品，则通过产品维度获取物料编码集合，然后调用扫码模块获取扫码数量
        List<String> dimensionCodes = configuration.getQuantifyRanges()
                .stream().map(QuantifyRange::getSpecialCode).collect(Collectors.toList());
        List<String> materialCodes = materialVoService.findMaterialCodeByDimensionCodesAndDimensionType(dimensionCodes, configuration.getDimensionFlag());
        if (!CollectionUtils.isEmpty(materialCodes)) {
          scanCodeRecordPageDto.setProductCodes(materialCodes);
          scanCount = scanCodeRecordVoService.countSalesByScanCodeRecordPageDto(scanCodeRecordPageDto);
        }
      }
      //用扫码次数/销量目标计算达标比例,四舍五入并保留两位小数
      BigDecimal reachRate = scanCount.divide(quantifyTask.getSaleTarget(), 4, RoundingMode.HALF_UP).multiply(new BigDecimal(100));
      for (QuantifyExpression expression : configuration.getQuantifyExpressions()) {
        QuantifyTaskDetail quantifyTaskDetail = nebulaToolkitService.copyObjectByWhiteList(expression, QuantifyTaskDetail.class, Set.class, ArrayList.class);
        quantifyTaskDetail.setId(null);
        quantifyTaskDetail.setBusinessCode(quantifyTask.getBusinessCode());
        quantifyTaskDetail.setSaleTarget(quantifyTask.getSaleTarget());
        quantifyTaskDetail.setActualSales(scanCount);
        quantifyTaskDetail.setReachRate(reachRate);
        //根据达标比例设定结算明细的分利状态
        quantifyTaskDetail.setProfitStatus(reachRate.compareTo(expression.getReachRate()) > -1 ? AutoSignEnum.YES.getCode() : AutoSignEnum.NO.getCode());
        quantifyTaskDetails.add(quantifyTaskDetail);
      }
    });
    return quantifyTaskDetails;
  }

  /**
   * 向消费层发布结算消息
   *
   * @param quantifyTask        包量任务
   * @param quantifyTaskDetails 包量结算信息列表
   */
  private void sendToCost(QuantifyTask quantifyTask, List<QuantifyTaskDetail> quantifyTaskDetails) {
    if (CollectionUtils.isEmpty(quantifyTaskDetails)) {
      return;
    }
    List<PolicyRewardConditionDto> policyRewardConditionDtos = Lists.newArrayList();
    quantifyTaskDetails.forEach(detail -> {
      if (AutoSignEnum.YES.getCode().equals(detail.getProfitStatus())) {
        //针对需要传递至费用层构造DTO
        PolicyRewardConditionDto policyRewardConditionDto = new PolicyRewardConditionDto();
        policyRewardConditionDto.setPolicyName(this.quantifyPolicyMountRegister.getName());
        //参与者信息
        policyRewardConditionDto.setActualParticipatorFlag(ParticipatorTypeEnum.TERMINAL.getDictCode());
        policyRewardConditionDto.setActualParticipatorCode(quantifyTask.getTerminalCode());
        policyRewardConditionDto.setParticipatorFlag(ParticipatorTypeEnum.TERMINAL.getDictCode());
        policyRewardConditionDto.setParticipatorName(ParticipatorTypeEnum.TERMINAL.getValue());
        policyRewardConditionDto.setTemplateCode(quantifyTask.getTemplateId());
        //奖励方式
        policyRewardConditionDto.setRewardMethodFlag(detail.getRewardMethodFlag());
        policyRewardConditionDto.setRewardMethodName(detail.getRewardMethodName());
        //奖励条件
        policyRewardConditionDto.setAwardConditionFlag(detail.getAwardConditionFlag());
        policyRewardConditionDto.setAwardConditionName(detail.getAwardConditionName());
        //奖励金额
        policyRewardConditionDto.setRewardData(detail.getRewardData().toString());
        //触发动作
        policyRewardConditionDto.setTriggerAction(QuantifyConstant.QUANTIFY_ACTION);
        policyRewardConditionDto.setTriggerObject(quantifyTask.getBusinessCode());
        //协议编码
        policyRewardConditionDto.setAgreementCode(quantifyTask.getAgreementId());
        policyRewardConditionDtos.add(policyRewardConditionDto);
      }
    });
    if (!CollectionUtils.isEmpty(policyRewardConditionDtos) && !CollectionUtils.isEmpty(policyRewardServiceObservers)) {
      //观察者和需要传递至费用层dto不为空时，进行结算传递
      policyRewardServiceObservers.forEach(policyRewardServiceObserver -> policyRewardServiceObserver.createRewardRecord(policyRewardConditionDtos));
    }
  }

  /**
   * 更新包量任务状态和达成率
   *
   * @param quantifyTask        包量任务
   * @param quantifyTaskDetails 包量任务结算列表
   */
  private void updateQuantifyTask(QuantifyTask quantifyTask, List<QuantifyTaskDetail> quantifyTaskDetails) {
    if (CollectionUtils.isEmpty(quantifyTaskDetails)) {
      //任务结算明细为空，直接设置任务状态为过期
      quantifyTask.setReachRate(BigDecimal.ZERO);
      quantifyTask.setTaskStatus(QuantifyTaskStatusEnum.STATUS_EXPIRED.getCode());
    }
    BigDecimal reachRate = BigDecimal.ZERO;
    String taskStatus = QuantifyTaskStatusEnum.STATUS_EXPIRED.getCode();
    for (QuantifyTaskDetail detail : quantifyTaskDetails) {
      reachRate = reachRate.add(detail.getReachRate());
      if (detail.getProfitStatus().equals(AutoSignEnum.YES.getCode())) {
        taskStatus = QuantifyTaskStatusEnum.STATUS_COMPLETE.getCode();
      }
    }
    quantifyTask.setReachRate(reachRate.divide(new BigDecimal(quantifyTaskDetails.size()), 4, RoundingMode.HALF_UP));
    quantifyTask.setTaskStatus(taskStatus);
    this.quantifyTaskRepository.updateById(quantifyTask);
  }

  /**
   * 定时更新任务时间已经开始的待执行任务状态
   */
  @Override
  @DynamicTaskService(cornExpression = "1 0 0 * * ?", taskDesc = "每日更新待执行任务状态")
  public void updateQuantifyTaskStatus() {
    List<QuantifyTask> quantifyTasks = this.quantifyTaskRepository.findByAgreementIdAndStatus(null, QuantifyTaskStatusEnum.STATUS_WAIT.getCode(), new Date());
    if (!CollectionUtils.isEmpty(quantifyTasks)) {
      quantifyTasks.forEach(quantifyTask -> quantifyTask.setTaskStatus(QuantifyTaskStatusEnum.STATUS_DOING.getCode()));
      this.quantifyTaskRepository.saveOrUpdateBatch(quantifyTasks);
    }
  }
}
