package com.biz.crm.cps.business.policy.quantify.local.utils;

import com.biz.crm.cps.business.policy.quantify.local.entity.QuantifyPolicy;

import java.math.BigDecimal;

/**
 * @author jerry7
 * @date 2021/8/20 10:01
 * 包量任务工具类
 */
public class QuantifyUtils {

  /**
   * 根据当前月份返回设定的对应月份设定的销量目标
   *
   * @param month          当前月份
   * @param quantifyPolicy 包量政策
   * @return 销量目标
   */
  public static BigDecimal getScaleByMonth(int month, QuantifyPolicy quantifyPolicy) {
    switch (month) {
      case 1:
        return quantifyPolicy.getJanuarySaleNum();
      case 2:
        return quantifyPolicy.getFebruarySaleNum();
      case 3:
        return quantifyPolicy.getMarchSaleNum();
      case 4:
        return quantifyPolicy.getAprilSaleNum();
      case 5:
        return quantifyPolicy.getMaySaleNum();
      case 6:
        return quantifyPolicy.getJuneSaleNum();
      case 7:
        return quantifyPolicy.getJulySaleNum();
      case 8:
        return quantifyPolicy.getAugustSaleNum();
      case 9:
        return quantifyPolicy.getSeptemberSaleNum();
      case 10:
        return quantifyPolicy.getOctoberSaleNum();
      case 11:
        return quantifyPolicy.getNovemberSaleNum();
      case 12:
      default:
        //默认返回12月份销量
        return quantifyPolicy.getDecemberSaleNum();
    }
  }

  /**
   * 根据当前月份，获取当前季度设定的包量销量目标，先获取当前季度剩余月份数量，然后根据月份获取销量目标相加
   *
   * @param month          当前月份
   * @param quantifyPolicy 包量政策
   * @return 季度销量目标
   */
  public static BigDecimal getScaleBySeason(int month, QuantifyPolicy quantifyPolicy) {
    //获取当前季度剩余月份数量
    int seasonMonths = 3 % month;
    switch (seasonMonths) {
      case 0:
        return getScaleByMonth(month, quantifyPolicy);
      case 1:
        return getScaleByMonth(month, quantifyPolicy).add(getScaleByMonth(month + 1, quantifyPolicy));
      case 2:
      default:
        //默认当前月份为当前季度第一个月
        return getScaleByMonth(month, quantifyPolicy).add(getScaleByMonth(month + 1, quantifyPolicy)
                .add(getScaleByMonth(month + 2, quantifyPolicy)));
    }
  }

  /**
   * 根据当前月份，获取当前半年度度设定的包量销量目标，先获取当前月份所属季度，然后根据季度来计算半年度销量目标
   *
   * @param month          当前月份
   * @param quantifyPolicy 包量政策
   * @return 半年度销量目标
   */
  public static BigDecimal getScaleByHalfYear(int month, QuantifyPolicy quantifyPolicy) {
    //获取当前季度数量
    int season = month / 3 + 1;
    switch (season) {
      case 1:
        return getScaleBySeason(month, quantifyPolicy).add(getScaleBySeason(4, quantifyPolicy));
      case 3:
        return getScaleBySeason(month, quantifyPolicy).add(getScaleBySeason(10, quantifyPolicy));
      case 2:
      case 4:
      default:
        //当为第二季度或者第四季度，则当前半年销量直接取当前季度销量
        return getScaleBySeason(month, quantifyPolicy);
    }
  }

  /**
   * 根据当前月份，获取当前年度度设定的包量销量目标，先获取当前月份所属半年度，然后根据半年度来计算年度销量目标
   *
   * @param month          当前月份
   * @param quantifyPolicy 包量政策
   * @return 年度销量目标
   */
  public static BigDecimal getScaleByYear(int month, QuantifyPolicy quantifyPolicy) {
    //获取当前季度数量
    int halfYear = month / 6;
    switch (halfYear) {
      case 0:
        //当前月份在上半年时，取当前半年度销量，加上下半年销量
        return getScaleByHalfYear(month, quantifyPolicy).add(getScaleBySeason(7, quantifyPolicy));
      case 1:
      default:
        //当为下半年时，则当前年销量直接取当前半年销量
        return getScaleBySeason(month, quantifyPolicy);
    }
  }
}
