package com.biz.crm.cps.business.reward.cost.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.cps.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.cps.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.cps.business.reward.cost.local.entity.CostDealerDetailEntity;
import com.biz.crm.cps.business.reward.cost.local.entity.CostDealerEntity;
import com.biz.crm.cps.business.reward.cost.local.repository.CostDealerDetailRepository;
import com.biz.crm.cps.business.reward.cost.local.service.CostDealerDetailService;
import com.biz.crm.cps.business.reward.cost.local.service.CostDealerService;
import com.biz.crm.cps.business.reward.cost.sdk.common.CostCodeConstant;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Date;
import java.util.List;
import java.util.Objects;

/**
 * @Author: zengxingwang
 * @Date: 2021/8/21 10:33
 */
@Slf4j
@Service
public class CostDealerDetailServiceImpl implements CostDealerDetailService {

  @Autowired
  private CostDealerDetailRepository costDealerDetailRepository;
  @Autowired(required = false)
  private LoginUserService loginUserService;
  @Autowired
  private CostDealerService costDealerService;
  @Autowired
  private GenerateCodeService generateCodeService;

  /**
   * 多条件分页查询
   * @param costDealerId
   * @param pageable
   * @return
   */
  @Override
  public Page<CostDealerDetailEntity> findByCostDealerId(String costDealerId, Pageable pageable) {
    if(StringUtils.isBlank(costDealerId)){
      return new Page<>();
    }
    ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    return costDealerDetailRepository.findByCostDealerId(costDealerId, pageable);
  }

  /**
   * 创建
   * @param dealerDetailEntity
   * @return
   */
  @Transactional
  @Override
  public CostDealerDetailEntity create(CostDealerDetailEntity dealerDetailEntity) {
    log.debug("新增费用流水传入参数={}", dealerDetailEntity);
    this.createForm(dealerDetailEntity);
    CostDealerEntity dealer = dealerDetailEntity.getDealer();
    /**
     * 主表里面是否存在此参与者信息,如果没有需要插入一条数据到主表里面,如果有 获取主表的id
     */
    CostDealerEntity old = costDealerService.findByDealerCode(dealer.getDealerCode());
    if (old != null) {
      // 如果不为空 更新红包表累计分利金额信息
      BigDecimal totalAmount = old.getTotalAmount() == null ? BigDecimal.ZERO : old.getTotalAmount();
      BigDecimal unBilledFee = old.getUnBilledFee() == null ? BigDecimal.ZERO : old.getUnBilledFee();
      BigDecimal terminalCirculationFee = old.getTerminalCirculationFee() == null ? BigDecimal.ZERO : old.getTerminalCirculationFee();
      if("终端核销流转".equals(dealerDetailEntity.getSourceType())){
        terminalCirculationFee = terminalCirculationFee.add(dealerDetailEntity.getAmount()).setScale(4, RoundingMode.HALF_UP);
        old.setTerminalCirculationFee(terminalCirculationFee);
      }else {
        totalAmount = totalAmount.add(dealerDetailEntity.getAmount()).setScale(4, RoundingMode.HALF_UP);
        old.setTotalAmount(totalAmount);
      }
      unBilledFee = unBilledFee.add(dealerDetailEntity.getAmount()).setScale(4, RoundingMode.HALF_UP);
      old.setUnBilledFee(unBilledFee);
      old.setModifyAccount(loginUserService.getLoginAccountName());
      old.setModifyTime(new Date());
      this.costDealerService.update(old);
      dealerDetailEntity.setCostDealerId(old.getId());
    } else {
      /**
       * 此客户没有红包信息 新增一条红包信息
       * redPacketDetailDto对象转RedPacketDto对象
       */
      dealer.setBilledFee(dealerDetailEntity.getBilledFee());
      dealer.setTerminalCirculationFee(BigDecimal.ZERO);
      dealer.setTotalAmount(dealerDetailEntity.getAmount());
      dealer.setUnBilledFee(dealerDetailEntity.getAmount());
      CostDealerEntity returnEntity = this.costDealerService.create(dealer);
      dealerDetailEntity.setCostDealerId(returnEntity.getId());
    }
    this.costDealerDetailRepository.saveOrUpdate(dealerDetailEntity);
    return dealerDetailEntity;
  }

  /**
   * 组装保存entity数据
   *
   * @param dealerDetailEntity
   */
  private CostDealerDetailEntity createForm(CostDealerDetailEntity dealerDetailEntity) {
    //校验
    this.createValidation(dealerDetailEntity);
    if (Objects.nonNull(this.generateCodeService)) {
      dealerDetailEntity.setCode(this.generateCodeService.generateCode(CostCodeConstant.COST_FLOW, 1).get(0));
    }
    CostDealerEntity dealer = dealerDetailEntity.getDealer();
    dealer.setTenantCode(TenantUtils.getTenantCode());
    Date date = new Date();
    String loginAccountName = loginUserService.getLoginAccountName();
    dealer.setCreateTime(date);
    dealer.setCreateAccount(loginAccountName);
    dealer.setModifyTime(date);
    dealer.setModifyAccount(loginAccountName);
    dealer.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    dealer.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    dealerDetailEntity.setCreateTime(date);
    dealerDetailEntity.setCreateAccount(loginAccountName);
    dealerDetailEntity.setModifyTime(date);
    dealerDetailEntity.setModifyAccount(loginAccountName);

    // 红包余额 查询此客户最新的流水 获取最新的红包余额以此为基准 再做本次流水的运算
    CostDealerEntity lastDealer = costDealerService.findTop1ByDealerCode(dealer.getDealerCode());
    if (lastDealer != null) {
      //已上账费用
      BigDecimal billedFee = lastDealer.getBilledFee() == null ? BigDecimal.ZERO : lastDealer.getBilledFee();
      dealerDetailEntity.setBilledFee(billedFee);
      //未上账费用
      BigDecimal unBilledFee = lastDealer.getUnBilledFee() == null ? BigDecimal.ZERO : lastDealer.getUnBilledFee();
      unBilledFee = unBilledFee.add(dealerDetailEntity.getAmount()).setScale(4, RoundingMode.HALF_UP);
      dealerDetailEntity.setUnBilledFee(unBilledFee);
    } else {
      dealerDetailEntity.setBilledFee(BigDecimal.ZERO);
      dealerDetailEntity.setUnBilledFee(dealerDetailEntity.getAmount());
    }
    return dealerDetailEntity;
  }

  /**
   * 校验费用流水信息
   *
   * @param dealerDetailEntity
   */
  private void createValidation(CostDealerDetailEntity dealerDetailEntity) {

    // 验证必填项，以及字段长度
    Validate.notNull(dealerDetailEntity, "新增操作时，新增对象不能为空！");
    Validate.notBlank(dealerDetailEntity.getSourceCode(), "业务来源编号不能为空;");
    Validate.notBlank(dealerDetailEntity.getSourceType(), "业务类型不能为空;");
    CostDealerEntity dealer = dealerDetailEntity.getDealer();
    Validate.notNull(dealer, "新增操作时，经销商费用信息不能为空！");
    Validate.notBlank(dealer.getDealerCode(), "经销商编号不能为空;");
    Validate.notBlank(dealer.getDealerName(), "经销商名称不能为空;");

    Validate.isTrue(dealerDetailEntity.getSourceCode().length() < 255, "业务来源编号，在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(dealerDetailEntity.getSourceType().length() < 255, "业务类型，在进行添加时填入值超过了限定长度(255)，请检查!");
  }

  /**
   * 按 参与者code 和扫码code集合汇总获得的分利
   * @param participatorCode
   * @param recordCodes
   * @return
   */
  @Override
  public BigDecimal sumByParticipatorCodeAndRecordCodes(String participatorCode, List<String> recordCodes) {
    if(StringUtils.isBlank(participatorCode) && CollectionUtils.isEmpty(recordCodes)){
      return null;
    }
    return this.costDealerDetailRepository.sumByParticipatorCodeAndRecordCodes(participatorCode,recordCodes);
  }

  /**
   * 按 租户和类型 汇总获得分利
   * @param tenantCode
   * @param triggerAction
   * @return
   */
  @Override
  public BigDecimal sumByTenantCodeAndTriggerAction(String tenantCode, String triggerAction) {
    if(StringUtils.isBlank(tenantCode) || StringUtils.isBlank(triggerAction)){
      return null;
    }
    return this.costDealerDetailRepository.sumByTenantCodeAndTriggerAction(tenantCode,triggerAction);
  }

  /**
   * 修改
   * @param detail
   */
  @Override
  @Transactional
  public void update(CostDealerDetailEntity detail) {
    this.updateValidation(detail);
    costDealerDetailRepository.updateById(detail);
  }

  /**
   * 校验费用流水信息
   *
   * @param dealerDetailEntity
   */
  private void updateValidation(CostDealerDetailEntity dealerDetailEntity) {

    // 验证必填项，以及字段长度
    Validate.notNull(dealerDetailEntity, "修改操作时，新增对象不能为空！");
    Validate.notBlank(dealerDetailEntity.getId(), "修改操作时，主键不能为空;");
    Validate.notBlank(dealerDetailEntity.getSourceCode(), "业务来源编号不能为空;");
    Validate.notBlank(dealerDetailEntity.getSourceType(), "业务类型不能为空;");
    Validate.isTrue(dealerDetailEntity.getSourceCode().length() < 255, "业务来源编号，在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(dealerDetailEntity.getSourceType().length() < 255, "业务类型，在进行添加时填入值超过了限定长度(255)，请检查!");
  }
}
