package com.biz.crm.cps.business.reward.cost.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.cps.business.reward.cost.local.entity.CostDealerEntity;
import com.biz.crm.cps.business.reward.cost.local.repository.CostDealerRepository;
import com.biz.crm.cps.business.reward.cost.local.service.CostDealerService;
import com.biz.crm.cps.business.reward.cost.sdk.dto.CostDealerDto;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * @Author: zengxingwang
 * @Date: 2021/8/21 10:33
 */
@Slf4j
@Service
public class CostDealerServiceImpl implements CostDealerService {

  @Autowired
  private CostDealerRepository costDealerRepository;
  /**
   * 多条件分页查询
   * @param costDealerDto
   * @param pageable
   * @return
   */
  @Override
  public Page<CostDealerEntity> findByConditions(CostDealerDto costDealerDto, Pageable pageable) {
    if(costDealerDto == null){
      return new Page<>();
    }
    ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    costDealerDto.setTenantCode(TenantUtils.getTenantCode());
    return costDealerRepository.findByConditions(costDealerDto, pageable);
  }

  /**
   * 根据经销商编码查询
   * @param dealerCode
   * @return
   */
  @Override
  public CostDealerEntity findByDealerCode(String dealerCode) {
    if(StringUtils.isBlank(dealerCode)){
      return null;
    }
    return costDealerRepository.findByDealerCode(dealerCode);
  }

  /**
   * 修改
   * @param dealer
   * @return
   */
  @Override
  @Transactional
  public CostDealerEntity update(CostDealerEntity dealer) {
    this.updateValidate(dealer);
    this.costDealerRepository.saveOrUpdate(dealer);
    return dealer;
  }

  /**
   * 修改验证
   * @param dealer
   */
  private void updateValidate(CostDealerEntity dealer) {
    Validate.notNull(dealer, "修改时，数据不能为空");
    Validate.notBlank(dealer.getId(), "修改时，主键不能为空");
    Validate.notBlank(dealer.getDealerCode(), "修改时，经销商编码不能为空");
    Validate.notBlank(dealer.getDealerName(), "修改时，经销商名称不能为空");
    Validate.notBlank(dealer.getChannel(), "修改时，渠道不能为空");
    Validate.notBlank(dealer.getOrgCode(), "修改时，所属组织编码不能为空");
    Validate.notBlank(dealer.getOrgName(), "修改时，所属组织名称不能为空");
    CostDealerEntity byDealerCode = this.findByDealerCode(dealer.getDealerCode());
    Validate.isTrue(byDealerCode != null, "修改时，未找到对应数据，请检查");

    Validate.isTrue(dealer.getDealerCode() == null || dealer.getDealerCode().length() < 128 , "经销商编码,在进行修改时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(dealer.getDealerName() == null || dealer.getDealerCode().length() < 255 , "经销商编码,在进行修改时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(dealer.getChannel() == null || dealer.getChannel().length() < 64 , "渠道编码,在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(dealer.getOrgCode() == null || dealer.getOrgCode().length() < 128 , "所属组织编码,在进行修改时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(dealer.getOrgName() == null || dealer.getOrgName().length() < 255 , "所属组织名称,在进行修改时填入值超过了限定长度(128)，请检查!");
  }


  /**
   * 创建
   * @param dealer
   * @return
   */
  @Override
  @Transactional
  public CostDealerEntity create(CostDealerEntity dealer) {
    this.createValidate(dealer);
    this.costDealerRepository.save(dealer);
    return dealer;
  }

  @Override
  public CostDealerEntity findTop1ByDealerCode(String dealerCode) {
    if(StringUtils.isBlank(dealerCode)){
      return null;
    }
    return costDealerRepository.findTop1ByDealerCode(dealerCode);
  }

  /**
   * 创建验证
   * @param dealer
   */
  private void createValidate(CostDealerEntity dealer) {
    Validate.notNull(dealer, "创建时，数据不能为空");
    Validate.notBlank(dealer.getDealerCode(), "创建时，经销商编码不能为空");
    Validate.notBlank(dealer.getDealerName(), "创建时，经销商名称不能为空");
    Validate.notBlank(dealer.getChannel(), "创建时，渠道不能为空");
    Validate.notBlank(dealer.getOrgCode(), "创建时，所属组织编码不能为空");
    Validate.notBlank(dealer.getOrgName(), "创建时，所属组织名称不能为空");
    CostDealerEntity byDealerCode = this.findByDealerCode(dealer.getDealerCode());
    Validate.isTrue(byDealerCode == null, "创建时，数据已存在，请检查");

    Validate.isTrue(dealer.getDealerCode() == null || dealer.getDealerCode().length() < 128 , "经销商编码,在进行创建时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(dealer.getDealerName() == null || dealer.getDealerCode().length() < 255 , "经销商编码,在进行创建时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(dealer.getChannel() == null || dealer.getChannel().length() < 64 , "渠道编码,在进行创建时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(dealer.getOrgCode() == null || dealer.getOrgCode().length() < 128 , "所属组织编码,在进行创建时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(dealer.getOrgName() == null || dealer.getOrgName().length() < 255 , "所属组织名称,在进行创建时填入值超过了限定长度(128)，请检查!");
  }
}
